/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGMGR_HH
#define TRIGMGR_HH
/**  @name Trigger Manager
  *  The trigger manager class will in general have exactly one instance: 
  *  the trigger manager process. The trigger manager accepts registration
  *  requests followed optionally by trigger reports from registered 
  *  monitor processes. The triggers are routed to the meta-Database,
  *  used to generate EPICS alarms or ignored, depending on the trigger 
  *  disposition field, annd on the Trigger manager configuration. The
  *  trigger manager also produces html status reports that summarize the
  *  state of the trigger manager and recent trigger activity.
  *
  *  The trigger monitor process is run with the following command line:
\begin{verbatim}
   TrigMgr [-ofile <trig-direc>] [-notify <e-mail>] [-ntrig <in-file>]
           [-debug <level>] [-noDB] [-noEpics] [-timeout <nsec>]
\end{verbatim}
  *  Where the meanings of the flags are summarized in the following table:
  *
  * \begin{tabular}{|l|l|}
  * #-access <a-file>#     & Specify an access (configuration) file \\
  * #-debug <level># & Specify a debug printout level. No debug messages are 
  * produced at the default level [0]. The message volume increases with the 
  * level. \\
  * #-idletime <t-idle>#   & Number of seconds to wait for next trigger. \\
  * #-maxtime <t-max>#     & Max seconds before forced file write. \\
  * #-noDB# & Disable database logging. Storage of xml trigger files 
  * will continue. \\
  * #-noEpics# & Disable Epics alarm generation. \\
  * #-notify <e-mail># & Specify the e-mail address to which messages will 
  * be sent by LDAS each time a trigger ingestion job finishes. \\
  * #-ntrig <in-file>#     & Specifies the number of triggers to write per 
  * XML file (barring timeout). \\
  * #-ofile <trig-direc>#  & Specify the directory to which the XML trigger 
  * tables are to be written. By default HOME/triggers is used. \\
  * #-timeout <nsec>#      & Watchdog timer - If no trigger arrive in the 
  * specified time, any saved triggers will be written out.
  * \end{tabular}
  *
  * {\Large TrigMgr Inputs:} \\
  *
  * {\bf Configuration File} \\
  * The TrigMgr configuration file specifies which triggers are accepted 
  * by the trigger manager and optionally places limits on the disposition 
  * of the trigger. The trigger configuration file contains one or more 
  * lines, each of which specifies the disposition of a group of triggers.
  * If a trigger matches more than one trigger group, the first group 
  * specified will take precedence.
  * Each line contains the following fields separated by spaces:
  *
  * \begin{tabular}{|l|l|}
  * Process Name & Name of the monitor generating the specified trigger \\
  * Process Version & Version number of the monitor. \\
  * Trigger ID & The trigger ID string \\
  * Trigger sub-ID & The trigger sub-ID string. \\
  * #-dmask <mask># & Mask of allowed dispositions for the trigger group. \\
  * #-maxprty <max># & Maximum allowed priority for the trigger group.
  * \end{tabular}
  * The first four fields must be specified in the order listed. An asterisk
  * (*) may be used to allow any value for the positional fields.
  *
  * {\bf Messages:} \\
  * The trigger manager accepts monitor messages as defined 
  * in TrigMsg.hh. The following message types are handled:
  *
  * \begin{tabular}{|l|l|}
  * #TMM_REGISTER# & The specified process is registered with the trigger 
  * manager. The message carries a TrigProc object identifying the 
  * registering process. If the registration succedes, the trigger manager  
  * replies with a TMM_ProcID message with a unique process ID determined 
  * by the trigger manager. If an identical process is already registered, 
  * the Trigger manager returns the existing process ID. Otherwise, if the 
  * ProcessID field of the message has a valid process ID, the process ID 
  * is used. In all other cases the trigger manager assigns the process a 
  * new ID. \\
  * #TMM_TRIGGER# & Defines a trigger to be processed. The trigger may be 
  * forwarded to either the EPICS alarm system or the meta-database or both 
  * depending on the trigger disposition field and the trigger control 
  * database.  The trigger manager will reply with either a #TMM_Ack# or 
  * #TMM_NAck# message depending on whether the trigger and triggering 
  * process were recognised. \\
  * #TMM_Close# & Causes the specified process to be unregistered. \\
  * #<other># & Other message types are ignored and no reply is returned.
  *\end{tabular}
  *
  * {\bf Signals:} \\
  *  Signals are used to terminate the trigger manager or to cause it
  *  to reread its configuration file.
  *
  * \begin{tabular}{|l|l|}
  * SIGHUP & Reread the configuration file. Counts from continuing 
  *              trigger classes are left intact \\
  * SIGINT  & write out pending triggers and terminate.\\
  * SIGTERM & write out pending triggers and terminate.
  * \end{tabular}
  *
  *  {\bf TrigMgr Outputs:}
  *
  *  {\bf Trigger meta-Data Tables:} TrigMgr writes the summarized trigger 
  *  data to XML table files. The table files are given a unique name of 
  *  the form #TrigMgr<unique-id>.xml#, where #<unique-id># is a unique 
  *  numeric identified composed of the date and time and a sequence number. 
  *  XML  tables are written whenever a fixed number of triggers (see 
  *  #-ntrig# command line argument) have been accumulated, a HUP signal is 
  *  caught or the watchdog timer expires without any new triggers having 
  *  been received. After writing a trigger table file, TrigMgr submits a 
  *  job to the LDAS Manager API with the ldas_submit script (which must be  
  *  available in the process path) to copy the trigger to the LDAS 
  *  meta-Database.
  *
  *  {\bf Trigger Manager Status:} TrigMgr writes the current status to html
  *  files. This is stored in #$DMTHTMLOUT/TrigMgr.html# making it accessible
  *  via the CDS Web server 
  *  (e.g. #http://blue/gds/monitor_reports/TrigMgr/TrigMgr.html#).
  *  The status page contains a list of all currently registered
  *  processes and a summary of all configured triggers. 
  *  @memo Trigger record broker.
  *  @author J. Zweizig
  *  @version 1.0; Modified April 24, 2003
  */
//@{
//@}

#include "SigFlag.hh"
#include "lmsg/MsgTypes.hh"
#include "AppServer.hh"
#include "lmsg/MsgHandler.hh"
#include "TrigMsg.hh"
#include "TrigStream.hh"
#include "TrigCtlTable.hh"
#include "TrigMgrLog.hh"

namespace html {
  class document;
  class text;
}

#include <vector>

//
//   The Following are bits for special purpose hacks set by the -hack 
//   argument field. These should not be used in general running.
//
//      HACK_SEQDISP  Rewrite the trigger disposition with a sequence number
#define HACK_SEQDISP 1

class TrigMgr : lmsg::AppServer {
public:

  /*   Initialize application.
   */
  TrigMgr(int argc, const char* argv[]);

  /*   Terminate application
   */
  ~TrigMgr(void);

  /*   Perform application processing.
   */
  int Application(void);

  /*   Close all output streams.
   *   @return Returns true if a configuration error occurred.
   */
  void closeStreams(void);

  /*   Configure output streams.
   *   @return Returns true if a configuration error occurred.
   */
  bool configStreams(const std::string& file);

  /*   Get latest write time.
   */
  Time getLastWrite(void) const;

  /*   Get number of triggers ready to be written.
   */
  int getNTrigs(void) const;

  /*   Handle client termination messages.
   */
  lmsg::error_type HandleClose(const lmsg::MsgHeader& h, const TMM_Close& msg);

  /*   Handle client registration messages
   *--------------------------------------*/
  lmsg::error_type HandleRegister(const lmsg::MsgHeader& h, const TMM_Register& msg);

  /*   Handle segment messages
   *--------------------------*/
  lmsg::error_type HandleSegment(const lmsg::MsgHeader& h, const TMM_Segment& msg);

  /*   Handle trigger messages
   *--------------------------*/
  lmsg::error_type HandleTrigger(const lmsg::MsgHeader& h, const TMM_Trigger& msg);

  /*   Handle rmNode messsages
   *--------------------------*/
  lmsg::error_type HandleRmNode(const lmsg::MsgHeader& h, const TMM_RemNode& msg);

  /*   Handle reConfig messsages
   *--------------------------*/
  lmsg::error_type HandleReconfig(const lmsg::MsgHeader& h, const TMM_Reconfigure& msg);

  void PurgeProcDB(void);

  void setTimer(Interval dt);

  void writeStats(void) const;
  void writeSegStats(void) const;
  void WriteDB(bool flush);

  void htmCreateLog(void);
  void htmAddLog(const trig::TrigBase& t, const trig::TrigProc& p);
  void htmWriteLog(void);

private:
  typedef std::vector<TrigStream*> StreamDB;
  typedef StreamDB::iterator stream_iter;

private:
    //----------------------------------  Trigger manager parameters
    bool     mEnableDB;
    bool     mEnableEpX;
    Interval mTimer;
    TrigStream::param mDefParam;
    std::string mOutDir;
    std::string mNotify;
    std::string mConfigFile;
    std::string mStreamConf;

    //----------------------------------  Trigger manager status
    SigFlag mTerm;
    SigFlag mForce;
    SigFlag mReconfigure;
    TrigCtlTable mCTable;
    Time mLastWrite;
    TrigMgrLog mTrigLog;

  //----------------------------------  Process, trigger, segment databases
    ProcessDB mProcDB;
    StreamDB  mStreamList;
};   //   class TrigMgr

#endif  //  TRIGMGR_HH
