#ifndef PROCESS_HH
#define PROCESS_HH

#include "TrigProc.hh"
#include <map>

/*   The process class contains a record of the trigger process volatile 
  *  database consisting of a TrigProc object defining the database, a
  *  counter reflecting the number of triggers received from the process,
  *  and several status flags.
  *  @memo Trigger process database class. 
  *  @author J. Zweizig
  *  @version 1.1; modified January 2, 2001
  */
class Process {
public:
  typedef unsigned long mask_t;
  typedef unsigned long count_t;
public:
  /*   Construct a null process record.
    *  @memo Null constructor.
    */
  Process(void);

  /*   Construct a process record from a TrigProc object.
    *  @memo Dta constructor.
    *  @param x TrigProc object describing the process.
    */
  Process(const trig::TrigProc& x);

  /*   Construct a process entry identical to an existing one.
    *  @memo Copy constructor.
    */
  Process(const Process& x);

  /*   Destroy a process record andete any space used.
    *  @memo Process destructor.
    */
  ~Process(void);

  /*   Get the number of triggers from the process that have been rejected
    *  due to errors.
    *  @memo Get the number of rejected triggers.
    *  @return Rejected trigger count.
    */
  int  getErrorCount(void) const;

  /*   Get the number of trigger received from the process.
    *  @memo Get the received trigger count.
    *  @return Number of triggers received from the process.
    */
  int  getRecvCount(void) const;

  /*   Get the number of triggers from the process that have been written 
    *  to the production database.
    *  @memo Get the number of triggers sent to the production database.
    *  @return Number of triggers written to the production database.
    */
  int  getWriteCount(void) const;

  /*   Test whether the process entry has been closed. Theclosed flag is
    *  set when a TMM_CLOSE message is received from the process. A closed
    *  process entry is not removed from the list until the trigger list 
    *  has been purged.
    *  @memo Test the closed flag.
    */
  bool isClosed(void) const;

   /*  Test if the process entry is releasable. For the process to be 
    *  released, it must be marked closed, and be written to the database or
    *  have a zero write count.
    *  @memo Test if the entry is releasable.
    */
  bool isReleasable(void) const;

  /*   Test the specified 'Written' flag. The 'Written' flag is cleared when 
    *  the Process record is constructed and set after a process meta-database
    *  table entry has been generated.
    *  @memo Test wheter process has been entered in spacified database.
    *  @return true if specified process entry was written.
    *  @param  id   Mask bit for desired database.
    */
  bool isWritten(void) const;

  /*   Increment the count of triggers and segments received from process.
    *  All triggers and segments are counted including duplicates that 
    *  are ignored at a lated point in the message handler.
    *  @memo Increment the received trigger counter.
    */
  void incrementRecv(void);

  /*   Increment the number of triggers or segments from this process that
    *  have been added to the primary database list.
    *  @memo Increment the written trigger counter.
    */
  void incrementWrite(void);

  /*   Increment the number of triggers or segments from this process that
    *  have been rejected because of errors.
    *  @memo Increment the error trigger counter.
    */
  void incrementError(void);

  /*   Increment the number of triggers or segments from this process that
    *  have been added to the alternate (test) database list.
    *  @memo Increment the written trigger counter.
    */
  void incrementTest(void);

  /*   Set the status flag to indicate that the process entry has been 
    *  written to the specified database. A process entry is written only 
    *  after one or more triggers from the process have been written.
    *  @memo Set process entry written flag.
    */
  void setWritten();

  /*   Set the status flag to indicated that a TMM_CLOSE message has been
    *  received from the associated process.
    */
  void setClosed(void);

  /*   Return a constant reference to the TrigProc data record for the 
    *  associated process.
    */
  const trig::TrigProc& refData(void) const;
  trig::TrigProc& refData(void);

  /*   Test whether a message ID is new.
    */
  bool testMsgID(int msgid);

private:
  trig::TrigProc mProcess;
  count_t        mRecvCount;
  count_t        mWriteCount;
  count_t        mErrorCount;
  bool           mWritten;
  bool           mClosed;
  int            mLastMsg;
};

//======================================  Process data base type
typedef std::map<std::string, Process> ProcessDB;
typedef ProcessDB::iterator proc_iter;
typedef ProcessDB::const_iterator const_proc_iter;

//======================================  In-line Process methods.
inline bool 
Process::isWritten(void) const {
    return mWritten;
}

inline bool 
Process::isClosed(void) const {
    return mClosed;
}

inline int
Process::getErrorCount(void) const {
    return mErrorCount;
}

inline int
Process::getRecvCount(void) const {
    return mRecvCount;
}

inline int
Process::getWriteCount(void) const {
    return mWriteCount;
}

inline void 
Process::incrementError(void) {
    mErrorCount++;
}

inline void 
Process::incrementRecv(void) {
    mRecvCount++;
}

inline void 
Process::incrementWrite(void) {
    mWriteCount++;
}

inline bool 
Process::isReleasable(void) const {
    return mClosed && (mWritten || !mWriteCount);
}
	   
inline const trig::TrigProc& 
Process::refData(void) const {
    return mProcess;
}

inline trig::TrigProc& 
Process::refData(void) {
    return mProcess;
}

#endif // !defined(PROCESS_HH)
