/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGMGRCLIENT_HH
#define TRIGMGRCLIENT_HH

#include "TrigWriter.hh"
#include "lmsg/MsgTypes.hh"
#include <string>

namespace trig {
    class TrigBase;
    class Segment;
}

/**  The %TrigClient class provides the client API for the trigger manager
  *  (TrigMgr) service. During off-line running, the trigger manager collects 
  *  trigger and writes then to a speciifed file-type.
  *  <h3> Online Running with the Trigger Manager </h3>
  *  The trigger manager maintains a registry of monitors 
  *  that  may produce triggers. The client API requests that it be added to
  *  the registry with the enroll method. The client may then send one or 
  *  more triggers to the trigger manager with the sendTrigger() method.
  *  <h3> Offline Trigger File Writing </h3>
  *  The trigger data may be also be saved and written to an appropriate file 
  *  periodically or when the %TrigClient is closed. Selecting an output file
  *  type is accomplished either by setting the file mode in the %TrigClient 
  *  contructor or the enroll function. The file types are specified by the
  *  TrigWriter::trig_mode enumerator. The output file type and path may 
  *  also be set by setting an appropriately named environment variable to 
  *  the file path. The currently available output modes are summarized in the
  *  table below:
  *  <center> <table>
  *  <tr><td>Mode</td><td>Environment</td><td>Data type</td></tr>
  *  <tr><td>kNone</td><td> - </td><td>File type is selected from the 
  *  environment at the time enroll is invoked.</td></tr>
  *  <tr><td>kMgr</td><td> - </td><td>Send trigger or segments directly to 
  *                                   the trigger manager</td></tr>
  *  <tr><td>kWriter</td><td>TRIGGER_FILE</td><td>LDAS Database</td></tr>
  *  <tr><td>kSegWrt</td><td>SEGMENT_FILE</td>
  *                      <td>Send to segment file.</td></tr>
  *  <tr><td>kS6Seg</td><td>S6SEGMENT_FILE</td>
*                     <td>Send to S6 segment file.</td></tr>
  *  <tr><td>kSBTrig</td><td>SBTRIGGER_FILE</td>
  *                     <td>Single-burst trigger xml file.</td></tr>
  *  </table> </center>
  *  The output file type may be selected by specifying a trig_mode flag
  *  (see the TrigWriter environment), by setting an appropriate environment
  *  variable to the path of the trigger file. The writer mode may also be 
  *  set explicitly in the %TrigClient constructor, or the enroll() method.
  *  \brief Trigger Client API
  *  \author John G. Zweizig
  *  \version $Id: TrigClient.hh 6802 2013-04-22 06:21:02Z john.zweizig@LIGO.ORG $
  */
class TrigClient {
public:
    /**  Construct a trigger client.
      *  @memo Construct a trigger client.
      */
    TrigClient(void);

    /**  Construct a trigger client with the specified mode. The trigger client
      *  structure is initialized and if the mode is non-zero, the output 
      *  mode is set and enrollment takes place as described below in the
      *  enroll() method.
      *  @memo Construct a trigger client.
      *  @param mode Trigger output mode as specified for the enroll() method.
      */
    TrigClient(trig::TrigWriter::trig_mode mode);

    /**  Destroy a trigger client.
      *  @memo Destroy a trigger client.
      */
    virtual ~TrigClient(void);

    /**  Write any remaining table data and close and delete the writer.
      *  @memo Close a trigger client.
      */
    void close(void);

    /**  The trigger client is initialized for use by this process. The
      *  \a mode argument selects an output file type or the trigger
      *  manager to receive the trigger data. If the \a mode argument is
      *  set to \e kNone, the enroll() method will search to determine whether 
      *  one of the environment variables in the table above has been set.
      *  If an environment vasriable is set, the client is placed in the 
      *  corresponding mode and the content of the variable is use as the 
      *  output file path. If none the environment variables are set, the 
      *  client instance will attempt to register itself with the trigger 
      *  manager. On successful registration, the trigger manager issues a 
      *  unique process identifier which is used to identify the process in 
      *  subsequent requests.
      *  \brief Enroll process with the Trigger Manager.
      *  \param mode Writer type selection.
      *  \return Error code enumerated in lmsg::Errorlist.hh.
      */
    lmsg::error_type enroll(trig::TrigWriter::trig_mode mode=trig::TrigWriter::kNone);

     /**  Flush any existing triggers or segments in the specified interval 
       *  to the appropriate output stream.
       *  @memo Get the debug level.
       *  \param start %Interval start time.
       *  \param end   %Interval end time.
       *  \return lmsg error return code.
       */
    lmsg::error_type flush(const Time& start, const Time& end);

     /**  Get the debug printout level.
       *  @memo get the debug level.
       *  \return Debug message level
       */
    lmsg::index_type getDebug(void) const;

    /**  Send a segment to be logged by the trigger manager.
      *  @memo Log a Segment with the trigger manager.
      *  @return Error code enumerated in lmsg::Errorlist.hh.
      *  @param segm Segment to be logged in the metadata data base.
      */
    lmsg::error_type sendSegment(const trig::Segment& segm);

    /**  Send a trigger to the trigger manager.
      *  @memo Log a trigger with the trigger manager.
      *  @return Error code enumerated in lmsg::Errorlist.hh.
      *  @param trigger Trigger to be logged in the metadata data base.
      */
    lmsg::error_type sendTrigger(const trig::TrigBase& trigger);

    /**  Set the debug level.
      *  @memo Set the debug level.
      *  @param level New debugging level.
      */
    void setDebug(lmsg::index_type level);

    /**  Set the table output file name.
      *  @memo Set the table file.
      *  @param name Table file name
      */
    void setTableFile(const char* name);

private:
    /**  This function writes out any data that is stored in the internal 
      *  database tables.
      */
    void writeTables(void);

private:
    lmsg::index_type  mDebug;
    trig::TrigWriter* mWriter;
    std::string       mTableFile;
};

//======================================  Inline methods.
inline lmsg::index_type 
TrigClient::getDebug(void) const {
    return mDebug;
}

#endif  //   TRIGMGRCLIENT_HH
