#ifndef ALARMMGR_HH
#define ALARMMGR_HH
/**  @name Alarm Manager
  *  The alarm manager class will in general have exactly one instance: 
  *  the alarm manager process. The alarm manager allows monitors to define 
  *  alarms followed optionally by alarm reports. The alarms are maintained
  *  in an internal database and are used to generate EPICS alarms. The
  *  alarm manager also produces html status reports that summarize the
  *  state of the alarm manager and recent alarm activity.
  *
  *  The alarm monitor process is run with the following command line:
\begin{verbatim}
   AlarmMgr [-debug <level>]
\end{verbatim}
  *  Where the meanings of the flags are summarized in the following table:
  *
  * \begin{tabular}{|l|l|}
  * #-debug <level>#       & Specifies a debug printout level. The default 
  *                          level (0) indicates no debugging messages are 
  *                          to be produced. The volume of messages increase 
  *                          with the level. \\
  * \end{tabular}
  *
  * {\Large AlarmMgr Inputs:} \\
  *
  * {\bf Configuration File} \\
  * The AlarmMgr configuration file specifies which alarms are accepted 
  * by the alarm manager and optionally places limits on the disposition 
  * of the alarm. The alarm configuration file contains one or more 
  * lines, each of which specifies the disposition of a group of alarms.
  * If a alarm matches more than one alarm group, the first group 
  * specified will take precedence.
  * Each line contains the following fields separated by spaces:
  *
  * \begin{tabular}{|l|l|}
  * Monitor Name & Name of the monitor generating the specified alarm \\
  * Alarm ID & The alarm ID string \\
  * #-dmask <mask># & Mask of allowed dispositions for the alarm group. \\
  * #-maxprty <max># & Maximum allowed priority for the alarm group.
  * \end{tabular}
  * The first four fields must be specified in the order listed. An asterisk
  * (*) may be used to allow any value for the positional fields.
  *
  * {\bf Messages:} \\
  * The alarm manager accepts monitor messages as defined 
  * in AlarmMsg.hh. The following message types are handled:
  *
  * \begin{tabular}{|l|l|}
  * #TMM_ALARM# & Defines a alarm to be processed. The alarm may be 
  * forwarded to either the EPICS alarm system or the meta-database or both 
  * depending on the alarm disposition field and the alarm control 
  * database.  The alarm manager will reply with either a #TMM_Ack# or 
  * #TMM_NAck# message depending on whether the alarm and alarming 
  * process were recognised. \\
  * #TMM_NAck# message depending on whether the alarm and alarming 
  * process were recognised. \\
  * #TMM_Close# & Causes the specified process to be unregistered. \\
  * #<other># & Other message types are ignored and no reply is returned.
  *\end{tabular}
  *
  * {\bf Signals:} \\
  * Signals are used to cause asynchronous state changes.
  *
  * \begin{tabular}{|l|l|} 
  * #SIGHUP#  & Reset the alarm disposition database. \\
  * #SIGINT#  & write out pending alarms and terminate. \\
  * #SIGTERM# & write out pending alarms and terminate.
  * \end{tabular}
  *
  *  {\bf AlarmMgr Outputs:}
  *
  *  {\bf Alarm meta-Data Tables:} AlarmMgr writes the summarized alarm 
  *  data to XML table files. The table files are given a unique name of 
  *  the form #AlarmMgr<unique-id>.xml#, where #<unique-id># is a unique 
  *  numeric identified composed of the date and time and a sequence number. 
  *  XML  tables are written whenever a fixed number of alarms (see 
  *  #-nalarm# command line argument) have been accumulated, a HUP signal is 
  *  caught or the watchdog timer expires without any new alarms having 
  *  been received. After writing a alarm table file, AlarmMgr submits a 
  *  job to the LDAS Manager API with the ldas_submit script (which must be  
  *  available in the process path) to copy the alarm to the LDAS 
  *  meta-Database.
  *
  *  {\bf Alarm Manager Status:} AlarmMgr writes the current status to html
  *  files. This is stored in #$DMTHTMLOUT/AlarmMgr.html# making it accessible
  *  via the CDS Web server 
  *  (e.g. #http://blue/gds/monitor_reports/AlarmMgr/AlarmMgr.html#).
  *  The status page contains a list of all currently registered
  *  processes and a summary of all configured alarms. 
  *  @memo Alarm record broker.
  *  @author J. Zweizig
  *  @version $Revision: 2258 $; Modified $Date: 2003-06-10 13:24:23 -0700 (Tue, 10 Jun 2003) $
  */
#include "SigFlag.hh"
#include "AppServer.hh"
#include "lmsg/MsgHandler.hh"
#include "AlarmMsg.hh"
// #include "AlarmCtlTable.hh"
// #include "AlarmMgrLog.hh"
#include "TimeStamp.hh"

namespace html {
  class document;
  class text;
}

#include <list>
#include <map>

class AlarmMgr : lmsg::AppServer {
public:

  /*   Initialize application.
   */
  AlarmMgr(int argc, const char* argv[]);

  /*   Terminate application
   */
  ~AlarmMgr(void);

  /*   Perform application processing.
   */
  int Application(void);

  /*   Remove expired alarms
   *------------------------*/
  void Expire(const Time& now);

  /*   Handle status request messages
   *---------------------------------*/
  lmsg::error_type HandleRequest(const lmsg::MsgHeader& h, const ALM_Request& msg);

  /*   Handle alarm acknowledgement messages
   *----------------------------------------*/
  lmsg::error_type HandleAcknowledge(const lmsg::MsgHeader& h, const ALM_Acknowledge& msg);

  /*   Handle active alar requests
   *------------------------------*/
  lmsg::error_type HandleGetActive(const lmsg::MsgHeader& h, const ALM_GetActive& msg);

  /*   Handle defined alarm requests
   *--------------------------------*/
  lmsg::error_type HandleGetDefined(const lmsg::MsgHeader& h, const ALM_GetDefined& msg);

  /*   Handle alarm definition messages
   *-----------------------------------*/
  lmsg::error_type HandleDefine(const lmsg::MsgHeader& h, const ALM_Define& msg);

  /*   Handle set messages
   *----------------------*/
  lmsg::error_type HandleSet(const lmsg::MsgHeader& h, const ALM_Set& msg);

  /*   Handle prolongation messages
   *-------------------------------*/
  lmsg::error_type HandleProlong(const lmsg::MsgHeader& h, const ALM_Prolong& msg);

  /*   Handle cancelation messages
   *------------------------------*/
  lmsg::error_type HandleCancel(const lmsg::MsgHeader& h, const ALM_Cancel& msg);

  /*   Handle cancelation messages
   *------------------------------*/
  lmsg::error_type HandleSetDisable(const lmsg::MsgHeader& h, const ALM_SetDisable& msg);

  /*   Handle cancelation messages
   *------------------------------*/
  lmsg::error_type HandleGetTemplate(const lmsg::MsgHeader& h, const ALM_GetTemplate& msg);
  /*   Handle cancelation messages
   *------------------------------*/
  lmsg::error_type HandleRemove(const lmsg::MsgHeader& h, const ALM_Remove& msg);


  void setTimer(void);
  void writeStats(void) const;


private:
  typedef std::map<AlarmHandle, AlarmData> ActiveDB;
  typedef ActiveDB::iterator activ_iter;
  typedef ActiveDB::const_iterator const_activ_iter;

  typedef std::list<AlarmData> AlarmDB;
  typedef AlarmDB::iterator alarm_iter;
  typedef AlarmDB::const_iterator const_alarm_iter;

  const_alarm_iter findAlarm(const AlarmID& a) const;
  alarm_iter       findAlarm(const AlarmID& a);
  void logAction(const std::string& action, const const_activ_iter& alarm);

private:
  //------------------------------------  Alarm manager parameters
  double   mTimer;
  Interval mInactive;
  Interval mMaxDelay;
  unsigned long mHack;

  //------------------------------------  Alarm manager status
  SigFlag mTerm;
  SigFlag mForce;
  // AlarmCtlTable mCTable;
  TimeStamp mNow;
  Time mNext;    /// Time that next alarm expires.

  //------------------------------------  Defined, active alarm databases
  AlarmDB   mAlarmDB;
  ActiveDB  mActiveDB;
};   //   class AlarmMgr

#endif  //  ALARMMGR_HH
