#ifndef ALARMMSG_HH
#define ALARMMSG_HH

#include "AlarmData.hh"
#include "lmsg/DataMsg.hh"
#include <vector>

#define AlarmServerName "Alarm Manager"

enum AlrmMsgID {
    ///  Tell server to define alarm (AlarmData)
    almDefine = 131072,
    ///  Server reference reply (Handle)
    almReference,
    ///  Set alarm (AlarmData)
    almSet,
    ///  Alarm status reply (AlarmData)
    almStatus,
    ///  Get list of defined alarms (AlarmID template)
    almGetDefined,
    ///  Get list of active alarms (AlarmID template)
    almGetActive,
    ///  Acknowledge Alarm (Handle)
    almAcknowledge,
    ///  Request Alarm Status (Handle)
    almRequest,
    ///  Cancel Alarm (Handle)
    almCancel,
    ///  Prolong Alarm (Handle, double)
    almProlong,
    ///  Active handles reply
    almActiveReply,
    ///  Active handles reply
    almDefinedReply,
    ///  Success
    almAck,
    ///  Failure
    almNAck,
    ///  Set disable bit in template (AlarmID, int)
    almSetDisable,
    ///  Get alarm template (AlarmID)
    almGetTemplate,
    ///  Remove alarm definition (AlarmID)
    almRemove
};

//======================================  Define Templated messages

/** The alarm define message is used to define an alarm and set the 
  * default parameters.
  */
typedef lmsg::DataMsg<AlarmData, almDefine> ALM_Define;

/** The getActive message is used to request a list of active alarm handles.
  */
typedef lmsg::DataMsg<AlarmID, almGetActive> ALM_GetActive;

/** The getActive message is used to request a list of active alarm handles.
  */
typedef lmsg::DataMsg<AlarmID, almGetDefined> ALM_GetDefined;

/**  ALM_Set message sets an alarm. The alarm manager returns a Handle of 
  *  the alarm instance created.
  */
typedef lmsg::DataMsg<AlarmData, almSet> ALM_Set;

/**  ALM_Status message sets an alarm. The alarm manager returns a Handle of 
  *  the alarm instance created.
  */
typedef lmsg::DataMsg<AlarmData, almStatus> ALM_Status;

/**  ALM_Reference contains reference to an alarm. It is a response to a
  *  ALM_Set message.
  */
typedef lmsg::DataMsg<AlarmHandle, almReference> ALM_Reference;

/**  ALM_Acknowledge instructs the Alarm manager to mark the specified
  *  alarm as having been acknowledged.
  */
typedef lmsg::DataMsg<AlarmHandle, almAcknowledge> ALM_Acknowledge;

/**  ALM_Cancel instructs the alarm manager to cancel the specified alarm.
  */
typedef lmsg::DataMsg<AlarmHandle, almCancel> ALM_Cancel;

/**  ALM_Request instructs the alarm manager to cancel the specified alarm.
  */
typedef lmsg::DataMsg<AlarmHandle, almRequest> ALM_Request;

/**  ALM_ActiveReply vector of handles in response to a getActive request.
  */
typedef lmsg::DataMsg< std::vector<AlarmHandle>, almActiveReply > ALM_ActiveReply;

/**  ALM_DefinedReply string in response to a getDefined request.
  */
typedef lmsg::DataMsg< std::string, almDefinedReply> ALM_DefinedReply;

/**  ALM_SetDisable - Set disable bit in template (AlarmID, int)
  */
typedef lmsg::DataMsg< std::pair<AlarmID, int>, almSetDisable > ALM_SetDisable;
/**  ALM_GetTemplate -  Get alarm template (AlarmID)
  */
typedef lmsg::DataMsg< AlarmID, almGetTemplate > ALM_GetTemplate;

/**  ALM_Remove - Remove alarm definition (AlarmID)
  */
typedef lmsg::DataMsg< AlarmID, almRemove > ALM_Remove;

/**  ALM_Prolong requests that an active alarm be prolonged. 
  */
typedef lmsg::DataMsg< std::pair<AlarmHandle, double>, almProlong > ALM_Prolong;

//======================================  Define header-only messge types
typedef lmsg::HdrOnlyMsg<almAck>  ALM_Ack;
typedef lmsg::HdrOnlyMsg<almNAck> ALM_NAck;

#endif // !defined(ALARMMSG_HH)
