#include <iostream>
#include <sstream>
#include <cstdlib>
#include <cstring>
#include "xml/Xsil.hh"
#include "xml/XsilTSeries.hh"
#include "xml/XsilFSpectrum.hh"
#include "xml/XsilHistogram.hh"
#include "TrigBase.hh"
#include "DVector.hh"
#include "Histogram1.hh"
#include "chInterface.hh"

namespace channel {
using namespace std;
using namespace xml;

/***************************************************************************
 ***************************************************************************
 **
 ** chInterface functions
 ** These functions are arranged as follows:
 **  ** standard class functions
 **  ** chInterface modifiers
 **  ** chInterface accessors
 **  ** chInterface miscellaneous functions
 **  ** chInterface functions related to filters
 **  ** chInterface functions related to glitches
 **  ** chInterface functions related to statistics
 **  ** chInterface functions related to results
 **
 ***************************************************************************
 ***************************************************************************/

/***************************************************************************
 *
 * chInterface standard class functions
 * (chInterface, ~chInterface, operator =)
 *
 ***************************************************************************/

/******************* Default constructor */
chInterface::chInterface(unsigned int debug) :mName(""),
  mSampleFreq(0), mOSCSatisfy(""), mDebug(debug), mInit(0), mData(0), 
  mDaccPtr(0), mTmpTime(0) {
  if (mDebug > 1) cout <<"Creating empty chInterface."<<endl;
  if (mDebug > 3) cout <<"   (chInterface "<<mName<<") "<<this<<endl;
  mData = new TSeries();
  if (mDebug > 3) cout <<"   Creating (mData) "<<mData<<endl;
  FBank.push_back(FilterStatus("raw", mDebug));
}

/******************* Do we need another constructor? */
chInterface::chInterface(string name, string osc, DaccAPI *dacc,
			 unsigned int debug) :
  mName(name), mSampleFreq(0), mOSCSatisfy(""), mDebug(debug), mInit(0),
  mData(0), mDaccPtr(dacc), mTmpTime(0) {
  if (mDebug > 1) cout <<"Creating chInterface -- " <<mName<<endl;
  if (mDebug > 3) cout <<"   (chInterface "<<mName<<") "<<this<<endl;
  if (osc.size() > 3) mOSCSatisfy=osc;
  if (mDaccPtr) {
    try           { mDaccPtr->setDebug(mDebug); }
    catch (int n) { if (n) mDaccPtr = 0; }
  }
  mData = new TSeries();
  if (mDebug > 3) cout <<"   Creating (mData) "<<mData<<endl;
  if (mDaccPtr) mDaccPtr->addChannel(mName.c_str(), 0, &mData);
  FBank.push_back(FilterStatus("raw", mDebug));
}

/// creates a chInterface instance, using another chInterface
chInterface::chInterface(const chInterface& ch) : mData(0) {
  if (ch.mDebug > 1) cout <<"Copying chInterface -- "<<ch.mName<<endl;
  *this = ch; 
}

/// destroys a chInterface instance
chInterface::~chInterface(void) {
  if (mDebug > 1) cout <<"Destroying chInterface -- " <<mName<<endl;
  if (mDebug > 3) cout <<"   (chInterface "<<mName<<") "<<this<<endl;
  if (mData) {
    if (mDebug > 3) cout <<"Destroying (mData) "<<mData<<endl;
    delete mData;
  }
  if (mDebug > 3) cout <<"Destroying (mGBank) "<<&mGBank<<endl;
  while (!mGBank.empty()) {
    delete mGBank.begin()->second;
    mGBank.erase(mGBank.begin());
  }
  if (mDebug > 3) cout <<"Destroying (mSBank) "<<&mSBank<<endl;
  while (!mSBank.empty()) {
    delete mSBank.begin()->second;
    mSBank.erase(mSBank.begin());
  }
  if (mDebug > 3) cout <<"Destroying (mRBank) "<<&mRBank<<endl;
  while (!mRBank.empty()) {
    delete mRBank.begin()->second;
    mRBank.erase(mRBank.begin());
  }
}

/// copies a chInterface instance
chInterface& chInterface::operator = (const chInterface& ch) {
  //  Reset();
  mDebug           = ch.mDebug;
  if (ch.mData) {
    mData = new TSeries(*ch.mData);
    const TSeries *cts = mData;
    if (mDebug > 3) cout <<"   Creating (mData) "<<mData<<endl;
    if (mDebug > 3) cout <<"TESTING mData: isGood()="<<mData->isGood()
			 <<",\tisEmpty()="<<mData->isEmpty()
			 <<",\trefDVect()="<<cts->refDVect()<<endl;
  }
  mName            = ch.mName;
  mSampleFreq      = ch.mSampleFreq;
  mOSCSatisfy      = ch.mOSCSatisfy;
  mInit            = ch.mInit;
  mDaccPtr         = ch.mDaccPtr;
  mTmpTime         = ch.mTmpTime;
  if (mDaccPtr) {
    mDaccPtr->rmChannel(mName.c_str());
    mDaccPtr->addChannel(mName.c_str(), 0, &mData);
  }
  /* Now the channel structures */
  FBank            = ch.FBank;
  for (GBciter i=ch.mGBank.begin(); i != ch.mGBank.end(); i++)
    addGlitch(i->first, *(i->second));
  for (SBciter i=ch.mSBank.begin(); i != ch.mSBank.end(); i++)
    addStatistic(i->first, *(i->second));
  for (RBciter i=ch.mRBank.begin(); i != ch.mRBank.end(); i++)
    addResult(i->first, *(i->second));

  if (mDebug > 3) cout <<"   (chInterface "<<mName<<") "<<this<<endl;
  return *this;
}  // chInterface& operator = (const chInterface &ch) {


/***************************************************************************
 *
 * chInterface modifiers
 * (setName, setOSC) -- in .hh file
 * chInterface accessors
 * (getName, getOSC, getData) -- in .hh file
 *
 ***************************************************************************/


/***************************************************************************
 *
 * chInterface miscellaneous functions
 * (Init, isTSContinuous, isInitialized)
 *
 ***************************************************************************/

/// Initialize any unset parameters in filters/glitches/statistics/results
bool chInterface::Init(void) {
  if (mDebug > 2) cout <<"chInterface: Initializing "<<mName<<endl;
  string chName("chInterface");
  bool allInitialized=true;
 if (mName.empty()) {
    cout <<"chInterface doesn't have a name."<<endl;
    allInitialized=false;
  } else chName = mName;
  if (mDaccPtr) {
    if (!mDaccPtr->isChannelRead(mName.c_str())) {
      mDaccPtr->addChannel(mName.c_str(), 0, &mData);
      mDaccPtr->synch();
      mDaccPtr->fillData(Interval(1.0));
    }
    mSampleFreq = mData->getNSample();
  } else {
    if (mDebug)
      cout <<chName<<" doesn't have a working Dacc.  Use getDacc()."<<endl;
    allInitialized=false;
  }
  if (mOSCSatisfy.empty()) {
    if (mDebug) cout <<chName<<" doesn't have a relevant OSC."<<endl;
    //    allInitialized=false;
  }
  if (!mSampleFreq) {
    if (mDebug) cout <<chName<<" doesn't have a sampling rate."<<endl;
    allInitialized=false;
  } else {
    for (FBiter  i=FBank.begin();  i !=  FBank.end(); i++) {
      i->setSampleRate(mSampleFreq);
      unsigned int usefilter=i->getUseFilterID();
      if (i->setupFilter(usefilter)) allInitialized &= i->isInitialized();
    }
  }
  if (mGBank.empty() && mSBank.empty()) {
    if (mDebug)
      cout <<chName<<" doesn't have glitches or statistics defined."<<endl;
    allInitialized=false;
  }
  for (GBciter i=mGBank.begin(); i != mGBank.end(); i++)
    allInitialized &= i->second->isInitialized();
  if (mDebug > 2)
    cout <<"chInterface {after mGBank[*].isInitialized()} : allInitialized = "
	 <<allInitialized<<endl;
  for (SBciter i=mSBank.begin(); i != mSBank.end(); i++)
    allInitialized &= i->second->isInitialized();
  if (mDebug > 2)
    cout <<"chInterface {after mSBank[*].isInitialized()} : allInitialized = "
	 <<allInitialized<<endl;
  if (mRBank.empty()) {
    if (mDebug) cout <<chName<<" doesn't have a result."<<endl;
    allInitialized=false;
  }
  for (RBciter i=mRBank.begin(); i != mRBank.end(); i++)
    allInitialized &= i->second->isInitialized();
  if (mDebug > 2)
    cout <<"chInterface {after mRBank[*].isInitialized()} : allInitialized = "
	 <<allInitialized<<endl;
  mInit = allInitialized;
  return mInit;
}

/// Is the new channel data a continuation of the previous data
bool chInterface::isTSContinuous(void) {
  bool cont=true;
  if (mTmpTime.getS()) {
    if (Almost( mData->getStartTime(), (mTmpTime + mData->getInterval()) ))
      cont=false;
    mTmpTime = mData->getEndTime();
  }
  return cont;
}

/// isInitialized checks if the chInterface is fully initialized -- in .hh file

/***************************************************************************
 *
 * chInterface functions related to filters
 * (refFilter, addFilter, applyFilters, ResetFilters)
 *
 ***************************************************************************/

/// 'refFilter' references a FilterStatus object -- in .hh file

/// adds a FilterStatus object using vectors of poles and zeroes
void chInterface::addFilter(const string& filterID, vector<dComplex>* poles,
			    vector<dComplex>* zeroes, double gain,
			    const string& useFilterID) {
  if (mDebug) cout <<"chInterface: Adding Filter -- " <<filterID<<endl;
  FBank.push_back(FilterStatus(filterID, mDebug));

  unsigned int usefilter=0;	// Identify which filtered data to use
  if( !useFilterID.empty() ) {
    for( unsigned int i=0;  i < FBank.size(); i++ )
      if( FBank[i].getName() == useFilterID ) {
	if (mDebug > 1) cout <<"Filter "<<filterID<<" using filter "
			     <<useFilterID<<" ["<<i<<"] as input"<<endl;
	usefilter=i;
	break;
      }
    if ( !usefilter && (mDebug > 1) ) cout <<"Can't find input filter "
					   <<useFilterID<<endl;
  }
  if (mDebug > 1) cout <<"gain: "<<gain<<endl;

  FBank.back().setupFilter(usefilter, poles, zeroes, gain);
}

/// adds a FilterStatus object using 
void chInterface::addFilter(const string& filterID, const char *pstr,
			    const char *zstr, double gain,
			    const string& useFilterID) {
  if (mDebug) cout <<"chInterface: Adding Filter -- " <<filterID<<endl;
  FBank.push_back( FilterStatus(filterID, mDebug) );

  unsigned int usefilter=0;	// Identify which filtered data to use
  if ( !useFilterID.empty() ) {
    for ( unsigned int i=0; i < FBank.size(); i++ )
      if ( FBank[i].getName() == useFilterID ) {
	if (mDebug > 1) cout <<"Filter "<<filterID<<" using filter "
			     <<useFilterID<<" ["<<i<<"] as input"<<endl;
	usefilter=i;
	break;
      }
    if ( !usefilter && (mDebug > 1) ) cout <<"Can't find input filter "
					   <<useFilterID<<endl;
  }

  vector<dComplex> poles,zeroes; // Parse out the poles and zeroes
  const char* ptr = pstr;
  double twopi= 2 * 3.1415926535;
  if (mDebug > 3) cout << "twopi = " << twopi << endl;

  if ( strlen(ptr) > 2 ) {	// default case has poles = zeroes = "0"
    if (mDebug > 2) cout << "Poles string length: "<< strlen(ptr)<<endl;
    while (*ptr) {
      double re = strtod(ptr, const_cast<char**>(&ptr));
      double im = strtod(ptr, const_cast<char**>(&ptr));
      if (mDebug > 1) cout << "pole "<<poles.size()<<" : "<< re<<" + "
			   <<im<<"i"<<endl;
      poles.push_back( dComplex(re,im) / twopi );
    }
  
    ptr = zstr;
    if (mDebug > 2) cout << "Zeroes string length: "<< strlen(ptr)<<endl;
    while (*ptr) {
      double re = strtod(ptr, const_cast<char**>(&ptr));
      double im = strtod(ptr, const_cast<char**>(&ptr));
      if (mDebug > 1) cout <<"zero "<<zeroes.size()<<" : "<< re<<" + "
			   <<im<<"i"<<endl;
      zeroes.push_back(dComplex(re,im)/twopi);
    }
  }
  if (mDebug > 1) cout <<"gain: "<<gain<<endl;
				// Setup the filter coefficients
  FBank.back().setupFilter(usefilter, &poles, &zeroes, gain);
}

/// applies (one or all) filters to the channel data
void chInterface::applyFilters(const string& filterID) {
  if (!mInit) cerr <<"chInterface "<<mName<<"isn't initialized yet."<<endl;
  if (!mInit && !mDebug) throw;

  for (FBiter i=FBank.begin(); i != FBank.end(); i++) {
    if (mDebug > 1)
      cout <<"Comparing IDs: "<<filterID<<" vs "<<i->getName()<<endl;
    if ( !filterID.empty() && filterID != i->getName() ) continue;
  //-----------  Need to have a different response for "raw" filter
    if (i->getName() == "raw") {
      if ( !i->isUpdated(mData->getEndTime()) ) i->apply(mData);
  //-----------  Now we'll just iterate through the rest of them
    } else {
      unsigned int use = i->getUseFilterID();
      applyFilters( FBank[use].getName() );
      if ( !i->isUpdated(FBank[use].getData()->getEndTime()) )
	i->apply(FBank[use].getData());
    }	
    if (filterID.empty()) continue;
    else if (filterID == i->getName()) return;
  }
}

/// resets (one or all) FilterStatus objects
bool chInterface::ResetFilters(const string& filterID) {
  if (mDebug > 1) {
    if (filterID.empty()) cout <<"Resetting all filters."<<endl;
    else cout <<"Resetting filter "<<filterID<<endl;
  }
  bool allset=true;
  for (FBiter i=FBank.begin(); i != FBank.end(); i++) {
    if (filterID.empty() || filterID == i->getName()) {
      unsigned int use = i->getUseFilterID();
      i->setSampleRate(mSampleFreq);
      if (! i->setupFilter(use) ) allset=false;
    }
  }
  return allset;
}

/***************************************************************************
 *
 * chInterface functions related to glitches
 * (refGlitch, addGlitch, applyGlitch, ResetGlitches)
 *
 ***************************************************************************/

/// 'refGlitch' references a GlitchConstruct object -- in .hh file

/// adds a GlitchConstruct-- defaults to base-class definition of a glitch
void chInterface::addGlitch(const string& glitchID) {
  if (mDebug) cout <<"chInterface: Adding default glitch -- " <<glitchID<<endl;
  mGBank[glitchID] = new GlitchConstruct(glitchID,0.0,0.0,mDebug);
}

/// adds a GlitchConstruct-- for adding your own derived glitch construct
void chInterface::addGlitch(const string& glitchID, GlitchConstruct& glitch) {
  if (mDebug) cout <<"chInterface: Adding glitch -- " <<glitchID<<endl;
  mGBank[glitchID] = glitch.clone();
  mGBank[glitchID]->setName(glitchID);
}

/// applies all glitch-tests to one or more (possibly filtered) data
const list<glitchResult>* chInterface::applyGlitch(const string& filterID) {
  mGResults.clear();
  for (FBiter i=FBank.begin(); i != FBank.end(); i++) {
    if (!filterID.empty() && filterID != i->getName() ) continue;
    for (GBiter j=mGBank.begin(); j != mGBank.end(); j++) {
      if (mDebug > 1) cout <<"Applying glitch " <<j->first<<" to data "
			   <<i->getName()<<endl;
      unsigned int use = i->getUseFilterID();
      if (!i->isUpdated( FBank[use].getData()->getEndTime() )) {
	applyFilters( FBank[use].getName() );
	i->apply( FBank[use].getData() );
      }
      list<glitchResult> *grOut(const_cast<list<glitchResult>*>
				(j->second->checkGlitch( *(i->getData()),
							 i->getName() ) ) );
      mGResults.splice(mGResults.end(),*grOut, grOut->begin(), grOut->end());
    }
    if (filterID.empty()) continue;
    else if (filterID == i->getName()) return &mGResults;
  }
  return &mGResults;
}

/// applies a glitch-test to (possibly filtered) data
const list<glitchResult>*
chInterface::applyGlitch(const string& glitchID, FilterStatus& filter ) {
  mGResults.clear();
  if ( !glitchID.empty() && (mGBank.find(glitchID) != mGBank.end()) ) {
    if (mDebug > 1) cout <<"Applying glitch " <<glitchID<<" to data " 
			 <<filter.getName()<<endl;
    unsigned int use = filter.getUseFilterID();
    if (!filter.isUpdated( FBank[use].getData()->getEndTime() )) {
      applyFilters( FBank[use].getName() );
      filter.apply( FBank[use].getData() );
    }
    mGResults = *(const_cast<list<glitchResult>*>
		  (mGBank[glitchID]->
		   checkGlitch( *filter.getData(),filter.getName())));
    return &mGResults;
  } else { cerr<<"Can't find glitch "<<glitchID<<endl; throw;}
}

/*
/// resets (one or all) GlitchConstruct objects
bool chInterface::ResetGlitches(string glitchID="") {
  if (mDebug > 1) {
    if (glitchID.empty()) cout <<"Resetting all glitches."<<endl;
    else cout <<"Resetting glitch "<<glitchID<<endl;
  }
  bool allset=true;
  for (GBiter i=mGBank.begin(); i != mGBank.end(); i++) {
    if (glitchID.empty() || glitchID == i->first) {
      i->second.Reset();
      if ( !i->second.isInitialized() ) allset=false;
    }
  }
  return allset;
}
*/

/***************************************************************************
 *
 * chInterface functions related to statistics
 * (refStats, addStatistics, applyStats, ResetStats)
 *
 ***************************************************************************/

/// 'refStats' references a StatisticsConstruct object (by name) -- in .hh file

/// adds a StatisticsConstruct-- defaults to base-class definition of statistic
void chInterface::addStatistic(const string& statID, bool cumulative) {
  if (mDebug) cout <<"chInterface: Adding default statistic -- "<<statID<<endl;
  updateType stType(uRefreshed);
  if (cumulative) stType = uContinuous;
  mSBank[statID] = new StatisticsConstruct(statID,stType,0.0,0.0,mDebug);
}

/// adds a StatisticsConstruct-- for adding your own derived statistics
void chInterface::addStatistic(const string& statID,
			       StatisticsConstruct& statistic) {
  if (mDebug) cout <<"chInterface: Adding statistic -- "<<statID<<endl;
  mSBank[statID] = statistic.clone();
  mSBank[statID]->setName(statID);
}

/// applies all statistics calculation to one or more (possibly filtered) data
const list<statResult>* chInterface::applyStats(const string& filterID) {
  mSResults.clear();
  for (FBiter i=FBank.begin(); i != FBank.end(); i++) {
    if (!filterID.empty() && filterID != i->getName() ) continue;
    for (SBiter j=mSBank.begin(); j != mSBank.end(); j++) {
      if (mDebug > 1) cout <<"Applying statistic " <<j->first<<" to data "
			   <<i->getName()<<endl;
      unsigned int use = i->getUseFilterID();
      if (!i->isUpdated( FBank[use].getData()->getEndTime() )) {
	applyFilters( FBank[use].getName() );
	i->apply( FBank[use].getData() );
      }
      list<statResult> *srOut( const_cast<list<statResult>*>
			       (j->second->updateStats( *(i->getData()),
							i->getName() ) ) );
      mSResults.splice(mSResults.end(),*srOut, srOut->begin(), srOut->end());
    }
    if (filterID.empty()) continue;
    else if (filterID == i->getName()) return &mSResults;
  }
  return &mSResults;
}

/// applies a statistics calculation to (possibly filtered) data
const list<statResult>*
chInterface::applyStats(const string& statID, FilterStatus& filter ) {
  mSResults.clear();
  if ( !statID.empty() && (mSBank.find(statID) != mSBank.end()) ) {
    if (mDebug > 1) cout <<"Applying statistic " <<statID<<" to data "
			 <<filter.getName()<<endl;
    unsigned int use = filter.getUseFilterID();
    if (!filter.isUpdated( FBank[use].getData()->getEndTime() )) {
      applyFilters( FBank[use].getName() );
      filter.apply( FBank[use].getData() );
    }
    mSResults = *(const_cast<list<statResult>*>
		  (mSBank[statID]->
		   updateStats( *filter.getData(),filter.getName())));
    return &mSResults;
  } else {cerr<<"Can't find statistic "<<statID<<endl; throw;}
}

void chInterface::dumpStats( std::ostream& out, unsigned int total ) {
  for (SBiter j=mSBank.begin(); j != mSBank.end(); j++) {
    if (mDebug > 1) cout <<"Dumping statistic " <<j->first<<endl;
      j->second->dumpResults( out, total );
    }
}

/// resets (one or all) StatisticsConstruct objects
bool chInterface::ResetStats(const string& statID, bool cumulative) {
  if (mDebug > 1) {
    if (statID.empty()) cout <<"Resetting all statistics."<<endl;
    else cout <<"Resetting statistic "<<statID<<endl;
  }
  bool allset=true;
  for (SBiter i=mSBank.begin(); i != mSBank.end(); i++) {
    if (statID.empty() || statID == i->first) {
      i->second->Reset(cumulative);
      if ( !i->second->isInitialized() ) allset=false;
    }
  }
  return allset;
}


/***************************************************************************
 *
 * chInterface functions related to results
 * (refResult, addResult, logResult, triggerResult, ResetResults)
 *
 ***************************************************************************/

/// 'refResult' references a ResultConstruct object (by name) -- in .hh file

/// adds a ResultConstruct object to the list
void chInterface::addResult(const string& resultID, bool cumulative) {
  if (mDebug) cout <<"chInterface: Adding default result -- " <<resultID<<endl;
  updateType reType(uRefreshed);
  if (cumulative) reType = uContinuous;
  mRBank[resultID] = new ResultConstruct(resultID,reType,0,cout,0.0,mDebug);
}

/// adds a ResultConstruct-- for adding your own derived result
void chInterface::addResult(const string& resultID, ResultConstruct& result) {
  if (mDebug) cout <<"chInterface: Adding result -- "<<resultID<<endl;
  mRBank[resultID] = result.clone();
  mRBank[resultID]->setName(resultID);
}

/** logs the result of applying (one or all) glitches to (possibly filtered)
  * data
  */
bool chInterface::writeResult(const string& resultID,
			      const list<glitchResult>& results,
			      unsigned int outmask, ostream& out, bool ignorestate) {
  if (mDebug > 1) {
    if (resultID.empty()) cout <<"Writing all results."<<endl;
    else cout <<"Writing result "<<resultID<<endl;
  }
  bool allwrite=true;
  for (RBiter rbi=mRBank.begin(); rbi != mRBank.end(); rbi++) {
    if (resultID.empty() || resultID == rbi->first) {
      unsigned int thismask =  (outmask & 64) ? (outmask ^ 64) : outmask;
      list<glitchResult>::const_iterator     next = results.begin();
      for (list<glitchResult>::const_iterator grs = results.begin();
	   grs != results.end(); grs++) {
	if (next++ == results.end()) thismask = outmask;
	allwrite &= rbi->second->output( *grs, thismask, ignorestate, out );
      }
    }
  }
  return allwrite;
}

/** logs the result of applying (one or all) statistics to (possibly filtered) 
  * data
  */
bool chInterface::writeResult(const string& resultID,
			      const list<statResult>& results,
			      unsigned int outmask, ostream& out, bool ignorestate) {
  if (mDebug > 1) {
    if (resultID.empty()) cout <<"Writing all results."<<endl;
    else cout <<"Writing result "<<resultID<<endl;
  }
  bool allwrite=true;
  for (RBiter rbi=mRBank.begin(); rbi != mRBank.end(); rbi++) {
    if (resultID.empty() || resultID == rbi->first) {
      unsigned int thismask =  (outmask & 64) ? (outmask ^ 64) : outmask;
      list<statResult>::const_iterator     next = results.begin();
      for (list<statResult>::const_iterator srs = results.begin();
	   srs != results.end(); srs++) {
	if (next++ == results.end()) thismask = outmask;
	allwrite &= rbi->second->output( *srs, thismask, ignorestate, out );
      }
    }
  }
  return allwrite;
}

/// resets (one or all) ResultConstruct objects
bool chInterface::ResetResults(const string& resultID, bool cumulative ) {
  if (mDebug) {
    if (resultID.empty()) cout <<"Resetting all results."<<endl;
    else cout <<"Resetting result "<<resultID<<endl;
  }
  bool allset=true;
  for (RBiter i=mRBank.begin(); i != mRBank.end(); i++) {
    if (resultID.empty() || resultID == i->first) {
      i->second->Reset(cumulative);
      if ( !i->second->isInitialized() ) allset=false;
    }
  }
  return allset;
}


/****************************************************************************/
/****************************************************************************/
/*****  Channel Interface Constructs  ***************************************/
/****************************************************************************/
/****************************************************************************/

/***************************************************************************
 *
 * FilterStatus functions
 * (FilterStatus, ~FilterStatus, operator =, apply, convertBuffer, setupFilter,
 *  setUseFilterID, setSampleRate, getName, getUseFilterID, isUpdated,
 *  isInitialized)
 *
 ***************************************************************************/

/// FilterStatus constructor -- in .hh file

/// FilterStatus copy constructor
FilterStatus::FilterStatus(const FilterStatus& filter) : mFilter(0), mTS(0) {
  if (filter.mDebug > 2) cout <<"Copying FilterStatus -- "<<filter.mName<<endl;
  *this = filter;
}

/// FilterStatus destructor
FilterStatus::~FilterStatus(void) {
  if (mDebug > 2) cout <<"Destroying FilterStatus -- "<<mName<<endl;
  if (mFilter) {
    delete mFilter;
    if (mTS) delete mTS;	// We put this here to get around raw data
  }
  if (mDebug > 3) cout <<"   (FilterStatus "<<mName<<") "<<this<<endl;
}

/// FilterStatus assignment operator
FilterStatus& FilterStatus::operator =(const FilterStatus& rhs) {
  mName          = rhs.mName;
  mSampleRate    = rhs.mSampleRate;
  mPoles         = rhs.mPoles;
  mZeroes        = rhs.mZeroes;
  mGain          = rhs.mGain;
  if (rhs.mFilter) mFilter = new IIRFilter(*rhs.mFilter);
  if (rhs.mTS) mTS = new TSeries(*rhs.mTS);
  mInit          = rhs.mInit;
  mDebug         = rhs.mDebug;
  mInFiltID      = rhs.mInFiltID;
  if (mDebug > 3) cout <<"   (FilterStatus "<<mName<<") "<<this<<endl;
  return *this;
}

/// applies an IIRFilter to a time series
const TSeries* FilterStatus::apply(const TSeries* ts) {
  mUpdate = ts->getEndTime();
  // In the case of "raw data" filtering, just return the calling TSeries
  if ( !mFilter || !mInit ) mTS = const_cast<TSeries*>(ts);
  // For all other filters, need to apply the timeseries to the filter
  else *mTS = mFilter->apply(*ts);
  return mTS;
}

/** convert filter order & frequency info to Butterworth poles/zeroes/gain
  * configuration to be backwards-compatible with glitchMon-type filters
  */
void FilterStatus::convertButter(int type, int order, double cutoff_1,
				 double cutoff_2) {
  cout << "FilterStatus::convertButter doesn't work yet." << endl;
}

/** actually create the IIRFilter instance, using sample rate and poles/zeroes
  * and gain values.  Returns true if setup correctly.  Will not setup
  * correctly if the FilterStatus is not initialized properly.  setupFilter is
  * necessary even if the FilterStatus is a "raw copy" of the original data.
  */
bool FilterStatus::setupFilter(unsigned int filtID,
			       vector<dComplex> *poles,
			       vector<dComplex> *zeroes,
			       double gain) {
  if (mDebug > 2) cout <<mName<<": resetting filter"<<endl;
  if (mFilter) delete mFilter;
  if (poles)  mPoles  = *poles;
  if (zeroes) mZeroes = *zeroes;
  if (gain)   mGain   = gain;
 //------------  Check for raw-data flags
  if ( !filtID && !mPoles.size() && !mZeroes.size() ) {
    if (mDebug > 1) cout <<"No filter implemented -- raw data output."<<endl;
    mInit = true;
 //------------  Check whether sample rate is defined
  } else if( !mSampleRate ) {
    if (mDebug > 1) cout <<"SampleRate is not specified yet."<<endl;
    mInit = false;
 //------------  Check whether poles and zeroes are defined
  } else if ( mPoles.empty() || mZeroes.empty() ) {
    if (mDebug > 1) cout <<"Poles and/or zeroes are not yet defined."<<endl;
    mInit = false;
 //------------  Create the IIRFilter instance
  } else {
    if (mDebug) cout << "Setting up IIRFilter..." << endl;
    if (mDebug > 1) {
      cout <<"Using FilterBank["<<filtID<<"]"<<endl
	   <<"Sample rate of " << mName <<": " <<mSampleRate <<endl
	   <<"Filter Poles  ";
      for (unsigned int i=0; i<mPoles.size(); i++)
	cout <<"["<<i<<"]" <<mPoles[i].Real()<<" + "
	     <<mPoles[i].Imag()<<"i\t";
      cout <<endl<<"Filter Zeroes ";
      for (unsigned int i=0; i<mZeroes.size(); i++)
	cout <<"["<<i<<"]" <<mZeroes[i].Real()<<"+"
	     <<mZeroes[i].Imag()<<"i\t";
      cout <<endl <<"Filter Gain   " <<mGain<<endl;
    }
    mFilter = new IIRFilter(mPoles.size(), &mPoles[0], mZeroes.size(),
			    &mZeroes[0], mSampleRate, mGain);
    mTS = new TSeries();
    mInit = true;
  }
  return mInit;
}

/// setUseFilterID sets the index of a FilterBank to use as input -- in .hh
/// setSampleRate sets the sample rate of the data coming in -- in .hh file
/// getName returns the name of the filter -- in .hh file
/// getUseFilterID returns the index of FilterBank input data -- in .hh file
/// getData returns the filtered data -- in .hh file
/// isUpdated error-checks whether a filter's already been applied -- in .hh
/// isInitialized checks if the FilterStatus object has been setup -- in .hh


/***************************************************************************
 *
 * GlitchConstruct functions
 * (GlitchConstruct, ~GlitchConstruct, operator =, setName, setThreshold,
 *  setWaitTime, getName, getThreshold, getWaitTime, dumpResults, checkGlitch,
 *  isInitialized)
 *
 ***************************************************************************/

/** GlitchConstruct default constructor. Uses the default algorithm of
 * detecting glitches-- measuring rms deviation from the mean and
 * comparing vs. absolute thresholds
 */
GlitchConstruct::GlitchConstruct(const string& glitchID, double thresh,
				 double wait, unsigned int debug):
  mThreshold(thresh), mDeadTime(wait) {
  setName(glitchID);
  mDebug = debug;
  mInit  = false;
  if (mDebug > 2) cout <<"Creating GlitchConstruct -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (GlitchConstruct "<<getName()<<") "<<this<<endl;
}

/// GlitchConstruct copy constructor
GlitchConstruct::GlitchConstruct(const GlitchConstruct& glitch,
				 unsigned int debug         ) {
  if (debug > 2) cout <<"Copying GlitchConstruct -- "<<glitch.getName()<<endl;
  *this = glitch;
  mDebug = debug;
}

/// GlitchConstruct destructor
GlitchConstruct::~GlitchConstruct(void) {
  if (mDebug > 2) cout <<"Destroying GlitchConstruct -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (GlitchConstruct "<<getName()<<") "<<this<<endl;
}

/// GlitchConstruct assignment operator
GlitchConstruct& GlitchConstruct::operator =(const GlitchConstruct& rhs) {
  setName(rhs.getName());
  mThreshold    = rhs.mThreshold;
  mDeadTime     = rhs.mDeadTime;
  tmpAmp        = rhs.tmpAmp;
  tmpSig        = rhs.tmpSig;
  tmpStart      = rhs.tmpStart;
  tmpDur        = rhs.tmpDur;
  results       = rhs.results;
  mDebug        = rhs.mDebug;
  mInit         = rhs.mInit;
  if (mDebug > 2) cout <<"Assigned GlitchConstruct -- "<<getName()<<endl;
  if (mDebug > 3) cout <<"   (GlitchConstruct "<<getName()<<") "<<this<<endl;
  return *this;
}

/// GlitchConstruct "clone" method, to allocate new GlitchConstructs
GlitchConstruct *GlitchConstruct::clone( void ) const {
  if (mDebug > 2) cout <<"Cloning GlitchConstruct -- "<<getName()<<endl;
  GlitchConstruct *p = new GlitchConstruct(*this, mDebug);
  if (mDebug > 3) cout <<"   (GlitchConstruct "<<getName()<<") "<<p<<endl;
  return p;
}
 
/* GlitchConstruct modifiers
 */
/// 'setName' sets the name of the GlitchConstruct -- in .hh file
/// 'setThreshold' sets the threshold -- in .hh file
/// 'setWaitTime' sets the wait time -- in .hh file
/* GlitchConstruct accessors
 */
/// 'getName' returns the name of the GlitchConstruct -- in .hh file
/// 'getThreshols' returns the threshold value -- in .hh file
/// 'getWaitTime' returns the wait time -- in .hh file

/* GlitchConstruct virtual functions
 */
/// dumps formatted output of (one or more) glitchResult structures
void GlitchConstruct::dumpResults(ostream& out, unsigned int total) {
  if (results.size() == 0) {
    if (mDebug) cout << "No glitchResults to output." << endl;
    return;
  }
  unsigned int current(0);
  out << xsilHeader() << endl
      << xsilTableBegin("glitchResults","gds")<<endl
      << xsilComment(results.begin()->name + "-- " + results.begin()->channel,
		     1)<<endl
      << xsilTableColumn<double>("Amplitude",1) << endl
      << xsilTableColumn<double>("Significance",1) << endl
      << xsilTableColumn<int>("Start_Time",1) << endl
      << xsilTableColumn<int>("Start_Time_NS",1) << endl
      << xsilTableColumn<double>("Duration",1) << endl
      << xsilTableColumn<double>("Frequency",1) << endl
      << xsilTableColumn<double>("Bandwidth",1) << endl
      << xsilTableColumn<int>("Priority",1) << endl
      << xsilTableColumn<int>("Disposition",1) << endl
      << xsilTableColumn<const char*>("DataType",1) << endl
      << xsilTableColumn<const char*>("Glitch_State",1) << endl
      << xsilTableDataBegin("glitchResults",1) << endl;
  for (list<glitchResult>::const_iterator i=results.begin();
       i != results.end();) {
    if ( total && (current >= total) ) break;
    out << xsilTableEntry<double>(i->amplitude)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(i->significance)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>(i->startTime.getS())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>(i->startTime.getN())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(double(i->duration))
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(i->frequency)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(i->bandwidth)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>((int)i->priority)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>((int)i->disposition)
	<< xsilTableEntryDelimiter();
    switch (i->type) {
    case dTSeries:    out << xsilTableEntry<const char*>("TSeries"); break;
    case dFSeries:    out << xsilTableEntry<const char*>("FSeries"); break;
    case dFSpectrum:  out << xsilTableEntry<const char*>("FSpectrum"); break;
    case dHistogram1: out << xsilTableEntry<const char*>("Histogram1"); break;
    default:          out << xsilTableEntry<const char*>("Unknown"); break;
    }
    out << xsilTableEntryDelimiter();
    switch (i->state) {
    case gNoEvent: out << xsilTableEntry<const char*>("NoEvent"); break;
    case gInEvent: out << xsilTableEntry<const char*>("InEvent"); break;
    case gWaiting: out << xsilTableEntry<const char*>("Waiting"); break;
    case gDone:    out << xsilTableEntry<const char*>("Done"); break;
    default:       out << xsilTableEntry<const char*>("Unknown"); break;
    }
    if ( (i++) != results.end() ) out << xsilTableEntryDelimiter();
    out << endl;
    current++;
  }
  out << xsilTableDataEnd(1) << endl
      << xsilTableEnd() << endl
      << xsilTrailer() << endl;
}

/// checks for glitches
list<glitchResult>*
GlitchConstruct::checkGlitch(const TSeries& ts, const string glitchID) {
  if (ts.isEmpty()) return &results; // taking care of the obvious stuff
 /* So the procedure here is to iterate through the list, taking out all the
  * "NoEvent" glitches, continuing the "InEvent" and "Waiting" glitches, and
  * appending any new glitches that come along.
  *
  * All that is pretty standard coding -- the new part comes in how to identify
  * a glitch.  In this example we're comparing each abs(datapt) to a threshold.
  * If the datapt exceeds the threshold for the first time, we start a result
  * and immediately go to state "gInEvent".  If the datapt falls short of the
  * threshold, then we switch to "gWaiting" and start our countdown from the
  * getWaitTime() function.  If the TSeries ends before we finish counting
  * down, we'll write the current interval to the result and pick it up later.
  * After we finish counting down, we can immediately determine the duration as
  *      (Current time - Waiting interval) - Start time.
  * The other values (frequency, bandwidth, priority, disposition) aren't used
  * here, but could be used in a derived class.  The only other value used is
  * 'significance', which in our case refers to the RMS during the glitch.
  * We'll find that by commandeering 'bandwidth' to give us the sum(x), while
  * 'significance' becomes sum(x^2).  When we're in "gWaiting" mode, we'll have
  * to do some finagling, because we still want to have the RMS handy in case
  * the glitch continues before time runs out.  In that case we'll keep the
  * sum(x^2) in 'significance' and start a "quiet" sum(x^2) calculation using
  * 'frequency', while 'bandwidth' continues to plop out sum(x).  Here we
  * assume that the mean doesn't change a whole lot around a glitch.  If (and
  * when) the glitch comes back, we'll weigh the two sum(x^2) values according
  * to segment-length, then save the result in 'significance' again.
  * Hopefully we don't have to do this too many times...
  */
  //----------- Got to have the right tools for the job
  if (results.empty()) results.insert(results.end(), glitchResult());
  list<glitchResult>::iterator cur=results.begin();
  list<glitchResult>::iterator next=results.begin();
  unsigned int idx=0;		// refers to the current index in the data
  unsigned int tot=ts.getNSample();
  tmpAmp = ts.getAverage();
  // Fortunately, we get to assume double's, since that's what IIRFilter uses
  if ( !(ts.refDVect()->D_data()) ) { cerr<<"Data's not double!"<<endl; throw;}
  const double *dptr =  (const double *)ts.refDVect()->refData();
 //----------------------  Gentlemen, start your iterations!
  while (next != results.end()) {
  //---------------------  First we'll check if this glitch is done for
    if (next->state == gNoEvent) {next = results.erase(next); continue;}
  //---------------------  Next we'll check if this is our work-in-progress!
    if ((next->comment == glitchID) || (next->name == "")) {cur = next; break;}
  //---------------------  Finally, if nothing else, we roll our own.
    next++;
    if (next == results.end()) {
      cur = results.insert(next, glitchResult());
      cur->name="";		// This initializes the last glitchResult --
      break;			// it can be used for any TSeries
    }
  }
  //----------- (by now cur should be pointing at the proper glitchResult)
 //----------------------  Now let's start moving through the data
  double sofar = (double)ts.getNSample() / double(ts.getInterval());
  for (; idx < tot; idx++) {
 //----------------------  Transferring the relevant params to the new TSeries
    if (!idx && (cur->name != "") && (cur->state == gWaiting) ) {
      if (mDebug > 2) cout <<"Transferring params for the new TSeries"<<endl;
      tmpDur = int( sofar * double(cur->duration) + .5 );
      tmpStart = ts.getBinT(idx) - cur->duration;
    }
    tmpSig = abs(dptr[idx]);
    if (mDebug > 3) cout <<"Threshold="<<getThreshold() <<", abs(dptr["<<idx
			 <<"]"<<tmpSig<<endl;
 //----------------------  Oops! We gotta gotta getta glitch
    if (abs(dptr[idx]) >= getThreshold()) {
      tmpDur = int( (sofar * getWaitTime()) + .5);
     //---------  Only setup the structure if we're starting a glitch
      if (cur->name == "") {
	if (mDebug > 1)
	  cout <<"  Starting new glitch-- "<<getName()<<":"<<glitchID<<endl;
	cur->name        = getName();
	cur->comment     = glitchID;
	cur->channel     = ts.getName();
	cur->units       = "";
	cur->amplitude   = tmpSig;
	// Remember, 'significance' = sum(x^2) and 'bandwidth' = sum(x)
	cur->significance= tmpSig * tmpSig;
	cur->bandwidth   = tmpSig;

	cur->duration    = cur->frequency = 0.0; // These will be used in Wait
	cur->startTime   = ts.getBinT(idx);
	cur->priority    = trig::p_info;
	cur->disposition = trig::d_metaDB;
	cur->type        = dTSeries;
	cur->data.dTS    = &ts;
	cur->state       = gInEvent;
	continue;
      }
     //---------  Otherwise just start where we left off...
      if (abs(dptr[idx]) > cur->amplitude) cur->amplitude = abs(dptr[idx]);
      if (cur->state == gInEvent) {
	if (mDebug > 3) cout <<"  Continuing event-- "<<glitchID<<endl;
	// Here we just keep summing (x) and (x^2)
	cur->significance += tmpSig * tmpSig;
	cur->bandwidth    += tmpSig;
      } else if (cur->state == gWaiting) {
	if (mDebug > 2) cout <<"  Coming back from a wait-- "<<glitchID<<endl;
	// Now we need to put 'frequency' and 'significance' together
	cur->significance += cur->frequency + tmpSig * tmpSig;
	cur->bandwidth    += tmpSig;
	cur->frequency     = 0.0;
	cur->duration      = 0.0;
	cur->state         = gInEvent;
      }
 //----------------------  Well now we've got our pants in a bundle!
    } else {
     //-------- Nothing to see here, folks!
      if (cur->name == "") continue;

     //-------- Sayonara, baby!
      if (tmpDur <= 0) {
	if (mDebug) cout <<"Finishing glitch-- "<<glitchID<<endl;
	if (tmpStart.getS()) cur->duration = tmpStart - cur->startTime;
	else cur->duration = 
	       (ts.getBinT(idx) - Interval(getWaitTime())) - cur->startTime;
	tmpDur = int(double(cur->duration) * sofar + .5); // a little trick

	// Have to change significance from sum(x^2) to RMS
	// Note: we assume that the mean doesn't change much from the waiting
	// periods to the in-glitch periods
	cur->significance =
	  sqrt((tmpDur * cur->significance) - cur->bandwidth ) / tmpDur;
	cur->bandwidth = cur->frequency = 0.0;
	cur->data.dTS  = &ts;
	cur->state     = gDone;

	// Re-initialize the list<glitchResult> (open a new glitch)
	cur = results.insert(results.end(),glitchResult());
	cur->name="";
	continue;

     //-------- But I just got here!
      } else if (cur->state == gInEvent) {
	if (mDebug > 2) cout <<"  Beginning a wait-- "<<glitchID<<endl;
	// Initialize relevant local variables
	tmpStart       = ts.getBinT(idx);
	cur->duration  = getWaitTime();
	tmpDur =
	  int(ts.getNSample() * double(getWaitTime())/ts.getInterval() + .5);

       // Remember, 'frequency'=sum(x^2) and 'bandwidth'=sum(x) when Waiting..
	cur->frequency  = tmpSig * tmpSig;
	cur->bandwidth += tmpSig;
	cur->state      = gWaiting;
	tmpDur--;

     //-------- Just a wee-bit longer...
      } else if (cur->state == gWaiting) {
	if (mDebug > 3) cout <<"  Still waiting-- "<<glitchID
			     <<", tmpDur = "<<tmpDur<<endl;
       // Remember, 'frequency'=sum(x^2) and 'bandwidth'=sum(x) when Waiting..
	cur->frequency += tmpSig * tmpSig;
	cur->bandwidth += tmpSig;
	tmpDur--;
      }
    }
  }

  // just some debug info-- not to worry!
  if (mDebug) cout<<glitchID<<": cur->name="<<cur->name
		  <<", cur-state="<<cur->state <<endl;
  // Save settings to glitchResult (if Waiting)
  if (cur->name != "" && (cur->state == gWaiting)) {
    if (mDebug > 2) cout <<"  Stopped in a wait-- "<<glitchID<<endl;
    //    cur->duration = ts.getInterval() * tmpDur / (double)ts.getNSample();
    cur->duration = Interval(getWaitTime()) - (ts.getBinT(idx-1) - tmpStart);
  }

  // Re-initialize internal variables (one of the few places where it's done!)
  tmpStart = Time(0);
  tmpDur = int ( sofar * getWaitTime() + .5);
  tmpAmp = 0.0;
  tmpSig = 0.0;

  return &results;
}

// /// if there exists a glitch which has not yet ended, this returns false
// bool GlitchConstruct::isDone(void) {
// 
// }

/// checks whether the GlitchConstruct object has been setup correctly.
bool GlitchConstruct::isInitialized(void) const {
  string gcname("GlitchConstruct");
  if (!mInit) {
    bool allinit=true;
    if (getName().empty()) {
      if (mDebug) cout <<"GlitchConstruct doesn't have a name."<<endl;
      allinit=false; // this should never happen, but could...
    } else gcname = getName();
    if (!mThreshold) {
      if (mDebug) cout <<gcname<<" doesn't have a valid threshold"<<endl;
      allinit=false;
    }
    if (!mDeadTime) {
      if (mDebug) cout<<gcname<<" doesn't have a positive time constant"<<endl;
      allinit=false;
    }
    mInit = allinit;
  }
  return mInit;
}


/***************************************************************************
 *
 * StatisticsConstruct functions
 * (StatisticsConstruct, ~StatisticsConstruct, operator =, setName, setType,
 *  setRefresh, getName, getType, getTypeName, getRefresh, dumpResults, Reset,
 *  updateSTats, isInitialized)
 *
 ***************************************************************************/

/// StatisticsConstruct default constructor
StatisticsConstruct::StatisticsConstruct(const string& statID,
					 const updateType type, 
					 const Interval& rate,
					 const double initval,
					 const unsigned int debug) : 
  mType(type),mRefresh(rate),mInitVal(initval) {
  setName(statID);
  mDebug = debug;
  mInit  = false;
  if (mDebug > 2) cout <<"Creating StatisticsConstruct -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (StatisticsConstruct "<<getName()<<") "
		       <<this<<endl;
}

/// StatisticsConstruct copy constructor
StatisticsConstruct::StatisticsConstruct(const StatisticsConstruct& stat,
					 const unsigned int debug     ) {
  if (debug > 2) cout <<"Copying StatisticsConstruct -- "
		      <<stat.getName()<<endl;
  *this = stat;
  mDebug = debug;
}

/// StatisticsConstruct destructor
StatisticsConstruct::~StatisticsConstruct(void) {
  if (mDebug > 2) cout <<"Destroying StatisticsConstruct -- "<<getName()<<endl;
  if (mDebug > 3)
    cout <<"   (StatisticsConstruct "<<getName()<<") "<<this<<endl;
}

/// StatisticsConstruct "clone" method, to allocate new StatisticsConstructs
StatisticsConstruct *StatisticsConstruct::clone( void ) const {
  if (mDebug > 2) cout <<"Cloning StatisticsConstruct -- "<<getName()<<endl;
  StatisticsConstruct *p = new StatisticsConstruct(*this, mDebug);
  if (mDebug > 3) cout <<"   (StatisticsConstruct "<<getName()<<") "<<p<<endl;
  return p;
}

/* StatisticsConstruct modifiers
 */
/// 'setName' sets the name of the statistics construct -- in .hh file
/// 'setType' sets the type of the construct -- in .hh file
/// 'setRefresh' sets the refresh rate for the construct (in seconds) -- in .hh
/* StatisticsConstruct accessors
 */
/// 'getName' returns the name of the statistics construct -- in .hh file
/// 'getType' returns the type of the construct -- in .hh file
/// 'getTypeName' returns the type name of the construct -- in .hh file
/// 'getRefresh' returns the refresh rate for the statistic -- in .hh file

/* StatisticsConstruct virtual functions
 */
/// dumps formatted output of (one or more) statResult structures
void StatisticsConstruct::dumpResults(ostream& out, unsigned int total) {
  if (results.size() == 0) {
    if (mDebug) cout << "No statResults to output." << endl;
    return;
  }
  unsigned int current(0);
  /// Setting up some of the non-trivial columns
  string comment = results.begin()->name + string("-- ") +
    results.begin()->channel;
  xsilTableColumn<double> working("Working_Value",1);
  working.SetUnit(results.begin()->units.c_str());
  xsilTableColumn<double> final("Final_Value",1);
  final.SetUnit(results.begin()->units.c_str());
  if (mDebug > 2)
    cout <<"   " <<getName()<<" set up non-trivial table columns." <<endl;

  out << xsilHeader() << endl
      << xsilTableBegin("statResults","gds")<<endl
      << xsilComment(comment.c_str(), 1)<<endl
      << xsilTableColumn<const char*>("Channel",1)<<endl
      << xsilTableColumn<const char*>("Stat_ID",1)<<endl
      << xsilTableColumn<const char*>("Result_Name",1)<<endl
      << working<<endl << final<<endl
      << xsilTableColumn<long long>("Stat_Count",1) << endl
      << xsilTableColumn<int>("Start_Time",1) << endl
      << xsilTableColumn<int>("Start_Time_NS",1) << endl
      << xsilTableColumn<double>("Duration",1) << endl
      << xsilTableColumn<const char*>("DataType",1) << endl
      << xsilTableColumn<const char*>("Stat_State",1) << endl
      << xsilTableDataBegin("statResults",1) << endl;
  for (list<statResult>::const_iterator i=results.begin(); i!=results.end();){
    if ( total && (current >= total) ) break;
    out << xsilTableEntry<const char*>(i->channel.c_str())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<const char*>(i->name.c_str())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<const char*>(i->comment.c_str())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(i->prelim)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(i->actual)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<long long>(i->statcount)
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>(i->startTime.getS())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<int>(i->startTime.getN())
	<< xsilTableEntryDelimiter()
	<< xsilTableEntry<double>(double(i->duration))
	<< xsilTableEntryDelimiter();
    switch (i->type) {
    case dTSeries:    out << xsilTableEntry<const char*>("TSeries"); break;
    case dFSeries:    out << xsilTableEntry<const char*>("FSeries"); break;
    case dFSpectrum:  out << xsilTableEntry<const char*>("FSpectrum"); break;
    case dHistogram1: out << xsilTableEntry<const char*>("Histogram1"); break;
    default:          out << xsilTableEntry<const char*>("Unknown"); break;
    }
    out << xsilTableEntryDelimiter();
    switch (i->state) {
    case sNewStat:  out << xsilTableEntry<const char*>("NewStat"); break;
    case sInStat:   out << xsilTableEntry<const char*>("InStat"); break;
    case sDoneStat: out << xsilTableEntry<const char*>("DoneStat"); break;
    default:        out << xsilTableEntry<const char*>("Unknown"); break;
    }
    if ( (i++) != results.end() ) out << xsilTableEntryDelimiter();
    out << endl;
    current++;
  }
  out << xsilTableDataEnd(1) << endl
      << xsilTableEnd() << endl
      << xsilTrailer() << endl;
}

/// resets the statistics -- continuous stats must be overridden to be reset
void StatisticsConstruct::Reset(const bool override) {
  if (mDebug && override) cout <<"Overriding cumulative stats."<<endl;
  if ((mType == uRefreshed) || override) {
    tmp1 = 0.0;
    tmp2 = 0.0;
    results.clear();
  }
}

/// updates the statistics for (possibly filtered) data
list<statResult>*
StatisticsConstruct::updateStats(const TSeries& ts, const string statID) {
  if (ts.isEmpty()) return &results; // taking care of the obvious stuff
 /* So the procedure here is to calculate the rms of every data stream.  Pretty
  * easy, and it should be easier with constructs like these.
  *
  * The trick is in storing the proper data values.  Since RMS can be construed
  * as <x^2> - <x>^2, we can save the sum(x) in prelim and the sum(x^2)
  * in actual, then we can divided by N, located in statcount, when we want the
  * RMS.  This should minimize the roundoff errors.
  */
  // We should probably check if now is a good time to reset the statistics
  if (isDue(ts.getStartTime()) && mDebug)
    cout <<"Statistics-- "<<getName()<<" has been refreshed."<<endl;
  //----------- Got to have the right tools for the job
  if (results.empty()) results.insert(results.end(), statResult());
  list<statResult>::iterator cur=results.begin();
  list<statResult>::iterator next=results.begin();
  // Fortunately, we get to assume double's, since that's what IIRFilter uses
  if ( !(ts.refDVect()->D_data()) ) { cerr<<"Data's not double!"<<endl; throw;}
 //----------------------  Gentlemen, start your iterations! Find that, stat!
  while (next != results.end()) {
   //---------------------  Next we'll check if this is our work-in-progress!
    if ((next->comment == statID) || (next->name == "")) {cur = next; break;}
  //---------------------  Finally, if nothing else, we roll our own.
    next++;
    if (next == results.end()) {
      cur = results.insert(next, statResult());
      cur->name="";		// This initializes the last statResult --
      break;			// it can be used for any TSeries
    }
  }
  //----------- (by now cur should be pointing at the proper statResult)
  // Setting some helpful local variables
  tmp1 = ts.getAverage() * ts.getNSample();
  tmp2 = ts * ts;
  if (mDebug > 2) cout <<"  Statistic-- "<<getName()<<": sum(x)="<<tmp1
		       <<", sum(x^2)="<<tmp2 <<endl;

  //-----------  Let's (re)create our new statResult if we don't have one yet.
  if (cur->name == "" || cur->state == sNewStat) {
    if (mDebug > 1)
      cout <<"  Starting new statistic-- "<<getName()<<":"<<statID<<endl;
    cur->name     = getName();
    cur->comment  = statID;
    cur->channel  = ts.getName();
    cur->units    = "";
    cur->statcount= ts.getNSample();
    cur->prelim   = tmp1;
    cur->actual   = tmp2;
    cur->startTime= ts.getStartTime();
    cur->duration = ts.getInterval();
    cur->type     = dTSeries;
    cur->data.dTS = &ts;
    cur->state    = sInStat;
  //-----------  Otherwise just start where we left off
  } else {
    cur->statcount += ts.getNSample();
    cur->actual    += tmp2;
    cur->prelim    += tmp1;
    cur->data.dTS   = &ts;
    if ((ts.getEndTime() - cur->startTime) > getRefresh()) {
      if (mDebug > 2)
	cout <<"  Finishing statistic-- "<<getName()<<":"<<statID<<endl;
      cur->state    = sDoneStat;
    } else cur->state = sInStat;
  }

  // Just some debug info... Not to worry!
  if (mDebug)
    cout<<statID<<": cur->name="<<cur->name<<", cur-state="<<cur->state<<endl;

  // Re-initialize internal variables (one of the few places where it's done!)
  tmp1 = tmp2 = 0.0;
  return &results;
}

/// checks whether the StatisticsConstruct object has been setup correctly
bool StatisticsConstruct::isInitialized(void) const {
  string scname("StatisticsConstruct");
  if (!mInit) {
    bool allinit=true;
    if (getName().empty()) {
      if (mDebug) cout <<"StatisticsConstruct doesn't have a name."<<endl;
      allinit=false; // this should never happen, but could...
    } else scname = getName();
    if (mType == uUnknown) {
      if (mDebug) cout <<scname<<" has unknown update type"<<endl;
      allinit=false;
    }
    if ((mType == uRefreshed) && mRefresh <= Interval(0.0)) {
      if (mDebug) cout <<scname<<" doesn't have a positive refresh rate"<<endl;
      allinit=false;
    }
    mInit = allinit;
  }
  return mInit;
}

/* StatisticsConstruct misc functions
 */
/** will check a refreshable statisic and reset it if the refresh interval time
  * has been exceeded since the last isDue() call
  */
bool StatisticsConstruct::isDue( const Time& compare) {
  if ((mType == uRefreshed) && (compare > (mRefreshTime + mRefresh)) ) {
    Reset();
    mRefreshTime = compare;
    return true;
  } else return false;
}

// /// returns false if any statistics still need more data to update
// bool StatisticsConstruct::isDone(void) {
// 
// }


/***************************************************************************
 *
 * ResultConstruct functions
 * (ResultConstruct, ~ResultConstruct, operator =, setName, setType, setOutput,
 *  setUpdate, getName, getType, getTypeName, getOutput, getUpdate,
 *  updateResult, Reset, isInitialized)
 *
 ***************************************************************************/

/// ResultConstruct default constructor
ResultConstruct::ResultConstruct(const string& resultID,
				 const updateType type,
				 const unsigned int outmask,
				 ostream& out,
				 const Interval& update,
				 const unsigned int debug) :
  mType(type), mUpdate(update), mOutput(&out), mOutmask(outmask) {
  setName(resultID);
  mDebug     = debug;
  mInit      = false;
  mEntryTime = Now();
  if (mDebug > 2) cout <<"Creating ResultConstruct -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (ResultConstruct "<<getName()<<") "<<this<<endl;
}

/// ResultConstruct copy constructor
ResultConstruct::ResultConstruct(const ResultConstruct& result,
				 const unsigned int debug) {
  if (debug > 2) cout <<"Copying ResultConstruct -- "<<result.getName()<<endl;
  *this = result;
  mEntryTime = Now();
  mDebug = debug;
}

/// ResultConstruct destructor
ResultConstruct::~ResultConstruct(void) {
  if (mDebug > 2) cout <<"Destroying ResultConstruct -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (ResultConstruct "<<getName()<<") "<<this<<endl;
}

/// ResultConstruct assignment operator
ResultConstruct& ResultConstruct::operator =(const ResultConstruct& rhs) {
  setName(rhs.getName());
  mType         = rhs.mType;
  mUpdate       = rhs.mUpdate;
  mEntryTime    = rhs.mEntryTime;
  mOutput       = rhs.mOutput;
  mOutmask      = rhs.mOutmask;
  mHistMap      = rhs.mHistMap;
  mDebug        = rhs.mDebug;
  mInit         = rhs.mInit;
  if (mDebug > 2) cout <<"Assigned ResultConstruct -- "<<getName()<<endl;
  if (mDebug > 3) cout <<"   (ResultConstruct "<<getName()<<") "<<this<<endl;
  return *this;
}

/// ResultConstruct "clone" method, to allocate new ResultConstructs
ResultConstruct *ResultConstruct::clone( void ) const {
  if (mDebug > 2) cout <<"Cloning ResultConstruct -- "<<getName()<<endl;
  ResultConstruct *p = new ResultConstruct(*this, mDebug);
  if (mDebug > 3) cout <<"   (ResultConstruct "<<getName()<<") "<<p<<endl;
  return p;
}
 
/* ResultConstruct modifiers
 */
/// 'setName' sets the name of the result -- in .hh file
/// 'setType' sets the type of the result -- in .hh file
/// 'setOutput' sets the I/O stream to use -- in .hh file
/// 'setOutmask' sets the bitwise output mask -- in .hh file
/// 'setUpdate' sets the update interval -- in .hh file
/* ResultConstruct accessors
 */
/// 'getName' returns the name of the result construct -- in .hh file
/// 'getType' returns the type of the result construct -- in .hh file
/// 'getTypeName' returns the type name of the result construct -- in .hh file
/// 'getOutput' returns the I/O stream used by the result -- in .hh file
/// 'getOutmask' returns the bitwise output mask -- in .hh file
/// 'getUpdate' returns the update interval for the result -- in .hh file

/* ResultConstruct virtual functions
 */    
/** updates a result entry -- will not output result if override is not set
  * and glitch has not finished
  */
bool
ResultConstruct::output(const glitchResult& result,
			const unsigned int outmask,
			const bool override, ostream& out) {
  bool outc  = (&out == &cout);
  bool moutc = (mOutput == &cout);
  unsigned int thismask = outmask;
  if (!thismask) thismask = mOutmask;
  string histname = result.name + "_" + result.comment;
  bool success=true;
 /* The object here is to produce formatted output of each glitch event when
  * it is done, then tag the result as "No Event".  The output right now is
  * only to ostream.  To produce triggers or serve data, we'll need to inherit
  * from TrigClient and/or MonServer.
  *
  * The 'outmask' and 'override' are used as follows: 'outmask', when set,
  * defines where the output will go.  output will return false if
  * 'outmask' directs output to the internally-defined output when there is
  * no internally-defined output. If 'outmask' is not set, then result
  * will be processed internally, and return true. 'override', when true,
  * will cause the result to be tabulated, even if the glitch hasn't finished.
  *
  * In addition, a special bitmask setting (64) will cause output of all
  * stored histograms of events to 'out', since last refreshed.
  *
  * This example only histograms the amplitudes of the glitches.  Derived
  * classes could store many of the glitch result values.
  */
 //----------------------  First see if we need to create a Histogram1 entry.
  // If so, we'll use mUpdate interval (or 3600) as the x-axis, and amplitude
  // as the y-axis.  If refreshable, the histogram will be cleared every
  // mUpdate interval.  If cumulative, the glitch results will be entered
  // modulo(3600) since the beginning of the monitor.
  if (mHistMap.find(histname) == mHistMap.end()) {
    unsigned int xaxis = (mType == uRefreshed) ? (unsigned int)mUpdate : 3600;
    string ylabel = "collective glitch amplitude ";
    ylabel += (mType == uRefreshed) ? "(over a single interval) " :
      "(cumulative) ";
    ylabel += result.units;
    ostringstream xlabel("Time (modulo ", ios::ate);
    xlabel <<xaxis<<")";
    mHistMap[histname] =
      Histogram1(histname.c_str(), xaxis,0.0,double(xaxis),
		 xlabel.str().c_str(), ylabel.c_str() );
  }
 //----------------------  If not, we'll just work with what we have
  if (mDebug > 1) cout <<histname<<": state= "<<result.state
		       <<", override="<<override<<endl;
  if ((result.state == gDone) || override) {
    //---------  regardless of output, make an entry in the histogram
    int xmax = mHistMap[histname].GetNBins();
    double x = double(result.startTime - getEntryTime());
    mHistMap[histname].Fill( result.amplitude, (int(x) % xmax) );
    //---------  Here's the formatted output (sent to any ostream)
    ostringstream output("", ios::ate);
    char tstr[40] = "";
    TimeStr(result.startTime, tstr, " %y.%m.%d %H:%N:%S ");
    output <<result.channel <<tstr << setw(17)<<setprecision(16)<<setfill('0')
	   <<double(result.startTime.totalS())<<" "
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.duration
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.amplitude
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.significance
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.frequency
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.bandwidth
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.priority
	   <<setw(9)<<setprecision(6)<<setfill(' ')<<result.disposition;
    switch (result.type) {
    case dTSeries:    output <<" TSeries    "; break;
    case dFSeries:    output <<" FSeries    "; break;
    case dFSpectrum:  output <<" FSpectrum  "; break;
    case dHistogram1: output <<" Histogram1 "; break;
    case dUnknown:
    default:          output <<" Unknown    "; break;
    }
    if (mDebug) output <<result.data.dTS;
    if (override)
      switch (result.state) {
      case gNoEvent: output <<" NoEvent "; break;
      case gInEvent: output <<" InEvent "; break;
      case gWaiting: output <<" Waiting "; break;
      case gDone:    output <<" Done    "; break;
      default: break;
      }
    output << histname << endl;
    //---------  standard output check
    if (thismask & 1) {
      cout << output.str();
    }
    //---------  default output stream check
    if (thismask & 2) {
      if (mOutput && (!(thismask & 1) || !moutc) )
	*mOutput << output.str();
      else success = false;
    }
    //---------  given output stream check
    if (thismask & 4) {
      if (!(thismask & 1) || !outc) out << output.str();
      else success = false;
    }
    if (result.state == gDone) result.state = gNoEvent;
  }
  //---------  dump all output check
  if (thismask & 64) {
    out << xsilHeader() << endl;
    for (map<string,Histogram1>::iterator i = mHistMap.begin();
	 i != mHistMap.end(); i++ ) out << xsilHistogram(i->second) << endl;
    out << xsilTrailer() << endl;
  }
  return success;
}

/** updates a result entry -- will not output result if override is not set
  * and the statistics calculation has not finished
  */
bool
ResultConstruct::output(const statResult& result, const unsigned int outmask,
			const bool override, ostream& out) {
  bool outc  = (&out == &cout);
  bool moutc = (mOutput == &cout);
  unsigned int thismask = outmask;
  if (!thismask) thismask = mOutmask;
  string histname = result.name + "_" + result.comment;
  bool success=true;
 /* The object here is to produce formatted output of each statistic
  * it is done, then tag the result as "NewStat".  The output right now is
  * only to ostream.  To produce triggers or serve data, we'll need to inherit
  * from TrigClient and/or MonServer.
  *
  * The 'thismask' and 'override' are used as follows: 'thismask', when set,
  * defines where the output will go.  output will return false if
  * 'thismask' directs output to the internally-defined output when there is
  * no internally-defined output. If 'thismask' is not set, then result
  * will be processed internally, and return true. 'override', when true,
  * will cause the result to be tabulated, even if the glitch hasn't finished.
  *
  * In addition, a special bitmask setting (64) will cause output of all
  * stored histograms of events to 'out', since last refreshed.
  *
  * This example only histograms the final values of the statistics.  Derived
  * classes could store many of the statistic result values.  In this case, the
  * result will either create a histogram entry for the statResult or log it,
  * but not both.  When finished, it will set the status to "sNewStat".
  */
 //----------------------  First see if we need to create a Histogram1 entry.
  // If so, we'll use the range [prelim,actual] as the x-axis, and count the
  // number of calculated statistics on y-axis.  We'll either take the
  // statcount as the number of bins, or, if statcount is zero, we'll use 20.
  // If refreshable, the histogram will be cleared every mUpdate interval.  If
  // cumulative, the statistics will be entered ad infinitum, until the
  // histogram is manually reset (with the manual override parameter).
  if (mHistMap.find(histname) == mHistMap.end()) {
    unsigned int xbins = (result.statcount) ? result.statcount : 20;
    double xlow =(result.prelim < result.actual) ? 
      result.prelim : result.actual;
    double xhigh=(result.prelim < result.actual) ?
      result.actual : result.prelim;
    if ( xlow == xhigh ) {
      xlow--; // this just helps make some space
      success = false; // this will help others know something's fishy
    }
    
    ostringstream title("", ios::ate);
    title << histname;
    if (mType == uRefreshed) 
      title <<" -- Refreshed every "<<mUpdate<<" seconds";
    string xlabel(result.name);
    xlabel += (result.units.empty()) ? "(arb. unitst)" : result.units;
    string ylabel("# of statistics calculated");
    mHistMap[histname] = Histogram1(title.str().c_str(), xbins,xlow,xhigh,
				    xlabel.c_str(), ylabel.c_str() );
    result.state = sNewStat;
 //----------------------  If not, we'll just work with what we have
  } else if ((result.state == sDoneStat) || override) {
    if (mDebug > 1) cout <<histname<<": state= "<<result.state
			 <<", override="<<override<<endl;
    //---------  regardless of output, make an entry in the histogram
    mHistMap[histname].Fill( result.actual );
    //---------  Here's the formatted output (sent to any ostream)
    ostringstream output("", ios::ate);
    char tstr[40] = "";
    TimeStr(result.startTime, tstr, " %y.%m.%d %H:%N:%S ");
    // We're making some mighty assumptions here;
    // see StatisticsConstruct:updateStats for details
    double sAvg  = result.prelim / double(result.statcount);
    double sDev = sqrt(result.actual / double(result.statcount) - sAvg * sAvg);
    output <<result.channel <<tstr << setw(17)<<setprecision(16)<<setfill('0')
	   <<double(result.startTime.totalS())<<" "
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.duration
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.actual
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.statcount
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.prelim
	   <<"'Avg'  = "<<setw(12)<<setprecision(5)<<setfill(' ')<<sAvg
	   <<"'SDev' = "<<setw(12)<<setprecision(5)<<setfill(' ')<<sDev;
    switch (result.type) {
    case dTSeries:    output <<" TSeries    "; break;
    case dFSeries:    output <<" FSeries    "; break;
    case dFSpectrum:  output <<" FSpectrum  "; break;
    case dHistogram1: output <<" Histogram1 "; break;
    case dUnknown:
    default:          output <<" Unknown    "; break;
    }
    if (mDebug) output <<result.data.dTS;
    if (override)
      switch (result.state) {
      case sNewStat:  output <<" NewStat  "; break;
      case sInStat:   output <<" InStat   "; break;
      case sDoneStat: output <<" DoneStat "; break;
      default: break;
      }
    output << histname << endl;
    //---------  standard output check
    if (thismask & 1) {
      cout << output.str();
    }
    //---------  default output stream check
    if (thismask & 2) {
      if (mOutput && (!(thismask & 1) || !moutc) )
	*mOutput << output.str();
      else success = false;
    }
    //---------  given output stream check
    if (thismask & 4) {
      if (!(thismask & 1) || !outc) out << output.str();
      else success = false;
    }
    if (result.state == sDoneStat) result.state = sNewStat;
  }
  //---------  dump all output check
  if (thismask & 64) {
    out << xsilHeader() << endl;
    for (map<string,Histogram1>::iterator i = mHistMap.begin();
	 i != mHistMap.end(); i++ ) out << xsilHistogram(i->second) << endl;
    out << xsilTrailer() << endl;
  }
  return success;
}

/// resets the result -- continuous results must be overridden to be reset
void ResultConstruct::Reset(const bool override) {
  if (mDebug && override) cout <<"Overriding cumulative results."<<endl;
  if ((mType == uRefreshed) || override) {
    mHistMap.clear();
  }
}

/// checks whether the ResultConstruct object has been setup correctly
bool ResultConstruct::isInitialized(void) const {
  string rcname("ResultConstruct");
  if (!mInit) {
    bool allinit=true;
    if (getName().empty()) {
      if (mDebug) cout <<"ResultConstruct doesn't have a name."<<endl;
      allinit=false; // this should never happen, but could...
    } else rcname = getName();
    if (mType == uUnknown) {
      if (mDebug) cout <<rcname<<" has unknown update type"<<endl;
      allinit=false;
    }
    if ((mType == uRefreshed) && mUpdate <= Interval(0.0)) {
      if (mDebug) cout <<rcname<<" doesn't have a positive refresh rate"<<endl;
      allinit=false;
    }
    if (mOutput == NULL) {
      if (mDebug) cout <<rcname<<" has no valid output pointer"<<endl;
      //      allinit=false;
    }
    mInit = allinit;
  }
  return mInit;
}

/* ResultConstruct misc functions
 */
/** will check a refreshable result and reset it if the refresh interval time
  * has been exceeded since the last isDue() call
  */
bool ResultConstruct::isDue(const Time& compare, const bool override) {
  if (override || 
      ( (mType == uRefreshed) && (compare >(mEntryTime + mUpdate)) ) ) {
    Reset();
    mEntryTime = compare;
    return true;
  } else return false;
}

} // end of namespace channel {

