#ifndef KLEINEWELLE_HH
#define KLEINEWELLE_HH

#include "MultiStream.hh"
#include "FilterDesign.hh"
#include "Pipe.hh"
#include "TSeries.hh"
#include "LPEFilter.hh"
#include <fstream>
#include <string>
#include <iostream>
#include <cstdio>
#include <vector>

class kleineWelle : public MultiStream {

public:

  kleineWelle(int argc, const char *argv[]);
  ~kleineWelle(void);

  void ProcessData(void);

  struct Event {
    Event(double startTime, double endTime, double centralTime, double frequency, double energy,
	  double normEnergy, int npts, double significance)
      : startTime(startTime), endTime(endTime), centralTime(centralTime), frequency(frequency),
	energy(energy), normEnergy(normEnergy), npts(npts), significance(significance) {}
    double startTime;
    double endTime;
    double centralTime;
    double frequency;
    double energy;
    double normEnergy;
    int npts;
    double significance;
  };

  struct Element {
    Element(double energy, double normEnergy, int index, int scale, double startTime,
	    double endTime, double dt, int clusterID)
      : energy(energy), normEnergy(normEnergy), index(index), scale(scale), startTime(startTime),
	endTime(endTime), dt(dt), clusterID(clusterID) {}
    double energy;
    double normEnergy;
    int index;
    int scale;
    double startTime;
    double endTime;
    double dt;
    int clusterID;
  };


  int dist(const Element& a, const Element& b) {
    if(a.scale <= b.scale) {
      return b.scale - a.scale + (int)(abs((a.index >> (b.scale - a.scale)) - b.index));
    }
    else {
      return a.scale - b.scale + (int)(abs((b.index >> (a.scale - b.scale)) - a.index));
    }
  }

  void mergeClust(std::vector<Element>& elements, std::vector<Element>::iterator i, int maxDist);

  // negative natural logarithm of significance
  double logSignificance(double normalizedEnergy, double timeFrequencyArea,
			 double precision);

private:

  std::string channel;       // channel name of data (noise)
  std::string injections;    // channel name of injections to add
  std::string filter;        // prefilter option
  std::ifstream segmentFile; // segment list
  std::ifstream optionsFile;
  std::ifstream waveFile;    // ASCII file with software injection to complete
  float*        waveInjection;    // array containing the values for the loaded injection
  unsigned long segmentStart;     // top bound of a segment to be analyzed
  unsigned long segmentStop;     // bottom bound
  Interval stride;     // number of seconds to analyze at a time
  double   significance;  // significance threshold for reporting clusters
  double   threshold;     // gaussian threshold for black pixels
  int      fs;         // sample rate (after any decimations)
  int      size;      // size of the wavelet decomposition array
  int      lpefLength;
  int      maxDist;    // max distance between two black pixels to be considered
  int      scale;      // log(size), base 2
  float    lowFrequency;  // low frequency boundary
  float    highFrequency; // high frequency boundary
  int      lowScale; //  obsolted but take precedence if defined
  int      highScale;
  float    factor;
  int      decimateFactor;    // number of decimations to do
  Interval transientDuration; // amount of data to skip at boundaries for transients
  Wavelet*     wavePtr;       // pointer to wavelet functions library
  FilterDesign fd;            // prefilter design class
  Pipe*        prefilter;     // decimator and highpass prefilter
  LPEFilter*   lpefilter;     // linear predictive whitening filter
  Time         nextTime;      // bookeeping to check for data continuity
  TSeries*     history;       // history kept for data handling in case of discontinuous data
};

bool operator<(const kleineWelle::Event& a, const kleineWelle::Event& b) {
  return a.startTime < b.startTime;
}

bool operator<(const kleineWelle::Element& a, const kleineWelle::Element& b) {
  return a.startTime < b.startTime;
}

std::ostream& operator<<(std::ostream& output, const kleineWelle::Event& a) {
  char line[128];
  sprintf(line, "%16.6f  %16.6f  %16.6f  %4.0f  %9.3e  %6.2f  %3d  %6.2f", a.startTime, a.endTime,
	  a.centralTime, a.frequency, a.energy, a.normEnergy, a.npts, a.significance);
  output << line << endl;
  return output;
}

#endif // KLEINEWELLE_HH 
