
#include <iostream>
#include <sstream>
#include <iomanip>
#include <string>
#include <vector>
#include <utility>

#include "Time.hh"
#include "xml/XsilTSeries.hh"
#include "xml/Xsil.hh"
#include "html/writer.hh"
#include "html/align.hh"  
#include "html/document.hh" 
#include "html/font.hh" 
#include "html/hline.hh" 
#include "html/table.hh"
#include "html/text.hh"
#include "html/color.hh" 

#include "glitchMon.hh"

using namespace std;
using namespace xml;
typedef std::pair<string, string> ConfPair;
typedef std::vector< ConfPair > ConfElements;

//===========================================================
bool TagCompare ( const std::string& key, const char* tag ) {
	if ( strcasecmp ( key.c_str(), tag ) != 0 ) {
		return false;
	}
	return true;
}

//===========================================================
bool GetBool (const std::string& s ) {
	bool ret = false;
	if ( TagCompare ( s , "ON")  ) {
		ret = true;
	}

	return ret;
}

//===========================================================
bool GetElements (const std::string& value_org, ConfElements& elements ) {
	std::string value = value_org + " ";
	// split string into elements
	std::string::size_type pos = 0, prev_pos = 0;
	std::vector<std::string> contents;
	std::string word;
	while( ( pos = value.find_first_of(' ', pos)) != string::npos ) {
		if ( value[prev_pos] == '{' ) {
			pos = value.find_first_of ('}', prev_pos);
			if ( pos == string::npos ) 
				return false;
			word = value.substr( prev_pos, pos - prev_pos + 1);
		}
		else if ( value[prev_pos] == '"' ) {
			std::string::size_type qpos = ++prev_pos;
			while ( ( pos = value.find_first_of ('"', qpos) ) != string::npos &&
					value[ pos - 1 ] == '\\' ) {
				qpos = ++pos;
			}
			if ( pos == std::string::npos ) {
				cerr << "\" mismatch. " << value_org << endl;
				return false;
			}
			word = value.substr( prev_pos, pos - prev_pos );
		}
		else word = value.substr( prev_pos, pos - prev_pos);
		if ( word.size() ) contents.push_back(word);
		prev_pos = ++pos;
	}

	// create std::map of tag and value
	std::vector<string>::iterator itr = contents.begin();
	while ( itr != contents.end() ) {
		std::string key = *itr;
		std::string val;
		++itr;
		if ( itr != contents.end() ) {
			val = *itr;
		}
		else 
			return false;

		// elements[key] = val;
		elements.push_back( ConfPair(key, val) );
		++itr;
	}

	return true;
}

//===========================================================
bool parse ( ifstream& file, std::string& tag, std::string& value, 
		std::string& remainder ) {
	bool done = false;
	int depth = 0;
	bool find_beg = false;
	bool find_end = false;

	std::string line = remainder;
	std::string newline;
	while ( !file.eof() && !done ) {
		// read one line from config file 
		getline( file, newline );
		// remove extra spaces
		while (!newline.empty() && isspace (newline[0])) {
			newline.erase (0, 1);
		}
		// process comment line
		if ( newline.empty() ) 
			continue;
		if ( newline[0] == '#' ) 
			continue;

		int newlen = newline.length()-1;
		if (!newline.empty() && newline[newlen] == '\n') 
			newline.erase(newlen,1);

		line += newline;
		while (!line.empty() && isspace (line[0])) {
			line.erase (0, 1);
		}
		line += " ";

		std::string::size_type pos = 0;
		while ( (pos = newline.find_first_of ("{}" , pos )) != std::string::npos && 
				!(find_beg && find_end) ) {
			if ( newline[pos] == '{'  ) {
				if ( !find_beg ) find_beg = true;
				++depth;
			}
			else if ( newline[pos] == '}' ) {
				--depth;
				if ( !find_end && depth == 0 ) find_end = true;
			}
			if ( depth < 0 ) {
				cerr << "{} mismatch error." << endl;
				return false;
			}

			++pos;
		}

		if ( find_beg && find_end ) done = true;
	}
	if ( done ) {
		std::string::size_type beg = line.find_first_of ('{');
		std::string::size_type end = line.find_last_of ('}');
		tag = line.substr ( 0, beg );
		while ( !tag.empty() && isspace (tag[tag.size() - 1]) ) {
			tag.erase(tag.size() - 1, 1);
		}
		value = line.substr ( beg + 1, end - beg - 1);
		remainder = line.substr(end + 1);
		return true;
	}
	else {
		return false;
	}
}

//===========================================================
//
//  glitchMon
//
//===========================================================
EXECDAT(glitchMon)

	// constructor
	glitchMon::glitchMon(int argc, const char *argv[]) :  
DatEnv(argc,argv), fFirstEvent (true)      
{
	fFList[ "Default" ] = FilterInfo();
	fTList[ "Default" ] = ThresholdInfo();
	fCPList[ "Default" ] = Parameter();
	fHList[ "Default" ] = HistogramInfo();

	const char* filename = 0;
	if (argc > 1) {
		filename = argv[1];
	}
	else {
		cerr << "Error : No Filename." << endl;
		finish ();
		return;
	}

	//open config file
	ifstream configfile (filename, ios::in);
	if (!configfile.is_open()){
		cerr << "Configuration File Not Found." << endl;
		finish ();
		return;
	}
	else {
		//read configuration file
		if ( !SetConfig ( configfile ) ) {
			finish();
			return;
		}
	}
	configfile.close();

	RegisterChannels();

	fMonServ.setServerName ( fMonConf.fName.c_str() );

	if ( !Reset() ) {
		finish();
		return;
	}

	ConfigLog ();
}

//===========================================================
// destructor
//===========================================================
glitchMon::~glitchMon(void)
{

	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->GetOnGoingEvent ( fEList );
	}
	OutputEvents ();

	MakeHtmlLog ();
	CloseFiles ();

	ofstream tsout ( kTSBackUpName, ios::out | ios::trunc );
	if ( tsout.is_open() ) {
		tsout << xsilHeader() << endl;
		for ( ChannelList::iterator itr = fChList.begin();
				itr != fChList.end(); ++itr ) {
			itr->TSSave ( tsout );
		}
		tsout << xsilTrailer() << endl;
	}
}


//===========================================================
void glitchMon::Attention() {
	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->Attention ();
	}

	fMonServ.Attention();
}


//===========================================================
// main loop
//===========================================================
void glitchMon::ProcessData(void)
{
	fLock.Check ( getDacc() );
	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->ProcessChannel ( getDacc(), fEList, fLock );
	}
	OutputEvents ();
	ProcessLog ();
	if ( fMonConf.fDuration.GetS() && 
			fMonConf.fDuration < getDacc().getCurrentTime() - fMonStartTime ) {
		finish ();
	}
}

//===========================================================
bool glitchMon::OpenFiles () {
	int t = Now().getS();

	if ( fMonConf.fDataPath.size() ) {
		ostringstream str;
		str << t;
		std::string data = fMonConf.fDataPath;
		data += fMonConf.fName + "_";
		data += str.str();
		data += ".xml";
		fDataFile.clear();
		fDataFile.open ( data.c_str(), ios::out|ios::ate );
		if ( !fDataFile.is_open() ) {
			cerr << "Error opening Data File." << endl;
			return false;
		}
		fDataFile << xsilHeader() << endl;
		fDataFile << xsilTableBegin (fMonConf.fName.c_str(), "glitchMon") << endl;
		fDataFile << xsilComment ("glitchMon events") << endl;
		fDataFile << "    <Column Name=\"Name\" Type=\"string\"/>" << endl;
		fDataFile << "    <Column Name=\"Ifo\" Type=\"string\"/>" << endl;
		fDataFile << "    <Column Name=\"Start_Time\" Type=\"int\"/>" << endl;
		fDataFile << "    <Column Name=\"Start_Time_NS\" Type=\"int\"/>" << endl;
		fDataFile << "    <Column Name=\"Duration\" Type=\"double\"/>" << endl;
		fDataFile << "    <Column Name=\"Peak\" Type=\"double\"/>" << endl;
		fDataFile << "    <Column Name=\"NSigma\" Type=\"double\"/>" << endl;
		fDataFile << "    <Column Name=\"Sigma\" Type=\"double\"/>" << endl;
		fDataFile << "    <Column Name=\"Mean\" Type=\"double\"/>" << endl;
		fDataFile << "    <Column Name=\"NPoints\" Type=\"int\"/>" << endl;
		fDataFile << "    <Column Name=\"Comment\" Type=\"string\"/>" << endl;
		fDataFile << xsilTableDataBegin ("Events") << endl;

		fFirstEvent = true;
	}

	if ( fMonConf.fDoLog ) {
		const char* htmldir = getenv("DMTHTMLOUT");
		std::string log;
		if (htmldir) log = string ( htmldir ) + "/";
		ostringstream str;
		str << t;
		log += fMonConf.fName + "_";
		log += str.str();
		log += "_log.html";
		fLogFile.clear();
		fLogFile.open ( log.c_str(), ios::out );
		if ( !fLogFile.is_open() ) {
			cerr << "Error opening Log File." << endl;
			return false;
		}
	}

	return true;
}

//===========================================================
void glitchMon::CloseFiles () {
	if ( fDataFile.is_open() ) {
		fDataFile << endl;
		fDataFile << xsilTableDataEnd () << endl;
		fDataFile << xsilTableEnd () << endl;
		fDataFile << xsilTrailer() << endl;
		fDataFile.close();
	}
	if ( fLogFile.is_open() ) {
		fLogFile.close();
	}
}

//===========================================================
void glitchMon::UpdateFiles () {
	CloseFiles ();
	OpenFiles ();
}

//===========================================================
void glitchMon::OutputEvents () {
	if ( fDataFile.is_open() ) {
		for ( EventList::iterator itr = fEList.begin();
				itr != fEList.end(); ++itr ) {
			if ( !fFirstEvent ) {
				fDataFile << "," << endl;
			}
			else {
				fFirstEvent = false;
			}
			itr->Dump ( fDataFile );
		}
	}

	if ( fMonConf.fDoTrigger ) {
		for ( EventList::iterator itr = fEList.begin();
				itr != fEList.end(); ++itr ) {

			trig::TrigRslt trigData( fMonConf.fName.c_str(), itr->fComment.c_str() );
			if ( itr->fIfo.size() ) trigData.setIfos( itr->fIfo.c_str() );
			trigData.setTime ( itr->fStartTime );
			trigData.setDuration( itr->GetDuration() );
			trigData.setIntensity( itr->fMaxData );
			trigData.setSignificance ( itr->fMaxSDev );
			trigData.setFrequency ( itr->GetPeakTime().GetSecs() );
			trigData.setBandwidth ( (double)itr->fEventNPoint );
			trigData.setPriority(trig::p_info);
			trigData.setDisposition(trig::d_metaDB);
			sendTrigger( trigData );
		}
	}

	if ( !fEList.empty() ) fEList.clear();
}

//===========================================================
void glitchMon::ProcessLog () {

	Time currenttime = getDacc().getCurrentTime();

	if ( fLogStartTime == Time(0,0) ) {
		fMonStartTime = currenttime;
		fLogStartTime = currenttime;
	}

	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->UpdateHistogram( currenttime, fTrend );
	}

	fTrend.Update ( currenttime );

	// update log file every fLogInterval
	if ( currenttime - fLogStartTime >= fMonConf.fLogInterval ){

		MakeHtmlLog ();

		UpdateFiles ();

		fLogStartTime = currenttime;

	}

	return;
}

//===========================================================
	void glitchMon::MakeHtmlLog () {
		if ( !fLogFile.is_open() ) 
			return;
		char buf[128];

		html::document doc ( "glitchMon Log" );
		html::text title ( "glitchMon Log" );
		html::block titleblk ( "center" );
		titleblk.addObject ( title );
		doc.addObject ( titleblk );

		doc.addObject ( html::hline() );
		html::block timeblk ( "left" );
		html::text timeinfo ("Log Start (UTC): ");
		timeinfo << TimeStr ( fLogStartTime, buf, "%M %d, %Y at %H:%N:%S");
		timeblk.addObject ( timeinfo );
		timeblk.lineBreak ();
		html::text durinfo ("Duration : ");
		durinfo << (getDacc().getCurrentTime() - fLogStartTime);
		durinfo << "(sec)";
		timeblk.addObject ( durinfo );
		doc.addObject ( timeblk );

		doc.addObject ( html::hline() );
		html::block descblk ( "left" );
		html::text red ( "Red : Raw data statistics." );
		red.setColor ( html::color (150, 0, 0) );
		html::text blue ( "Blue : Filtered data statistics." );
		blue.setColor ( html::color (0, 0, 150) );

		descblk.addObject ( red );
		descblk.lineBreak();
		descblk.addObject ( blue );
		doc.addObject ( descblk );

		html::table results;
		results.setBorder ( true );
		results.addColumn ( "Channel Name" );
		results.addColumn ( "Mean" );
		results.addColumn ( "Sigma" );
		results.addColumn ( "Max SNR" );
		results.addColumn ( "# of Events (Low)" );
		results.addColumn ( "# of Events (High)" );
		results.addColumn ( "Length of processed data (sec)" );
		for ( ChannelList::iterator itr = fChList.begin();
				itr != fChList.end(); ++itr ) {
			itr->OutputLog ( results );
		}
		doc.addObject ( results );

		doc.addObject ( html::hline() );
		html::block updateblk ("center"); 
		html::text lasttime ("This page was last updated : "); 
		Time t = Now(); 
		lasttime << TimeStr ( t, buf, "%M %d, %Y at %H:%N:%S");
		lasttime << " (UTC) | ";
		lasttime << LocalStr ( t, buf, "%M %d, %Y at %H:%N:%S");
		lasttime << " (Local) | ";
		lasttime << double(t.getS()) << " (GPS)";
		updateblk.addObject (lasttime);
		updateblk.lineBreak(); 
		html::text author ("This monitor was written by M.Ito");  
		updateblk.addObject (author); 
		doc.addObject (updateblk); 
		doc.addObject (html::hline());

		html::writer htmlout ( fLogFile );
		doc.write (htmlout);
	}

//===========================================================
void glitchMon::ConfigLog () {
	ofstream conflog;
	Time t = Now();
	if ( fMonConf.fDoLog ) {
		const char* htmldir = getenv("DMTHTMLOUT");
		std::string log;
		if (htmldir) log = string ( htmldir ) + "/";
		ostringstream time;
		time << t;
		log += fMonConf.fName + "_";
		log += time.str();
		log += "_conf.html";
		conflog.clear();
		conflog.open ( log.c_str(), ios::out );
		if ( !conflog.is_open() ) {
			cerr << "Error opening configuration summary File." << endl;
			return;
		}
	}

	html::document doc ( "glitchMon configuration summary" );
	html::text title ( "glitchMon configuration summary" );

	html::block titleblk ( "center" );
	titleblk.addObject ( title );
	doc.addObject ( titleblk );

	doc.addObject ( html::hline() );
	html::text timeinfo ("Monitor Start (UTC) : ");
	char buf[128];
	timeinfo << TimeStr ( t, buf, "%M %d, %Y at %H:%N:%S");
	doc.addObject ( timeinfo );

	doc.addObject ( html::hline() );
	html::table monconf;
	monconf.setBorder ( true );
	monconf.addColumn ( "Monitor Name" );
	monconf.addColumn ( "Site" );
	monconf.addColumn ( "Monitor Duration (min)" );
	monconf.addColumn ( "Data Location" );
	monconf.addColumn ( "Log" );
	monconf.addColumn ( "Log Interval (min)" );
	monconf.addColumn ( "Trigger" );
	monconf.addColumn ( "Trend" );
	monconf.addColumn ( "Time Series Length (min)" );

	monconf.addRow();
	monconf.insertData ( 0, 0, fMonConf.fName );
	std::string confstr = fMonConf.GetSiteName();
	monconf.insertData ( 0, 1, confstr );
	monconf.insertData ( 0, 2, fMonConf.fDuration.GetS()/60 );
	monconf.insertData ( 0, 3, fMonConf.fDataPath );
	confstr = (fMonConf.fDoLog) ? "ON" : "OFF";
	monconf.insertData ( 0, 4, confstr );
	monconf.insertData ( 0, 5, fMonConf.fLogInterval.GetS()/60 );
	confstr = (fMonConf.fDoTrigger) ? "ON" : "OFF";
	monconf.insertData ( 0, 6, confstr );
	confstr = (fMonConf.fDoTrend) ? "ON" : "OFF";
	monconf.insertData ( 0, 7, confstr );
	monconf.insertData ( 0, 8, (long)fMonConf.fTSInterval );
	doc.addObject ( monconf );

	doc.addObject ( html::hline() );
	html::table results;
	results.setBorder (true);
	results.addColumn ( "Channel Name" );
	results.addColumn ( "Check Lock" );
	results.addColumn ( "Threshold Type" );
	results.addColumn ( "Threshold Low" );
	results.addColumn ( "Threshold High" );
	results.addColumn ( "Filter Formula" );
	results.addColumn ( "Minimum Separation" );
	results.addColumn ( "Minimum Duration" );
	results.addColumn ( "Maximum Duration" );
	results.addColumn ( "Minimum Density" );
	results.addColumn ( "Comment" );

	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->ConfigLog ( results );
	}
	doc.addObject ( results );

	doc.addObject ( html::hline() );
	html::block totalblk ("LEFT");
	html::text nchannel ("Number of Channels: ");
	nchannel << fChList.size();
	totalblk.addObject (nchannel);
	totalblk.lineBreak();
	html::text nconf ("Number of Configurations: ");
	nconf << results.getNRow ();
	totalblk.addObject (nconf);
	doc.addObject ( totalblk );

	doc.addObject (html::hline());
	html::block updateblk ("center"); 
	html::text lasttime ("This page was last updated : "); 
	lasttime << TimeStr ( t, buf, "%M %d, %Y at %H:%N:%S");
	lasttime << " (UTC) | ";
	lasttime << LocalStr ( t, buf, "%M %d, %Y at %H:%N:%S");
	lasttime << " (Local) | ";
	lasttime << TimeStr ( t, buf, "%s (GPS)");
	updateblk.addObject (lasttime);	 
	updateblk.lineBreak(); 
	html::text author ("This monitor was written by M.Ito");  
	updateblk.addObject (author); 
	doc.addObject (updateblk); 
	doc.addObject (html::hline());

	html::writer htmlout ( conflog );
	doc.write (htmlout);

	conflog.close();
}

//===========================================================
bool glitchMon::Reset () {

	if ( !OpenFiles() ) {
		return false;
	}

	getDacc().setStride (1.0);
	getDacc().setIgnoreMissingChannel(true);

	vector<TSeries> tlist;
	xsilHandlerQueryTSeries tsQ (tlist);
	xsilParser parser;
	parser.AddHandler (tsQ);
	parser.ParseFile ( kTSBackUpName );

	if ( fMonConf.fDoTrend ) {
		fTrend.setName ( fMonConf.fName.c_str() );
		fTrend.setType (Trend::kMinute);
		if ( fMonConf.fSite == MonitorConf::kLHO ) {
			fTrend.setIFO("H");
		}
		else if ( fMonConf.fSite == MonitorConf::kLLO ) {
			fTrend.setIFO("L");
		}
		else {
			fTrend.setIFO("X");
		}
		fTrend.setAutoUpdate ( false );
	}

	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr ) {
		itr->ResetStat( fMonServ, fTrend, tlist, fMonConf.fTSInterval );
	}

	if ( fMonConf.fDoTrend ) fTrend.writeIndex();

	return true;
}

//===========================================================
void glitchMon::RegisterChannels () {

	fLock.Register ( getDacc(), fMonConf.fSite );

	// channels to be monitored
	for ( ChannelList::iterator itr = fChList.begin();
			itr != fChList.end(); ++itr) {
		itr->Register ( getDacc() );
	}
}

//===========================================================
//
// setup configuration
//
//===========================================================
bool glitchMon::SetConfig( ifstream& configfile )
{

	std::string tag;
	std::string value;
	std::string remainder;
	bool valid = true;
	while (!configfile.eof() && valid) {
		if ( parse (configfile, tag, value, remainder) ) {
			if ( TagCompare ( tag, "MONITOR" )  ) {
				valid = SetMonitorParam ( value );
			}
			else if ( TagCompare( tag, "FILTER" )  ) {
				valid = AddFilterInfo ( value );
			}
			else if ( TagCompare( tag, "THRESHOLD" )  ) {
				valid = AddThresholdInfo ( value );
			}
			else if ( TagCompare( tag, "PARAMETER" )  ) {
				valid = AddParameter ( value );
			}
			else if ( TagCompare( tag, "Histogram" )  ) {
				valid = AddHistogramInfo ( value );
			}
			else if ( TagCompare( tag, "CHANNEL" )  ) {
				valid = AddChannelInfo ( value );
			}
			else {
				cerr << "Unknown Tag Error." << endl;
				valid = false;
			}

			if ( !valid ) {
				cerr << "Cofiguration File Error" << endl;
				return false;
			}
		}
	}

	return true;
}

//===========================================================
bool glitchMon::SetMonitorParam ( const std::string& value ) {
	ConfElements elements;
	if ( !GetElements ( value, elements ) ) 
		return false;

	ConfElements::iterator itr = elements.begin();
	ConfElements::iterator end = elements.end();
	for ( ; itr != end; ++itr ) {
		if ( TagCompare (itr->first, "NAME")  ) {
			fMonConf.fName = itr->second;
		}
		else if ( TagCompare (itr->first, "SITE")  ) {
			if ( TagCompare (itr->second, "LHO")  ) {
				fMonConf.fSite = MonitorConf::kLHO;
			}
			else if ( TagCompare (itr->second, "LLO")  ) {
				fMonConf.fSite = MonitorConf::kLLO;
			}
			else {
				cerr << "Invalid Site Name -> " << itr->second << endl;
				return false;
			}
		}
		else if ( TagCompare (itr->first, "DATAPATH")  ) {
			fMonConf.fDataPath = itr->second;
			if ( fMonConf.fDataPath[fMonConf.fDataPath.size() - 1] != '/' ) {
				fMonConf.fDataPath += '/';
			}
		}
		else if ( TagCompare (itr->first, "LOG")  ) {
			fMonConf.fDoLog = GetBool ( itr->second );
		}
		else if ( TagCompare (itr->first, "DURATION")  ) {
			fMonConf.fDuration =  Interval ( atoi( itr->second.c_str() ) * 60, 0);
		}
		else if ( TagCompare (itr->first, "LOGINTERVAL")  ) {
			fMonConf.fLogInterval =  Interval ( atoi( itr->second.c_str() ) * 60, 0 );
		}
		else if ( TagCompare (itr->first, "TRIGGER")  ) {
			fMonConf.fDoTrigger = GetBool ( itr->second );
		}
		else if ( TagCompare (itr->first, "TREND")  ) {
			fMonConf.fDoTrend = GetBool ( itr->second );
		}
		else if ( TagCompare (itr->first, "TSINTERVAL")  ) {
			int min = atoi( itr->second.c_str() );
			// min has to be within 4 - 24 hrs
			if ( min > 60 * 24 ) min = 60 * 24;
			else if ( min < 240 ) min = 240;

			fMonConf.fTSInterval = min;
		}
		else {
			cerr << "Monitor Parameter : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	return true;
}

//===========================================================
bool glitchMon::AddFilterInfo ( const std::string& value ) {
	ConfElements elements; 
	if ( !GetElements ( value, elements ) ) 
		return false;

	FilterInfo f;
	std::string name;

	for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
		if ( TagCompare( itr->first, "NAME")  ) {
			name = itr->second;
			f.fName = itr->second;
		} 
		else if ( TagCompare( itr->first, "FORMULA") ) {
			f.fFormula = itr->second;
			std::string::size_type pos = 0;
			while ( ( pos = f.fFormula.find_first_of('\\', pos)) != std::string::npos ) {
				f.fFormula.erase( pos, 1 );
			}
		}
		else if ( TagCompare( itr->first, "LPEFLENGTH") ) {
			f.fLPEFLength = atoi( itr->second.c_str() );
		}
		else if ( TagCompare( itr->first, "LPEFTRAINPERIOD") ) {
			f.fLPEFTrainPeriod = atoi( itr->second.c_str() );
		}
		else if ( TagCompare( itr->first, "LPEFTRAINLENGTH") ) {
			f.fLPEFTrainLength = atoi( itr->second.c_str() );
		}
		else {
			cerr << "Filter : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	if ( name.empty() || fFList.find( name.c_str() ) != fFList.end() ) {
		cerr << "AddFilterInfo error." << endl;
		return false;
	}

	fFList[ name.c_str() ] = f;

	return true;

}

//===========================================================
bool glitchMon::AddThresholdInfo ( const std::string& value ) {
	ConfElements elements;
	if ( !GetElements ( value, elements ) ) 
		return false;

	ThresholdInfo t;
	std::string name;

	for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
		if ( TagCompare( itr->first, "NAME")  ) {
			name = itr->second;
		}
		else if  (TagCompare( itr->first, "TYPE")  ) {
			if ( TagCompare( itr->second, "Absolute")  ) {
				t.fType = ThresholdInfo::kAbs;
			}
			else {
				t.fType = ThresholdInfo::kRel;
			}
		}
		else if ( TagCompare( itr->first, "LOW")  ) {
			t.fLow = atof ( itr->second.c_str() );
		}
		else if ( TagCompare( itr->first, "HIGH")  ) {
			t.fHigh = atof ( itr->second.c_str() );
		}
		else {
			cerr << "Threshold : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	if ( name.empty() || fTList.find ( name.c_str() ) != fTList.end() ) {
		cerr << "AddThresholdInfo error." << endl;
		return false;
	}
	fTList[ name.c_str() ] = t;

	return true;
}

//===========================================================
bool glitchMon::AddParameter ( const std::string& value ) {
	ConfElements elements;
	if ( !GetElements ( value, elements ) ) 
		return false;

	Parameter p;
	std::string name;

	for ( ConfElements::iterator itr = elements.begin(); itr != elements.end(); ++itr ) {
		if ( TagCompare( itr->first, "NAME")  ) {
			name = itr->second;
		}
		else if ( TagCompare( itr->first, "MINSEPARATION")  ) {
			p.fMinSeparation = Interval ( atof ( itr->second.c_str() ) );
		}
		else if ( TagCompare( itr->first, "MINDURATION")  ) {
			p.fMinDuration = Interval ( atof ( itr->second.c_str() ) );
		}
		else if ( TagCompare( itr->first, "MAXDURATION")  ) {
			p.fMaxDuration = Interval ( atof ( itr->second.c_str() ) );
		}
		else if ( TagCompare( itr->first, "MINDENSITY")  ) {
			p.fMinDensity = atof ( itr->second.c_str() );
		}
		else {
			cerr << "Parameter : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	if ( name.empty() || fCPList.find( name.c_str() ) != fCPList.end() ) {
		cerr << "AddParameter error." << endl;
		return false;
	}
	fCPList[ name.c_str() ] = p;

	return true;
}

//===========================================================
bool glitchMon::AddHistogramInfo ( const std::string& value ) {
	ConfElements elements;

	if ( !GetElements ( value, elements ) ) {
		return false;
	}

	HistogramInfo h;
	std::string name;

	for ( ConfElements::iterator itr = elements.begin();
			itr != elements.end(); ++itr ) {
		if ( TagCompare( itr->first, "NAME")  ) {
			name = itr->second;
		}
		else if ( TagCompare( itr->first, "BIN") ) {
			h.fNBins = atoi ( itr->second.c_str() );
		}
		else if ( TagCompare( itr->first, "LOW") ) {
			h.fLow = atof ( itr->second.c_str() );
		}
		else if ( TagCompare( itr->first, "HIGH") ) {
			h.fHigh = atof ( itr->second.c_str() );
		}
		else {
			cerr << "HistogramInfo : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	if ( name.empty() || fHList.find( name.c_str() ) != fHList.end() ) {
		cerr << "AddHistogramInfo error." << endl;
		return false;
	}
	fHList[ name.c_str() ] = h;

	return true;
}

//===========================================================
bool glitchMon::AddChannelInfo ( const std::string& value ) {
	ConfElements elements;
	if ( !GetElements ( value, elements ) ) {
		return false;
	}

	ChannelInfo ch;
	std::string name;

	for ( ConfElements::iterator itr = elements.begin();
			itr != elements.end(); ++itr ) {
		if ( TagCompare( itr->first, "NAME")  ) {
			if ( !ch.SetName( itr->second.c_str() ) ) {
				cerr << "Invalid Channel Name." << endl;
				return false;
			}
		}
		else if ( TagCompare( itr->first, "CHECKLOCK")  ) {
			if ( TagCompare( itr->second, "H1")  ) {
				ch.SetCheckLock ( LockStatus::kH1 );
			}
			else if ( TagCompare( itr->second, "H2")  ) {
				ch.SetCheckLock ( LockStatus::kH2 );
			}
			else if ( TagCompare( itr->second, "L1")  ) {
				ch.SetCheckLock ( LockStatus::kL1 );
			}
			else if ( TagCompare( itr->second, "H1MC")  ) {
				ch.SetCheckLock ( LockStatus::kH1MC );
			}
			else if ( TagCompare( itr->second, "H2MC")  ) {
				ch.SetCheckLock ( LockStatus::kH2MC );
			}
			else if ( TagCompare( itr->second, "L1MC")  ) {
				ch.SetCheckLock ( LockStatus::kL1MC );
			}
			else {
				ch.SetCheckLock ( LockStatus::kInvalid );
			}
		}
		else if ( TagCompare( itr->first, "CONFIG")  ) {
			ConfigInfo cinfo;
			if ( GetConfigInfo ( itr->second, cinfo ) ) {
				ch.AddConfig ( cinfo );
			}
			else {
				cerr << value << endl;
				return false;
			}
		}
		else {
			cerr << "ChannelInfo : Unknown Tag -> " << itr->first << endl;
			return false;
		}
	}

	for ( ChannelList::iterator itr = fChList.begin(); 
			itr != fChList.end(); ++itr) {
		if ( TagCompare( itr->GetName(), ch.GetName() ) ) {
			cerr << "Channel Name Redundancy." << endl;
			return false;
		}
	}
	fChList.push_back( ch );

	return true;
}

//===========================================================
bool glitchMon::GetConfigInfo ( const std::string& value, ConfigInfo& cinfo ) {

	std::string::size_type brkt_beg = value.find_first_of ('{');
	std::string::size_type brkt_end = value.find_last_of ('}');
	if ( brkt_beg >= brkt_end || brkt_beg == std::string::npos || 
			brkt_end == std::string::npos ) {
		cerr << "Config info {} mismatch." << endl;
		return false;
	}

	ConfElements elements;
	if ( !GetElements ( value.substr ( brkt_beg + 1, brkt_end - brkt_beg - 1), elements ) ) 
		return false;

	bool nothreshold = true;
	bool nofilter = true;
	bool noparameter = true;
	bool nohistogram = true;

	bool ret = true;
	for ( ConfElements::iterator itr = elements.begin();
			itr != elements.end(); ++itr ) {
		if ( itr->first == "THRESHOLD" ) {
			ThresholdList::iterator titr = fTList.find( itr->second );
			if ( titr != fTList.end() ) {
				cinfo.fThreshold = &(titr->second);
				nothreshold= false;
			}
			else {
				cerr << "Can't find THRESHOLD " << itr->second << endl;
				ret = false;
			}
		}
		else if ( itr->first == "FILTER" ) {
			FilterList::iterator fitr = fFList.find( itr->second );
			if ( fitr != fFList.end() ) {
				cinfo.fFInfo = &(fitr->second);
				nofilter = false;
			}
			else {
				cerr << "Can't find FILTER " << itr->second << endl;
				ret = false;
			}

		}
		else if ( itr->first == "PARAMETER" ) {
			ParameterList::iterator pitr = fCPList.find( itr->second );
			if ( pitr != fCPList.end() ) {
				cinfo.fParam = &(pitr->second);
				noparameter = false;
			}
			else {
				cerr << "Can't find PARAMETER " << itr->second << endl;
				ret = false;
			}
		}
		else if ( itr->first == "HISTOGRAM" ) {
			HistogramList::iterator hitr = fHList.find( itr->second );
			if ( hitr != fHList.end() ) {
				cinfo.fHist = &(hitr->second);
				nohistogram = false;
			}
			else {
				cerr << "Can't find HISTOGRAM " << itr->second << endl;
				ret = false;
			}
		}
		else if ( itr->first == "COMMENT" ) {
			cinfo.fComment = itr->second;
		}
		else {
			cerr << "Configuration : Unknown Tag -> " << itr->first << endl;
			ret = false;
		}

	}

	if ( nothreshold ) {
		ThresholdList::iterator titr = fTList.find( "Default" );
		cinfo.fThreshold = &(titr->second);
	}
	if ( nofilter ) {
		FilterList::iterator fitr = fFList.find( "Default" );
		cinfo.fFInfo = &(fitr->second);
	}
	if ( noparameter ) {
		ParameterList::iterator pitr = fCPList.find( "Default" );
		cinfo.fParam = &(pitr->second);
	}
	if ( nohistogram ) {
		HistogramList::iterator hitr = fHList.find( "Default" );
		cinfo.fHist = &(hitr->second);
	}

	if ( !ret ) {
		cerr << "ConfigInfo error." << endl;
		return false;
	}

	return true;

}

