#include "eqMon.hh"
#include <iostream>
#include <cstdlib>
#include <sstream>
#include <cmath>
#include <iomanip>

#include "FSeries.hh"
#include "Dacc.hh"
#include "Time.hh"
#include "DVector.hh"
#include "ParseLine.hh"

#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/eqMon/eqMon.cc 6557 2011-12-27 05:12:17Z john.zweizig@LIGO.ORG $"
#define PIDTITLE "Earthquake Monitor"
#include "ProcIdent.hh"

#include "TrigRslt.hh"

using namespace std;

EXECDAT(eqMon)

eqMon::eqMon(int argc, const char *argv[]) :  
  DatEnv(argc,argv),count(0),sampRate(8),eqON(0),triggerOn(false),fullData(0),
  verbose(0), quiet(0), eqStartTime(0),eqEndTime(0) {
  int i, updateSeconds(60), totalMinutes(10);
  string cfile="eqMon.conf";
  const char *sitename = getenv("LIGOSITE");
  if (sitename) cfile="eqMon_" + (string)sitename + ".conf";

  ///////////////////////  Read the configuration file
  // Note to self:  Don't use finish() in the Constructor as it must complete
  // constructor before exiting.
  for (int i=1; i<argc; i++) {
    if (!strcmp("-conf", argv[i])) {
      if (chkNxtArg(i,argv,argc)) cfile = argv[++i];
      else {
	cerr << "Couldn't find configuration file argument"<<endl;
	Usage(argv[0]);
	exit(2);
      }
    } else if (!strcmp("-verbose", argv[i])) {
      if (chkNxtArg(i,argv,argc)) {Usage(argv[0]); exit(1);}
      else {verbose=true; continue;}
    } else if (!strcmp("-quiet", argv[i])) {
      if (chkNxtArg(i,argv,argc)) {Usage(argv[0]); exit(1);}
      else {quiet=true; continue;}
    } else if (!strcmp("-h",argv[i])|| !strcmp("--help",argv[i])) {
      Usage(argv[0]);
      Help();
      exit(0);
    } else if (!strcmp("-v",argv[i])|| !strcmp("--version",argv[i])) {
      cout << string(EQMONVERSION) << endl;
      exit(0);
    } else if (isDatEnvArg(argv[i])) {i++;
    } else {Usage(argv[0]); exit(1);}
  }

  if ((verbose && quiet) || (Debug() && quiet)) {
    cerr<<"Conflicting arguments: '-quiet' should not be used with '-verbose' or '-debug'"
	<<endl;
    Usage(argv[0]);
    exit(2);
  }

  //=====================  Initialize the Parameter Dictionary
  mDict.addPar("DataPrefix","eqMon_data");
  mDict.addPar("EventPrefix","eqMon_events");
  mDict.addPar("MaxThresh",(double) 4.5);
  mDict.addPar("MinThresh",(double) 2.0);
  mDict.addPar("ChanThresh",(int) 7);
  mDict.addPar("TriggerOn",(int) 0);
  mDict.addPar("HoldTime",(int) 5);
  mDict.addPar("WaitTime",(int) 25);
  mDict.addPar("SubSample",(int) 1);
  //=====================  Read the Configuration File
  ConfigName = cfile;
  // For now, read the Config file using ConfigName
  ReadConfig();


  /////////////  Set some variables
  mDataName = mDict.getString("DataPrefix");
  mEventName = mDict.getString("EventPrefix");
  if (mDataName[0] |= '/' && mDataName[0] != '.') {
    const char* datadir = getenv("DMTOUTPUT");
    if (datadir) mDataName = string(datadir) + "/" + mDataName;
  }
  if (mEventName[0] |= '/' && mEventName[0] != '.') {
    const char* htmldir = getenv("DMTHTMLOUT");
    if (htmldir) mEventName = string(htmldir) + "/" + mEventName;
  }
  threshold = mDict.getDouble("MaxThresh");
  threshAlt = mDict.getDouble("MinThresh");
  thNStations = mDict.getInt("ChanThresh");
  triggerOn = mDict.getInt("TriggerOn");
  waitTime = mDict.getInt("HoldTime");
  durationTime = mDict.getInt("WaitTime");
  sample = mDict.getInt("SubSample");
  NChannel = mChans.size();
  if (verbose)
    cout <<"threshold="<<threshold<<", threshAlt="<<threshAlt<<", thNStations="
	 <<thNStations<<", triggerOn="<<triggerOn<<", Nchannel="<<NChannel
	 <<", waitTime="<<waitTime<<", durationTime="<<durationTime
	 <<", sample="<<sample<<endl;
  channelsOn=false;
  numRecent = sampRate * updateSeconds;
  numAveraged = numRecent * totalMinutes;
  maxCount = updateSeconds * totalMinutes;
  tdataMax= sampRate * (maxCount + updateSeconds);
  // coeff.^totalMinutes = 1/2, for moving average
  decay=exp(-log(2.0)/totalMinutes);
  chStat = new ChannelStatus[NChannel];
  ts = new TSeries*[NChannel];

  ///////////////////////  Write the output file header info
  getDacc().synch();
  char outSuffix[12];
  mDataName  += TimeStr(getDacc().getCurrentTime(),outSuffix,".%y.%m.%d");
  mEventName += TimeStr(getDacc().getCurrentTime(),outSuffix,".%y.%m.%d");

  ///////////////////////  Begin the output logs
  /////////////  First get a timestamp
  getDacc().synch();
  Time initTime(getDacc().getCurrentTime());
  char initTStr[40];
  if (!quiet) {
    dataFile.open(mDataName.c_str(),ios::out|ios::app);
    if(!dataFile.is_open()){
      cerr<<"Output file error; couldn't open "<<mDataName<<endl;
      exit(-1);
    }
    dataFile <<TimeStr(initTime,initTStr,"%Z %y/%m/%d %H:%N:%S")
	     <<"-- Starting monitor..."<<endl;
  }
  eventFile.open(mEventName.c_str(),ios::out|ios::app);
  if (!eventFile.is_open()) {
    cerr<<"Event file error; couldn't open "<<mEventName<<endl;
    exit(-1);
  }
  eventFile << TimeStr(initTime,initTStr,"%Z %y/%m/%d %H:%N:%S")
	    << "-- eqMon firing up (note ~10min initialization time)..."
	    << endl;
    
  /////////////  Add the channels from the config file
  if (!quiet) dataFile<<"Channel list from " << ConfigName << ":" << endl;
  for (i=0;i<NChannel;i++){
    ts[i] = new TSeries(Time(0),Interval(0.0),256);
    chStat[i].name = mChans[i];
    chStat[i].lastnorm=0;
    chStat[i].max = chStat[i].min = 0.0;
    if (!quiet) dataFile<<"\t" <<chStat[i].name << endl;
    getDacc().addChannel(chStat[i].name.c_str(),1,&ts[i]);
    ts[i]->setName(chStat[i].name.c_str());
  }
  getDacc().setStride(1.0);
  mChans.clear();		// No longer need a list of channel names
}

eqMon::~eqMon(void)
{
  ///////////////////////  Let's let everyone know we're shutting down
  /////////////  Get a handle on the time
  getDacc().synch();
  Time finalTime(getDacc().getCurrentTime());
  char *finalTStr = new char[40];

  /////////////  Clear out any earthquake in EPICS
  string syscall("earthquake_alarm 0 > /dev/null");
  if (!Debug() && triggerOn) {
    if (system(syscall.c_str())) cerr << "Error calling: " << syscall << endl;
  }

  /////////////  Mark the end of both logs
  if (verbose) {
    dataFile << "Closing monitor...";
    dataFile << syscall << endl;
  }
  eventFile<<"eqMon shutting down...";

  /////////////  print final statistics and close logs
  if (!quiet) {
    dataFile << TimeStr(finalTime,finalTStr,"%Z %y/%m/%d %H:%N:%S")
	     << "-- Final statistics (in order): <current_avg1> "
	     << "total_sdev1> ..." << endl;
    for (int n=0; n<NChannel;n++)
      dataFile <<setprecision(6)<<chStat[n].mean<<" "<<chStat[n].sdev<<" || ";
    dataFile <<endl;
    dataFile.close(); //// End of data file
  }
  eventFile <<TimeStr(finalTime,finalTStr,"%Z %y/%m/%d %H:%N:%S")<<endl;
  eventFile.close(); /// End of event file
  /////////////  Unallocate any data
  delete [] finalTStr;
  delete[] chStat;
  delete[] ts;
}


void eqMon::ReadConfig(void) {
  if (Debug() > 2) cout<<"Reached ReadConfig(void)"<<endl;
  /////////////////////////////////  Set up the line parser
  ParseLine lp(ConfigName.c_str());
  if (Debug()) lp.setLog(cout);
  int nLine(0);
  for (int nArg=lp.getLine(); nArg>=0; nArg=lp.getLine()) {
    nLine++;
    ///////////  Empty command line
    if (!lp[0]) {
      continue;
    ///////////  Parse a 'Parameter' line
    } else if (!strcmp(lp[0],"Parameter")) {
      if (nArg != 3) {
	cerr<<"Parameter not specified, line:"<<nLine<<endl;
	continue;
      } else for (int i=1; i<nArg; i+=2) {
	if (!mDict.exists(lp[i])) {
	  cerr<<"No such parameter: "<<lp[i]<<endl;
	  break;
	} else {
	  if (Debug()) cout<<"Set parameter: "<<lp[i]<<endl;
	  mDict.setPar(lp[i], lp[i+1]);
	}
      }
    ///////////  Add a channel for monitoring
    } else if (nArg == 1) {
      if (Debug()) cout<<"Adding channel: "<<lp[0]<<endl;
      mChans.push_back(lp[0]);
    ///////////  Invalid statement
    } else {
      cerr<<"Invalid command in "<<ConfigName<<", line "<<nLine<<": "
	  <<lp[0]<<endl;
    }
  }
}


void eqMon::ProcessData(void)
{
  int i,j,n;
  float norm,tmpSum2(0.0),sdev;
  float tmpSum1, tmpData[256];

  // 'baseTime' reset every 'maxCount' seconds
  if (!count) baseTime=ts[0]->getStartTime();

  ///////////////////////  Decimate the data to 'sampRate' counts per second
  for (n=0;n<NChannel;n++){
    ts[n]->getData(256,tmpData); // copy each channel's time series into array
    int numPts(256/sampRate);

    for (i=0;i<sampRate;i++){	// calculate the average of 32-datapoint chunks
      tmpSum1=0;
      for(j=0;j<numPts;j++) tmpSum1+=tmpData[j+i*numPts];
      if (fullData) chStat[n].tail[i+count*sampRate]= (float)tmpSum1/numPts;
      else  chStat[n].tdata.push_back((float)tmpSum1/numPts);
      if (tmpSum1 > chStat[n].max || chStat[n].max == 0) 
	chStat[n].max = (float)tmpSum1/numPts;
      if (tmpSum1 < chStat[n].min || chStat[n].min == 0) 
	chStat[n].min = (float)tmpSum1/numPts;
    }
  }
  count++;

  ///////////////////////  Check for events every 'maxCount' seconds 
  ///////////////////////  (initially 600, then 60)
  if (Debug()>1) cout << "count=" <<count<<", maxCount="<<maxCount<<" ";
  if (count==maxCount){
    if (!fullData) { // To get 11min of data, copy the last minute over again
      for (n=0;n<NChannel;n++) {
	chStat[n].tail = chStat[n].tdata.end();	// Keep track of insertion pt.
	chStat[n].place = chStat[n].tdata.begin();
	for (int iCopy(numAveraged - numRecent); iCopy < (numAveraged); 
	     iCopy++) chStat[n].tdata.push_back(chStat[n].tdata[iCopy]);
      }
      fullData=true;
    }
    if (Debug()>1)cout <<endl;
    tmpSum2=0;
    // Check for events only if some channels are ON ("noisy")
    if(channelsOn) eventCheck(); 
    if(!eqON){			
      // Recalculate each channel's decimated std. dev. using
      // decaying mean (only if we're not already in a potential earthquake)
      char *baseTStr=new char[40];
      if (!quiet) dataFile<<setprecision(15)
			  <<TimeStr(baseTime,baseTStr,"%Z %y/%m/%d %H:%N:%S")
			  <<": Updated statistics (in order): <current_mean>"
			  << "<current_sdev> <total_sdev> ..."<<endl;
      delete [] baseTStr;
      // Here we're checking the total "noisi-ness" of the channels
      for (n=0;n<NChannel;n++) {
	// calculate the average of each decimated data stream
	if (verbose && Debug()>1) 
	  dataFile << "Channel "<<n+1<<" begins at " << 0 
		   << "="<<chStat[n].tdata[0]<<"\n\t and ends at "
		   << chStat[n].tail - chStat[n].tdata.begin() - 1
		   << "="<<chStat[n].tdata[numAveraged - 1];
	chStat[n].mean=getAvg(numAveraged,chStat[n].tdata.begin(),sample);
	if (verbose && Debug()>1) dataFile<<". Made it past getAvg ";
	sdev=getSDev(numAveraged,chStat[n].tdata.begin(),
		     chStat[n].mean,sample);
	if (verbose && Debug()>1) dataFile<<"and getSDev."<<endl;
	tmpSum2+=sdev;		// tmpSum2 = 0 if no channels are active
	// Not quite proper use of moving average-- this weighs t-1 through 
	// t-10 min. more than current minute.  Would rather calculate weighted
	// avg/sdev each min
	norm=1+decay*chStat[n].lastnorm;      
	chStat[n].sdev=((sdev+chStat[n].sdev*
			 decay*chStat[n].lastnorm)/norm);
	chStat[n].lastnorm=norm;
	// output updated statistics
	if (!quiet) dataFile<<setprecision(4)<<chStat[n].mean<<" "<<sdev
			    <<" "<<chStat[n].sdev<<" || "; 
	// write over 1 min of data (= 'sampRate' * 60sec/min)
	for (int iRemap(0); iRemap<numRecent; iRemap++) 
	  chStat[n].place[iRemap]=chStat[n].tail[iRemap];
	if (Debug()>1) {
	  if (chStat[n].place == chStat[n].tdata.begin()) cout<<"Tdata Start:";
	  cout<<"copying data from "<<chStat[n].tail - chStat[n].tdata.begin()
	      <<" to "<<chStat[n].place - chStat[n].tdata.begin() <<endl;
	}
	((chStat[n].place + numRecent)== chStat[n].tail) ? 
	  chStat[n].place = chStat[n].tdata.begin():
	  chStat[n].place += numRecent;
	if (verbose && Debug()) 
	  dataFile<<"  \t"<<setprecision(15)<<baseTime<<", channel "<<n
		  <<": (Max - Min)/sigma= "<<setprecision(3)
		  <<abs(chStat[n].max - chStat[n].min)/chStat[n].sdev<<endl;
	chStat[n].min = chStat[n].max = 0;
      }
      if (!quiet) dataFile<<endl;
      tmpSum2 ? channelsOn=true: channelsOn=false;
    // Else we're in a potential  earthquake. Just update the mean and pop 1min
    } else {  
      if (verbose) 
	dataFile<<setprecision(15)<<baseTime<<": Currently in an event."<<endl;
      for (n=0;n<NChannel;n++) {
	//chStat[n].mean=getAvg(chStat[n].tdata.size(),chStat[n].tdata.begin(),8);
	if (verbose && Debug()) 
	  dataFile<<"\t"<<setprecision(15)<<baseTime<<", channel "<<n
		 <<": (Max - Min)/sigma = "<<setprecision(3)
		 <<abs(chStat[n].max - chStat[n].min)/chStat[n].sdev<<endl;
	chStat[n].min = chStat[n].max = 0;
      }
    }
    // Reset 'maxCount' to check events every 60 seconds
    if(maxCount!=60) maxCount=60;
    count=0;
  }
}

////////////////////////////////////////  Check for earthquake events
void eqMon::eventCheck(void) {
 int NeqStation;
 Time currentTime;
 if (verbose) dataFile<<"Checking for event."<<endl;
 if (Debug()) cout<<"Checking for event."<<endl;
 for (int i=0;i<numRecent;i++){	
   // Run through a minute of data -- keep track of time and check for events
   currentTime=baseTime+(double)i/sampRate; 
   NeqStation=0;
   //////////////////////  Check how many channels are continuously noisy
   for (int n=0;n<NChannel;n++){
     if (abs(chStat[n].tail[i]-chStat[n].mean) > threshold*chStat[n].sdev){
       chStat[n].signalON=1;	// (still) noisy
     } else if(chStat[n].signalON==1){ // tentatively not noisy
       chStat[n].signalON=2;
       chStat[n].endTime=currentTime;
       if (currentTime > eqEndTime) eqEndTime = currentTime;
     } else if(chStat[n].signalON==2 && // hasn't been noisy in a while
	       currentTime-chStat[n].endTime > Interval(waitTime)){
       chStat[n].signalON=0;
     }
     if (Debug()) cout <<setw(1)<<chStat[n].signalON;
     NeqStation+=(chStat[n].signalON>=1); // add a noisy channel vote
   }
   if (Debug()) cout <<endl;

   //////////////////////  Are there enough noisy channels for an event?
   if (NeqStation>=thNStations){
     if (Debug()) 
       cout << "We have minimum threshold met.  Testing..."<<endl
	    << "(currentTime, eqStartTime, triggertime) = "
	    << currentTime<<"\t" <<eqStartTime<<"\t" 
	    << (durationTime + waitTime) << endl;
     //////////  let's see if this is an event...
     if (!eqStartTime) {
       eqStartTime = currentTime;
       eqON = 1;
       if (Debug()) cout << "Is this the start of an event?"<<endl;
     //////////  Okay, this is an event, initialize...
     } else if(eqON==1 && 
	       (currentTime - eqStartTime >=Interval(durationTime+waitTime))){ 
       // Note that shortest eqGlitch= <wT+dT> - (currentTime - eqEndTime) = dT
       if (Debug()) cout << "Event's already started. This is an earthquake."
			 <<endl;
       // report the highest number of coincidently noisy channels
       maxNeqStation=NeqStation; 
       // Output the start time all channels are noisy...
       char *eqStartTStr = new char[40];
       eventFile<<"Beginning of an event: "
		<<TimeStr(eqStartTime,eqStartTStr,"%Z %y/%m/%d %H:%N:%S")
		<<"-- Threshold level="<<setprecision(2)<<threshold<<endl;
       // ...and Output the histories for each channel to this point
       if (!quiet) {
	 dataFile<<"Start of event: "
		 <<TimeStr(eqStartTime,eqStartTStr,"%Z %y/%m/%d %H:%N:%S")
		 <<"-- Threshold level="<<setprecision(2)<<threshold<<endl
		 <<"Mean and Standard Deviation for each channel (in order):"
		 <<endl;
         // (Originally this was the current difference between
         //  datapoint & mean, but this didn't make sense to R. Rahkola)
	 for (int n=0;n<NChannel;n++)
	   dataFile<<setprecision(6)<<setw(10)<<chStat[n].mean<<" "
		   <<chStat[n].sdev<<" "; 
	 dataFile<<endl;
       }
       delete [] eqStartTStr;
       // Change the threshold to threshAlt (the minimum)
       float tmpThresh(threshold);
       threshold = threshAlt;
       threshAlt = tmpThresh;
       if (triggerOn){    
	 // kluge for sending "triggers" not through TrigMgr
	 // Format: earthquake_alarm <significance#> > /dev/null
	 string syscall("earthquake_alarm ");
	 syscall += tostring(2); // significance of the alarm (>=2 is major)
	 syscall +=" > /dev/null";
	 if (verbose) dataFile << syscall << endl;
	 if (!Debug()) {
	   if (system(syscall.c_str())) cerr << "Error calling: " << syscall 
					     << endl;
	 }
       }
       eqON=2;
     //////////  else just keep track of the highest # of coincident channels
     } else if (NeqStation>maxNeqStation) maxNeqStation=NeqStation;
   ////////////////  No, not enough channels for an event.
   } else { 
     if(eqON==2) { // End the event-- nothing's noisy anymore
       char *eqEndTStr = new char[40];
       if (!quiet) {
	 dataFile<<"End of event:   "
		 <<TimeStr(eqEndTime,eqEndTStr,"%Z %y/%m/%d %H:%N:%S")
		 <<"-- Max coincident channels="<<maxNeqStation<<endl;
       }
       eventFile<<"End of event:   "
		<<TimeStr(eqEndTime,eqEndTStr,"%Z %y/%m/%d %H:%N:%S")
		<<"-- Max coincident channels="<<maxNeqStation<<endl;
       delete [] eqEndTStr;
       // Change threshold back to normal
       float tmpThresh(threshold);
       threshold = threshAlt;
       threshAlt = tmpThresh;
       // Only write triggers if the config file tells us to do so
       if (triggerOn){    
	 trig::TrigRslt trigData("eqMon","Earthquake");
	 trigData.setIfos(string(ts[0]->getName()).substr(0,2).c_str());
	 trigData.setTime(eqStartTime);
	 trigData.setPriority(trig::p_error);
	 trigData.setIntensity(double(maxNeqStation));
	 trigData.setDuration(currentTime - eqStartTime);
	 // This code is on hold until a better method of handling
	 // DMT EPICS triggers comes along
	 // trigData.setDisposition(trig::d_metaDB | trig::d_alarm);
	 trigData.setDisposition(trig::d_metaDB);
	 int trigerr=sendTrigger(trigData);
	 if (trigerr) cerr<<"eqMon: Error sending trigger=" << trigerr << endl;
	 // Kluge to send an alarm through the normal EPICS alarm handler
	 string syscall("earthquake_alarm 0 > /dev/null");
	 if (verbose) dataFile << syscall << endl;
	 if (!Debug()) {
	   if (system(syscall.c_str())) cerr << "Error calling: " << syscall 
					     << endl;
	 }
       }
     }
     if (verbose && eqON==1) {
       dataFile <<"Max glitch duration: "<< currentTime - eqStartTime <<endl;
       eventFile <<"Max glitch duration: "<< currentTime - eqStartTime <<endl;
     }
     eqON=0;
     eqStartTime = Time(0);
   } // if (NeqStation>=thNStations) ... else ...
 } // for (int i=0;i<numRecent;i++) ...	
 return;
} // void eqMon::eventCheck(void)

bool eqMon::chkNxtArg(int i, const char* argv[], const int argc) {
  if (++i < argc) {
    std::string args(argv[i]);
    if (args[0] != '-') return true;
  } // Note: Doesn't check next arg for number value
  return false;
}

int eqMon::Usage(const char* progname) {
  cerr 
    << "Usage: "<< progname <<" [-conf <file>] [-verbose [-debug n]| -quiet]\n"
    << "         [-h | --help] [-v | --version]"<<endl;
  return 1;
}

void eqMon::Help(void) {
  cerr 
    <<"eqMon detects an earthquake roughly ~1min after it starts.\n"
    <<"Various output verbosities are available.  Choose:\n"
    <<"  '-quiet' for no output (but still send triggers to EPICS)\n"
    <<"  '-verbose' for monitor status\n"
    <<"  '-debug [1]' for preventing system calls and status to stdout\n"
    <<"  '-verbose -debug [1]' for all of the above and basic channel stats\n"
    <<"  '-debug {#>1} for timing & memory status\n"
    <<"  '-verbose -debug {#>1}' for all of the above and full channel stats\n"
    <<"Please see documentation for config file parameters."<<endl;
}

//'tostring' changes an integer into a string
template<class T>
string tostring(const T& n) {
  ostringstream ost;
  ost<<n<<ends;
  return ost.str();
}

//'getAvg' averages a subsample of the data (every 'step' datapoints)
float getAvg(const int Ndata,FvCIt data,const int step)
{				// Commented lines are for debugging
  float avg(0.0);
  for (int i=0;i<Ndata;i+=step) {
    avg+=data[i];
    //    cout<<"datapt="<<data[i]<<"\t New avg="<<avg<<endl;
  }
  avg=avg*step/Ndata;
  //  cout<<"Final avg="<<avg<<endl;
  return avg;
}

//'getSDev' computes the standard deviation for a subsample of the data
float getSDev(const int Ndata,FvCIt data, const float mean, const int step)
{				// Commented lines are for debugging purposes
  float sdev(0.0);
  for (int i=0;i<Ndata;i+=step) {
    sdev+=(data[i]-mean)*(data[i]-mean); 
    //    cout<<"datapt="<<data[i]-mean<<"\t New Sdev="<<sdev<<endl;
  }
  sdev=sqrt(sdev/(float)(Ndata/step-1));
  //  cout <<"Final Sdev="<<sdev<<endl;
  return sdev;
}
