/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File NormTest.cc
//
//    Normal data monitor class. Test that all the data values are 
//    normalized floats.
//
#include <iostream>
#include <iomanip>
#include <fstream>
#include <cmath>
#include "NormTest.hh"
#include "Dacc.hh"
#include "Time.hh"
#include "TrigRslt.hh"
#include "TSeries.hh"
#include "DVector.hh"
#include "ParseLine.hh"
#ifdef __sun
#include <ieeefp.h>
#define USE_IEEE_FPCLASS
#endif

using namespace std;

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Id: NormTest.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Channel "
#include "ProcIdent.hh"

//======================================  Generate the main function.
EXECDAT(NormTest)

//======================================  Constructor
NormTest::NormTest(int argc, const char *argv[])
    : DatEnv(argc, argv), mConfig("NormTest.conf"), mGblLog(false), 
      mGblTrig(false), mLastPrint(0)
{
    //----------------------------------  Parse the arguments
    mConfig  = "NormTest.conf";
    for (int i=1 ; i<argc ; i++) {
	string argi = argv[i];
        if (argi == "-conf") {
	    mConfig = argv[++i];
        } else if (argi == "-log") {
	    mGblLog = true;
        } else if (argi == "-trig") {
	    mGblTrig = true;
        } else if (isDatEnvArg(argv[i])) {
	    ++i;
        } else {
	    cerr << "Invalid argument: " << argi << endl;
	    cerr << "Command syntax is :" << endl;
	    cerr << *argv << " [-conf <config>]" << endl;
	    finish();
	    return;
	}
    }
    getDacc().setIgnoreMissingChannel(true);

    //----------------------------------  Read the configuration file
    if (ReadConfig()) {
        finish();
    }

    const char* htmloutenv = "DMTHTMLOUT";
    mStatusFile.clear();
    if (getenv(htmloutenv)) mStatusFile = getenv(htmloutenv);
    mStatusFile += "/NormTest.txt";

    cout << "Starting NormTest $Id: NormTest.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $" << endl;
}

//======================================  Read in the configuration
bool
NormTest::ReadConfig(void) {

    //----------------------------------  Open the configuration file
    ParseLine fd(mConfig.c_str());
    if (!fd.isOpen()) {
        cerr << "Unable to open NormTest configuration file: " 
	     << mConfig << endl;
        finish();
        return true;
    }
    
    //----------------------------------  Read in the configuration file
    bool syntax   = false;
    while (fd.getLine() >= 0) {
	int nArg = fd.getCount();
	if (!nArg) continue;
	bool trig     = mGblTrig;
	bool log      = mGblLog;
	for (int i=1; i<nArg; ++i) {
	    string argi = fd[i];
	    if (argi == "-log") {
		log = true;
	    } else if (argi == "-nolog") {
		log = false;
	    } else if (argi == "-notrigger") {
		trig = false;
	    } else if (argi == "-trigger") {
		trig = true;
	    } else {
		cerr << "Undefined configuration argument: " << argi << endl;
		syntax = true;
		break;
	    }
	}
	mChanList.push_back(ChanNORM(fd[0], mTrig));
	mChanList.back().setTrigEnable(trig);
	mChanList.back().setLog(log);
	getDacc().addChannel(fd[0]);
	dynamic_cast<Dacc&>(getDacc()).setChannelFlag(fd[0], Channel::kAllowNonNormal);
    }
    return syntax;
}

//======================================  Process one frame (FrameCPP).
void
NormTest::ProcessData(void) {
    Time tFrame(getDacc().getFillTime());

    //----------------------------------  Allocate local data storage
    for (channel_iter itr=mChanList.begin(); itr!=mChanList.end(); itr++) {
        const char* ChName = itr->getChannel();

	if (Debug() > 1) cerr << "Scan AdcData: " << ChName << endl;
	itr->Scan(*getDacc().refData(ChName));
    }

    Time now = Now(); 
    if (now > mLastPrint + Interval(300.0)) {
	ofstream out(mStatusFile.c_str());
	if (out.is_open()) PrintStatus(out);
	mLastPrint = now;
    }
}

//======================================  Print final Results
NormTest::~NormTest(void) {
    //----------------------------------  Flush any remaining segments
    for (channel_iter itr=mChanList.begin(); itr!=mChanList.end(); itr++) {
	itr->writeSegment();
    }
}

void 
NormTest::PrintStatus(std::ostream& out) const {
    ChanNORM::PrintHeader(out);

    for (const_chan_iter itr=mChanList.begin(); itr!=mChanList.end(); itr++) {
	itr->PrintStatus(out);
    }
}

//======================================  Channel constructor
ChanNORM::ChanNORM(const char *name, TrigClient& tc) 
  : mChanName(name), mTrigEnable(true), mLogEnable(false), mTrig(tc),
    mCurVal(0), mStart(0), mCurrent(0), mCount(0), mLast(0)
{
}

//======================================  Channel destructor
ChanNORM::~ChanNORM() {
    writeSegment();
}

//======================================  Channel destructor
void 
ChanNORM::PrintHeader(std::ostream& out) {
    char stamp[64];
    out << "NormTest Channel statistics: " 
	<< TimeStr(Now(), stamp, "%Y-%02m-%02d %2H:%02N:%02S")
	<< endl;
    out << endl;
    out << "Channel                          Errors    Last" << endl;
}

//======================================  Channel destructor
void 
ChanNORM::PrintStatus(std::ostream& out) const {
    char stamp[64];
    std::streamsize wsav = out.width();
    out << setw(32) << left << mChanName.c_str() 
	<< setw(10) << right << mCount;
    if (!mLast) out << " --" << endl;
    else        out << TimeStr(mLast, stamp, " %Y-%02m-%02d %2H:%02N:%02S")
		    << endl;
    out.width(wsav);
}

//======================================  Check everything
void
ChanNORM::Scan(const TSeries& ts) {
    if (ts.empty()) {
	cout << "No data for channel: " << mChanName << endl;
	return;
    }
    if (!(ts.refDVect()->F_data() || ts.refDVect()->D_data())) return;

    //-----------------------------------  Scan the dataValid vector
    int       N = ts.getNSample();
    Time  tBase = ts.getStartTime();
    Interval dT = ts.getTStep();
    bool found  = false;
    bool marked = !ts.refDVect()->normal();

    int start = -1;

    //------------------------------------  Scan float data
    if (ts.refDVect()->F_data()) {
	const float* p = reinterpret_cast<const float*>(ts.refData());
	for (int i=0; i<N; ++i) {
#ifdef USE_IEEE_FPCLASS
	    switch (fpclass(p[i])) {
	    case FP_SNAN:
	    case FP_QNAN:
	    case FP_PINF:
	    case FP_NINF:
	    case FP_NDENORM:
	    case FP_PDENORM:
		if (start < 0) {
		    start = i;
		}
		found = true;
		break;
	    default:
		if (start >= 0) {
		    segment(tBase+double(start)*dT, dT*double(i-start));
		    start=-1;
		}
		break;
	    }
#else
	    switch (fpclassify(p[i])) {
	    case FP_NAN:
	    case FP_INFINITE:
	    case FP_SUBNORMAL:
		if (start < 0) {
		    start = i;
		}
		found = true;
		break;
	    default:
		if (start >= 0) {
		    segment(tBase+double(start)*dT, dT*double(i-start));
		    start= -1;
		}
	    }
#endif
	}

    //------------------------------------  Scan double data
    } else {
	const double* p = reinterpret_cast<const double*>(ts.refData());
	for (int i=0; i<N; ++i) {
#ifdef USE_IEEE_FPCLASS
	    switch (fpclass(p[i])) {
	    case FP_SNAN:
	    case FP_QNAN:
	    case FP_PINF:
	    case FP_NINF:
	    case FP_NDENORM:
	    case FP_PDENORM:
		if (start < 0) {
		    start = i;
		}
		found = true;
		break;
	    default:
		if (start >= 0) {
		    segment(tBase+double(start)*dT, dT*double(i-start));
		    start=-1;
		}
	    }
#else
	    switch (fpclassify(p[i])) {
	    case FP_NAN:
	    case FP_INFINITE:
	    case FP_SUBNORMAL:
		if (!start) {
		    start = i;
		}
		found = true;
		break;
	    default:
		if (start) {
		    segment(tBase+double(start)*dT, dT*double(i-start));
		    start=0;
		}
	    }
#endif
	} // for (i=...)
    } // if()

    //----------------------------------  Mark the end of an open interval
    if (start >= 0) {
	segment(tBase+double(start)*dT, dT*double(N-start));
    }

    //----------------------------------  Log the results
    if (mLogEnable && (found || marked)) {
	cout << "Channel " << mChanName << " found (" << found << ") marked (" 
	     << marked << ") data error in frame at GPS " << tBase.getS() 
	     << endl;
    }
}

//======================================  Check everything
void
ChanNORM::segment(const Time& t0, Interval dT) {
    if (mTrigEnable) {
	trig::TrigBase t("NormTest", mChanName.c_str(), t0, dT);
	mTrig.sendTrigger(t);
    }
}

//======================================  Set trigger enable
void 
ChanNORM::setTrigEnable(bool te) {
    mTrigEnable = te;
}

//======================================  Set log enable
void 
ChanNORM::setLog(bool le) {
    mLogEnable = le;
}

//======================================  Check everything
void
ChanNORM::writeSegment(void) {
}
