// definition of filter class
//
// Edward Daw, 2nd September 2000
#ifndef IIRfilter_HH
#define IIRfilter_HH

/**  IIRfilter is a class to store the data that defines the 
  *  function performed by the filter. This data is actually 
  *  stored in multiplication coefficients for a cascaded set
  *  of second order sections. The idea is that the second order
  *  section coefficients are set by a variety of methods, many
  *  of which will be filter design methods (eg input poles and
  *  zeros, specify passband ripple and stopband attenuation for
  *  an elliptic filter, etc). For now the only method supported
  *  is setting the filter coefficients manually from an array of
  *  doubles. More methods will be added presently.
  *  @author Edward Daw
  *  @version 1.1; Last modified October 4th, 2000
  */

class IIRfilter {

public:
  /** Default (Null) constructor.
   */
  IIRfilter( );
  
  /** Manually set filter size, name, and sampling rate.
   *  @param no_of_sections Number of second order sections in filter
   *  @param name A text name for the filter of less than 30 characters
   *  @param srate  The sampling rate (Hz) for which the filter was designed.
   */
  int set_name_srate_and_size( int no_of_sections,
			       char* name,
			       int srate );

  /** Dump filter data to screen for debugging purposes.
   */
  int put_filter_data( );

  /** Set coefficients for second order sections.
   * @param coeff_set A pointer to an array of 5 doubles giving the
   * taps for the second order section. The order is b0,b1,b2,a1,a2 , 
   * where the definitions of b,a follow the matlab convention.
   * @param section_index An integer determining which second order
   * section coefficients are being set. Should be less than the  
   * total number of second order sections in the filter.
   */
  int set_sos_coeff( double* coeff_set, int section_index );

  // get filter coefficients
  /** Retrieve coefficient b0 for filter section section_index.
   *  @param section_index which section do you want to read
   *  @param b0 pointer to memory where the coefficient will be written.
   */
  int get_b0( int section_index, double* pb0 );
  /** Retrieve coefficient b1 for filter section section_index.
   *  @param section_index which section do you want to read
   *  @param b1 pointer to memory where the coefficient will be written.
   */
  int get_b1( int section_index, double* pb1 );
  /** Retrieve coefficient b2 for filter section section_index.
   *  @param section_index which section do you want to read
   *  @param b2 pointer to memory where the coefficient will be written.
   */
  int get_b2( int section_index, double* pb2 );
  /** Retrieve coefficient a1 for filter section section_index.
   *  @param section_index which section do you want to read
   *  @param a1 pointer to memory where the coefficient will be written.
   */
  int get_a1( int section_index, double* pa1 );
  /** Retrieve coefficient a2 for filter section section_index.
   *  @param section_index which section do you want to read
   *  @param a2 pointer to memory where the coefficient will be written.
   */
  int get_a2( int section_index, double* pa2 );

  /** Destructor. Frees memory allocated to second order section array.
   */
  ~IIRfilter( );

  int no_of_sos_in_filter;
  char* filter_name;
  int sampling_rate;

private:

  filter_coeffs* second_order_sections;

};

#endif // IIRfilter_HH
