//=============================================================================
//Class Declaration: StochSensSummary.hh version 2.0 (Aug 14 2002)
//=============================================================================


#ifndef StochSensSummary_HH
#define StochSensSummary_HH

#include <fstream>
#include <list>
#include "Time.hh"

/** Helper struct for class StochSensSummary.
  * @memo Helper struct for class StochSensSummary.
  */
struct StochSensDataSummary
{
        Time::ulong_t gpstime;
        double invomegasq;
};


/** Helper class for SenseMonitor.
  * StochSensSummary produces two-week summary data files containing 
  * the 1/Omega^2 estimates and other monitor output averaged over 
  * 1000 second intervals.
  * @author 
  * @memo Produces two-week summary data files.
  */
class StochSensSummary
{
public:
    /** StochSensSummary constructor.
      * @memo Constructor.
      */
    StochSensSummary(void) {};

    /** StochSensSummary destructor.
      * @memo Destructor.
      */
    ~StochSensSummary(void);

    /// What the hell does this do?
    void init(const std::string &htmldir,
              const Time &now);

    /// Append data point to end of list.
    void append(StochSensDataSummary state);

    /// Dump data to file
    void dumpList(void);

private:
    //  length of list
    static const long int mLength = 1210;

    //  Summary "sampling" interval
    static const long int mDT = 1000;

    // list of sensitivity data for last 2 weeks
    typedef std::list<StochSensDataSummary> StochSens_List;
    typedef StochSens_List::const_iterator list_iter;
    StochSens_List mStochSensList;

    //  status file name
    std::string        mFilename;

    //  status file stream
    std::fstream       mFile;

    //  time this object was created
    Time               mCreationTime;
};


//=============================================================================
//Class Member Definitions
//=============================================================================


void StochSensSummary::init(const std::string &summary_name,
                        const Time &now)
{
//  mFilename = htmldir + "/summary.txt";
    mFilename = summary_name;

    // set the current time back to the last '000,
    // ditto the start time
    Time::ulong_t nowRound = now.getS() - (now.getS() % mDT);
    Time::ulong_t startTime = nowRound - ( mLength - 1 ) * mDT;

    std::ifstream File(mFilename.c_str(), std::ios::in);
    if ( !File.is_open( ) ) {
        // the summary file does not exist; fill in the list with -1's
        // and create the file for output

        StochSensDataSummary tmpstate;
        for (long int i = 0; i < mLength; ++i) {
            tmpstate.gpstime  = startTime + i*mDT;
            tmpstate.invomegasq = -1.;

            mStochSensList.push_back(tmpstate);
        }

        dumpList();
    } else {
        // the summary file exists; need to read it and restore
        // history;
        // there are two possibilities:
        //   1. the start of the summary file is what the start time
        //      should be
        //   2. the start of the summary file has "dropped off the
        //      screen", i.e. it is further back in the past than
        //      we need.

        // read file
        StochSensDataSummary tmpstate;
        while ( File.is_open( ) ) {
            File >> tmpstate.gpstime >> tmpstate.invomegasq;
            if (File.good() && !File.eof()) {
                mStochSensList.push_back(tmpstate);
	    } else {
	        // done reading. close file
                File.close( );
	    }
        }

        // long int fileStartTime = mStochSensList.front().gpstime;

        // pop off data that is too old
        while (!mStochSensList.empty() && mStochSensList.front().gpstime < startTime) {
            mStochSensList.pop_front();
        }

        // if the summary list is too short, pad the missing data,
        // and write out the file
        // WARNING:  This assumes any gaps in the data are at the 
	// most recent end, and not, say, in the middle.  Those gaps 
	// will not be filled.
	if (mStochSensList.size() < (size_t)mLength) {

            StochSensDataSummary tmpstate;
            while (mStochSensList.back().gpstime < nowRound) {
                tmpstate.gpstime = mStochSensList.back().gpstime + mDT;
                tmpstate.invomegasq = -1.;
                mStochSensList.push_back(tmpstate);
            }

            // write out list
            dumpList();
        }
    }
}

//
// StochSensSummary::~StochSensSummary()
//
StochSensSummary::~StochSensSummary(void)
{
  //    if (mFile.is_open()) {
  //      mFile.close();
  //  }
}

//
// StochSensSummary::append()
//
void StochSensSummary::append(StochSensDataSummary state)
{
    mStochSensList.pop_front();
    mStochSensList.push_back(state);
}


//
// StochSensSummary::dumpList()
//
void StochSensSummary::dumpList(void)
{
    // make sure output file is open
    std::ofstream File(mFilename.c_str(), std::ios::out);
    
    // set output format
    File.setf(std::ios::scientific);
    File.precision(4);
    if (!File.good() || !File.is_open()) {
        std::cerr << "StochSensSummary::dumpList(): summary status file is "
                  << " not open: opening" << std::endl;
	return;
    }

    for (list_iter iter = mStochSensList.begin(); iter != mStochSensList.end(); ++iter)
        File << (*iter).gpstime << "\t" << (*iter).invomegasq << std::endl;

    File.close();
}

#endif     //  StochSensSummary_HH
