/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef PSLBAND_HH
#define PSLBAND_HH

#include <string>
#include <vector>
#include <stdexcept>
#include "PSLTool.hh"
#include "PSLOChan.hh"
#include "FixedLenTS.hh"
#include "SegAccountant.hh"
#include "Histogram1.hh"

class MonServer;


//======================================  Band monitor class
class PSLBand : public PSLTool {
public:
    PSLBand(const std::string& name, const PSLChan* chan);
    ~PSLBand(void);
    PSLBand* clone(void) const;
    bool crunch(void);
    void params(void);
    void postConfig(MonServer& v);
    void reset(void);
    void sendTrigger(TrigClient& tc);
    void confSegment(trig::SegAccountant& sa);
    void sendSegment(trig::SegAccountant& sa);

    //------------------------------------  Accessors
    double getAvgFreq(void) const;
    double getAvgTime(void) const;
    double getFHigh(void) const;
    double getFLow(void) const;
    double getLast(void) const;
    double getLowerLimit(void) const;
    double getPeakF(void) const;
    double getPower(void) const;
    double getUpperLimit(void) const;

    /**  Reference the output channel object.
     */
    const OutChan& refOutChan(void) const;
    OutChan& refOutChan(void);

    const Histogram1& refPSDHist(void) const;

    /** Print statistics
     */
    void printStats(std::ostream& o) const;

    /** Print statistics header
     */
    void printStatHdr(std::ostream& o) const;

    //------------------------------------  Mutators
    void setAvgTime(double tAvg);
    void setAbsLimit(double Alow, double Ahigh);
    void setBand(double Flow, double Fhigh);
    void excludeBand(double Flow, double Fhigh);
    void setDeltLimit(double Dmax);
    void setFracLimit(double Fmax);
    void setFracLimit(double Fmin, double Fmax);
    void setHistEnable(bool he);
    void setSegment(const std::string& seg);

private:
    typedef std::pair<double, double> limit_node;
    typedef std::vector< limit_node > BandList;
    typedef BandList::iterator band_iter;

private:
    ///  Maximum frequency in band.
    BandList  mBandLim;

    ///  Absolute lower limit
    double    mAmin;

    ///  Absolute upper limit
    double    mAmax;

    ///  Absolute delta upper limit
    double    mDmax;

    ///  Fractional delta upper limit
    double    mFmin;

    ///  Fractional delta upper limit
    double    mFmax;

    ///  Alarm enable flag
    double mAvgTime;

    ///  Histogram enable
    bool mHisto;

    //------------------------------------  Parameters
    double mBand;
    double mLast;
    double mAvgFreq;
    double mPeakF;
    bool   mTriggered;
    trig::SegAccountant::seg_id mSegID;

    //------------------------------------  Statistics
    long       mTrigTot;
    Time       mLastValid;
    OutChan    mOutChan;
    Histogram1 mPSDHist;
};


//======================================  Inline functions
#ifndef __CINT__

inline double
PSLBand::getAvgTime(void) const {
    return mAvgTime;
}

inline double
PSLBand::getAvgFreq(void) const {
    return mAvgFreq;
}

inline double 
PSLBand::getFLow(void) const {
    if (mBandLim.empty()) return 0;
    return mBandLim[0].first;
}

inline double 
PSLBand::getFHigh(void) const {
    if (mBandLim.empty()) return 0;
    return mBandLim[mBandLim.size()-1].second;
}

inline double 
PSLBand::getLast(void) const {
    return mLast;
}

inline double
PSLBand::getPeakF(void) const {
    return mPeakF;
}

inline double 
PSLBand::getPower(void) const {
    return mBand;
}

inline const OutChan&
PSLBand::refOutChan(void) const {
    return mOutChan;
}

inline OutChan&
PSLBand::refOutChan(void) {
    return mOutChan;
}

inline const Histogram1&
PSLBand::refPSDHist(void) const {
    return mPSDHist;
}

#endif  // !defined(__CINT__)

#endif  // !defined(PSLBAND_HH)
