/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Photon Calibration Monitor
//    Greg Mendell; started November 1, 2014
//
//    For the Method used, see: LIGO-P1300120 "Spectral Line Monitoring Tool"
//
// beginincludes
#include "PCalMon.hh"
//#include "FilterDesign.hh"
#include "DaccAPI.hh"
#include "Time.hh"
#include "TSeries.hh"
#include "constant.hh"

#include <iostream>
#include <string>
#include <string.h>
#include <cstdlib>
#include <unistd.h>
#include <pwd.h>

#include <vector>
#include <cstdlib>
#include <complex>
#include <fftw3.h>

//endincludes

//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt/svn/gds/trunk/DMT/Monitors/PCalMon/PCalMon.cc 0 2014-11-1 12:00:00 gregory.mendell@LIGO.ORG $"
#define PIDTITLE  "Photon Calibration Monitor"
#include "ProcIdent.hh"

using namespace std;

//======================================  Generate the main routine.
EXECDAT(PCalMon)

//======================================  Skeleton object constructor.
PCalMon::PCalMon(int argc, const char *argv[]) 
  : DatEnv(argc, argv), maxFrame(999999), mStep(2.0),
    mLowLimit(0), mHiLimit(32767.0), mSeg(false), mSegAcct(mTClient)
{
    //----------------------------------  Look for arguments
    bool syntax  = false;
    char *pc;                    // pointer to character array
    //TSeries *pts;                // pointer to time series
    //double tendinterval=1.;      // interval between trend data points           
    int iChan;                   // index to channels
    channelName.resize(0);       // array of data channels
    frequencies.resize(0);       // array of frequencies at which to find amplitudes
    amplitude_name.resize(0);    // array of output channel names with the amplitudes
    amplitude_tseries.resize(0); // array of time series with output amplitude data
   
    nChan = 0;                // number of channels
    char serverName[256];     // data server name for DMT

    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-c") {
            // push channel names onto the channelName vector  
            pc = new char[256];
            sprintf(pc,"%s",argv[++i]);
            channelName.push_back(pc);
	} else if (argi == "-f") {
            // push frequencies as strings onto the frequencies vector  
            pc = new char[256];
            sprintf(pc,"%s",argv[++i]);
            frequencies.push_back(pc);
        } else if (argi == "-OSCcond") {
            mOsc_cond = argv[++i];
        } else if (argi == "-OSCfile") {
            mOsc_file_name = argv[++i];
	} else if (argi == "-filter") {
	    mFilter = argv[++i];
	} else if (argi == "-frame") {
	    maxFrame = strtol(argv[++i], 0, 0);
	} else if (argi == "-hilimit") {
	    mHiLimit = strtod(argv[++i], 0);
	} else if (argi == "-lolimit") {
	    mLowLimit = strtod(argv[++i], 0);
	} else if (argi == "-stride") {
	    mStep = strtod(argv[++i], 0);
	} else if (argi == "+seg") {
	    mSeg = true;
	} else if (argi == "-settle") {
	    mSettle = strtod(argv[++i], 0);
 	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    } // end parse loop

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " -c <channel> -f <frequency> [-stride <time>] \\" 
	     << endl
	     << "      [-OSCfile <oscFilename>] [-OSCcond <lockCondition] \\"
	     << endl
	     << "      [-frames <nMax>] [-dvIndex] \\"
	     << endl
	     << "      [-lolimit <segMax>] [-hilimit <segMin>] [+seg]" 
	     << endl
	     << "      [<DatEnv-args>]" 
	     << endl;
	finish();
	return;
    }

    //----------------------------------  Make sure there's a valid channel
    if(!channelName.size()) {
      cout << "No channels specified. " << endl;
      finish();
      return;
    }

    nChan = int(channelName.size()); // number of channels

    if( int(frequencies.size()) != nChan) {
      cout << "Number of frequencies specified must match number of channels specified. " << endl;
      finish();
      return;
    }
    
    if (!mFilter.empty()) {
      cout << "Filtering is not implemented." << endl;
      finish();
      return;
    }
    
    //----------------------------------  set the data accessor before Process
    getDacc().setStride(mStep);       // Length of data to be read
    // set channels
    for(iChan=0; iChan<nChan; iChan++) {
      if (iChan == 0) {
          getDacc().addChannel(channelName[iChan]);
      } else {
        if( strcmp(channelName[iChan],channelName[iChan-1]) == 0 ) {
          // same channel, so continue
        } else {
          getDacc().addChannel(channelName[iChan]);        
        }
      }
      // Add the name of the output channel to the amplitude_name list:
      pc = new char[256];
      sprintf(pc,"%s_amplitude_%sHz",channelName[iChan],frequencies[iChan]);
      amplitude_name.push_back(pc);
      // Add the pointer to the output amplitude time series to the amplitude_tseries list:  
      amplitude_tseries.push_back(new FixedLenTS(43200));
    }

    //----------------------------------  Specify for DMT Viewer
    //
    //   These methods refer to the monServer base class. The User name is 
    //   Used as a base service name, but something more descriptive is better.
    //struct passwd *pws;
    //pws = getpwuid(geteuid());
    //setServerName(pws->pw_name);
    // set server Name
    string ifo(channelName[0], 2);
    sprintf(serverName,"PCalMon_DataServer_%s",ifo.c_str());
    setServerName(serverName);

    //---- If Initialize Operator State Condition config file given, initialize and read to get conditions to check for IFO lock.
    mOsclist = NULL;
    if (!mOsc_file_name.empty()) {
       mOsclist = new OperStateCondList(getDacc());
       mOsclist->readConfig(mOsc_file_name.c_str());
       //cout << "Read mOsc_file_name = " << mOsc_file_name << endl;
       if (mOsc_cond.empty()) {
          cout << "An OSCfile was given without an OSCcond." << endl;
          finish();
          return;
       }
    } else {
      if (!mOsc_cond.empty()) {
        cout << "An OSCcond was given without an OSCfile." << endl;
        finish();
        return;
      }
    }

    //----------------------------------  Set up the trender
    
    mTrend.setName("PCalMon");      //  Set trend frame name
    mTrend.setFrameCount(1);        //  Set frames per file
    mTrend.setType(Trend::kMinute); //  Set trend type (minute trends)
    mTrend.setIFO(ifo.c_str());     //  Set trend IFO

    for(iChan=0; iChan<nChan; iChan++) {
      serveData(amplitude_name[iChan],amplitude_tseries[iChan]);
      mTrend.addChannel(amplitude_name[iChan]);
    }

    //----------------------------------  Set up segment accountant.
    if (mSeg) {
      mTClient.enroll();
      mLoSegId = segid_type("Blrms_Low",  1);
      mLoSegId.setIfo(ifo);
      mSegAcct.addSegment(mLoSegId, "Channel Blrms is low", 16);
      mHiSegId = segid_type("Blrms_High", 1);
      mHiSegId.setIfo(ifo);
      mSegAcct.addSegment(mHiSegId, "Channel Blrms is high", 16);
    }
}

//======================================  Skeleton object destructor.
PCalMon::~PCalMon() 
{
    int iChan;
    // delete channel, frequencies, amplitude_name, and amplitude_tseries lists
    for(iChan=0; iChan<nChan; iChan++) {
        delete channelName[iChan];
        delete frequencies[iChan];
        delete amplitude_name[iChan];
        delete amplitude_tseries[iChan];
    }

    if (!mOsc_file_name.empty()) {
       delete mOsclist;
    }

    cout << "PCalMon is finished" << endl;
    if (mSeg) mSegAcct.close();
}

//======================================  Frame processing function.
void
PCalMon::ProcessData(void) {

  // For use with fftw
  double* fftw_in = NULL;
  fftw_complex* fftw_out = NULL;
  int N = 0;
  int Nout = 0;
  int iChan = 0;

  // For use with time series and FFT output
  double timestep = 0.0;
  double samplerate = 0.0;
  double duration = 0;
  int freqIndex = 0;
  double freq = 0.0;
  float amplitude = 0.0;
  bool newchan = false;
  bool process_stride = true;
  TSeries* ts = NULL;

  //---- if an Operating State Condition File was given; check that IFO is locked.
  if (!mOsc_file_name.empty()) {
    if (!mOsclist->satisfied(mOsc_cond.c_str())) {
       process_stride = false;
    }
  }

  //cout <<  "Process Stride = " << process_stride << endl;

  //----------------------------------  Get pointers to the current data.
  for(iChan=0; iChan<nChan; iChan++) {
    
    //cout <<  "Channel number " << iChan << " is "<<  channelName[iChan] << endl;

    // Check if we are working on a new channel name
    if (iChan == 0) {
         newchan = true;
    } else {
       if( strcmp(channelName[iChan],channelName[iChan-1]) == 0 ) {
         // same channel, so continue
         newchan = false;
       } else {
         newchan = true;
         // Free memory from last channel
         if (process_stride) {
           fftw_free( fftw_in );
           fftw_free( fftw_out );
         }
       }
    }

    // If new channel, need to get the data, window it, and FFT it.    
    if (newchan) {

      ts = getDacc().refData(channelName[iChan]);
      if (!ts || ts->isEmpty()) {
          cout << "Channel: " << channelName[iChan] << " was not found." << endl;
          return;
      }

      // Get the number of samples in the time series and set the number of samples in the output FFT.
      N = ts->getNSample();
      Nout = ( N / 2 ) + 1;
      //cout <<  "Time Series number of elements = " <<  N << endl;
      //cout <<  "FFT expected number of elements = " <<  Nout << endl;
    
      // Get sample rate and duration
      timestep = double(ts->getTStep()); 
      samplerate = 1.0/timestep;
      duration = double(N)*timestep;
      //cout <<  "timestep = " <<  timestep << endl;
      //cout <<  "samplerate = " <<  samplerate << endl;
      //cout <<  "duration = " <<  duration << endl;

      if (process_stride) {
        fftw_plan plan_forward;
              
        // Allocate memory   
        fftw_in = (double*) fftw_malloc(sizeof(double)*N);
        fftw_out = (fftw_complex*) fftw_malloc(sizeof(fftw_complex)*Nout);

        // Set up the input time series.
        for (int i = 0; i < (N-1); i++) {
          // Hann window the data:
          double x = 2.*pi*((double)i)/((double)(N-1));
          double w = 0.5*(1.0-cos(x));
          fftw_in[i] = w*ts->getDouble(i);        
        }
    
        // Use fftw to do the real to complex 1D DFT
        plan_forward = fftw_plan_dft_r2c_1d( N, fftw_in, fftw_out, FFTW_ESTIMATE );    
        fftw_execute( plan_forward );

        // free the plan
        fftw_destroy_plan( plan_forward );

      }
      // END if (process_stride)
    }
    // END if (newchan)
    
    //----------------------------------  Calculate channel amplitude.

    // Get the corresponding frequency from the frequencies array
    freq = strtod(frequencies[iChan], 0);
    //cout <<  "Frequency number " << iChan << " is "<<  frequencies[iChan] << " = " << freq << endl;

    freqIndex = (int)(freq*duration);
    if (process_stride) {
      amplitude = (float)sqrt(fftw_out[freqIndex][0]*fftw_out[freqIndex][0] + fftw_out[freqIndex][1]*fftw_out[freqIndex][1]);
      amplitude = 4.0*amplitude/((float)N); // Get the amplitude if the line, correct for Hann Windowing. See: LIGO-P1300120 "Spectral Line Monitoring Tool"
    } else {
      amplitude = 0.0;
    }
    //cout <<  "amplitude = " << amplitude << endl;

    //----------------------------------  Trend the data points
    Time t0 = ts->getStartTime();
    //cout <<  "start time of the above: " << t0 << endl;
    mTrend.trendData(amplitude_name[iChan], t0, amplitude);
    mTrend.Update();
    amplitude_tseries[iChan]->fixedAppend(t0, ts->getInterval(), &amplitude);
    
    //----------------------------------  Add a segment
    if (mSeg) {                       //  Segments requested?
	mSegAcct.start_stride(t0);    //  Set the start time of this stride
	Time tEnd = ts->getEndTime();  //  Get the end time
	mSegAcct.set_segment("Seg_Low",  t0, tEnd, amplitude < mLowLimit);
	mSegAcct.set_segment("Seg_High", t0, tEnd, amplitude > mHiLimit);
	mSegAcct.update(tEnd);        //  This stride is over
    }

  }
  // END for(iChan=0; iChan<nChan; iChan++)

  if (process_stride) {
    // Free memory
    fftw_free( fftw_in );
    fftw_free( fftw_out );
  }

}

//======================================  Handle Messages
void
PCalMon::Attention(void) {
    MonServer::Attention();
}
