/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_WSTRCHAN_HH
#define WPIPE_WSTRCHAN_HH

#include "weventlist.hh"
#include "wparameters.hh"
#include "wtile.hh"
#include "wresample.hh"
#include <vector>

class MultiDacc;

namespace wpipe {
   class woutput;

   /**  The %wstream_chan class defines a processes a group of channels that
     *  are to be processed in an identical manner and with a common tiling. 
     *  Any channels that are to be analysed coherently (not implemented yet)
     *  must be in the same group.
     */
   class wstream_chan {
   public:
      /**  Construct a %wstream_chan instance. The channel names, analysis 
        *  parameters, etc. are specified in the wparameters instance passed
	*  to the constructor.
	*  \param par Channel names, sources and analysis parameters for 
	*             this channel group.
	*  \param debugLevel Debugging printout level.
	*/
      wstream_chan(const wparameters& par, int dbugLvl);
      ~wstream_chan(void);
      void addChannels(MultiDacc& md) const;
      void addOutputFiles(woutput& wo, const std::string& fileTime, 
			  const std::string& outdir) const;
      void inject(MultiDacc& md);
      double minimumOverlap(void) const;
      size_t numberOfChannels(void) const;
      void process(const woutput& wo, const Time& blockStartTime, 
		   Interval blockDuration, Interval blockOverlap);

      /**  Read in state data for times from start - start+dT and select 
        *  channels that should have good data. Read data from the good 
	*  channels for the specified times. On return, validChannels 
	*  indicates channels that pass the state criteria and input data 
	*  for the valid channels are available in input_data[].
	*/
      void read_data(MultiDacc& md, const Time& start, double dT);
      const wparameters& refParameters(void) const;
      const wtile& refTiling(void) const;
      void reset(void);
      void start_chunk(void);
      weventstack& trigClusters(void);
      weventstack& trigDownselect(void);
      void validChans(str_vect& processedChannels) const;

   private:
      wparameters pars;
      wtile       tiling;
      int         debugLevel;
      double      blockOverlap;
      double      minimumTimeShift;
      double      maximumTimeShift;
      tser_vect   input_data;
      tser_vect   state_data;
      resampler   resample;

      /**  The state_index vector indicates which states need to be evaluated 
        *  independently. If state_index < 0, the state is always true, \e i.e.
	*  the state channel name was empty or "NONE" or the mask was zero.
	*  If \c state_index>=0 it indicates the first channel with an identical
	*  state. The state need be evaluated only for states where 
	*  state_index[i] == i
	*/ 
      std::vector<int> state_index;
      bool_vect   validChannels;
      weventstack triggersDownselected;
      weventstack clusters;
   };

   typedef std::vector<wstream_chan> strchan_vect;

   //===================================  Inline methods
   inline size_t
   wstream_chan::numberOfChannels(void) const {
      return pars.numberOfChannels();
   }

   inline const wparameters& 
   wstream_chan::refParameters(void) const {
      return pars;
   }

   inline const wtile& 
   wstream_chan::refTiling(void) const {
      return tiling;
   }

   inline weventstack& 
   wstream_chan::trigClusters(void) {
      return clusters;
   }

   inline weventstack& 
   wstream_chan::trigDownselect(void) {
      return triggersDownselected;
   }
}
#endif // !defined(WPIPE_WSTRCHAN_HH)
