// WMEASURE Measure peak and weighted signal properties from Q transforms
//
// WMEASURE reports the peak and significance weighted mean properties of Q
// transformed signals within the specified time-frequency region.
//
// usage:
//
//   measurements = wmeasure(transforms, tiling, startTime, referenceTime, 
//                           timeRange, frequencyRange, qRange, debugLevel);
//
//   transforms           cell array of input Q transform structures
//   tiling               discrete Q transform tiling structure from WTILE
//   startTime            GPS start time of Q transformed data
//   referenceTime        reference time for time range to search over
//   timeRange            vector range of relative times to search over
//   frequencyRange       vector range of frequencies to search over
//   qRange               scalar Q or vector range of Qs to search over
//   debugLevel           verboseness of debug output
//
//   measurements         cell array of measured signal properties
//
// WMEASURE returns a cell array of measured signal properties, with one cell per
// channel.  The measured signal properties are returned as a structure that
// contains the following fields.
//
//   peakTime                 center time of peak tile [gps seconds]
//   peakFrequency            center frequency of peak tile [Hz]
//   peakQ                    quality factor of peak tile []
//   peakDuration             duration of peak tile [seconds]
//   peakBandwidth            bandwidth of peak tile [Hz]
//   peakNormalizedEnergy     normalized energy of peak tile []
//   peakAmplitude            amplitude of peak tile [Hz^-1/2]
//   signalTime               weighted central time [gps seconds]
//   signalFrequency          weighted central frequency [Hz]
//   signalDuration           weighted duration [seconds]
//   signalBandwidth          weighted bandwidth [Hz]
//   signalNormalizedEnergy   total normalized energy []
//   signalAmplitude          total signal amplitude [Hz^-1/2]
//   signalArea               measurement time frequency area []
//
// The user can focus on a subset of the times and frequencies available in
// the transform data by specifying a desired range of central times,
// central frequencies, and Qs to threshold on.  Ranges should be specified
// as a two component vector, consisting of a minimum and maximum value.
// Alternatively, if (only a single Q is specified, WMEASURE is only applied to
// the time-frequency plane which has the nearest value of Q in a
// logarithmic sense to the requested value.
//
// To determine the range of central times to search over, WMEASURE requires
// the start time of the transformed data in addition to a reference time
// and a relative time range.  Both the start time and reference time should
// be specified as absolute quantities, while the range of times to analyze
// should be specified relative to the requested reference time.
//
// By default, WMEASURE is applied to all available frequencies and Qs, and the
// reference time and relative time range arguments are set to exclude data
// potentially corrupted by filter transients as identified by the transient
// duration field of the tiling structure.  The default value can be
// obtained for any argument by passing the empty matrix [].
//
// See also WTILE, WCONDITION, WTRANSFORM, WTHRESHOLD, WSELECT, WEXAMPLE, WSCAN,
// and WSEARCH.

// Notes:
// 1. Compute absolute or normalized energy weighted signal properties?
// 2. Only include tiles with Z>Z0 in integrands?

// Shourov K. Chatterji
// shourov@ligo.caltech.edu

// $Id: wmeasure.m 1716 2009-04-10 17:00:49Z jrollins $
#include "wtypes.hh"
#include "Time.hh"
#include "Interval.hh"
#include <iosfwd>

namespace wpipe {
  class wtile;
  class wtransform;

  /**  The wrprops structure holds both tile and signal properties. The 
    *  signal properties are an average of all significant tiles. The 
    *  number of average terms is used to count terms added. When the 
    *  average is completed, the number of terms is reset.
    */
  struct wprops {
    /** construct an empty properties structure
     */
    wprops(void);

    /** construct a properties structuree for a single tile.
     */
    wprops(const Time& tRef, const Time& tTile, double fTile, double qTile, 
	   double normE, double meanE=1.0, double dArea=1.0);

    Time centerTime(void) const;
    void Average(void);
    void dump(std::ostream& out) const;
    wprops& operator+=(const wprops& prop);

    std::string id;               // identifier string.
    Time refTime;
    double TimeOff;               // center time offset [seconds]
    double Frequency;             // center frequency [Hz]
    double Q;                     // quality factor []
    double Duration;              // duration [seconds]
    double Bandwidth;             // bandwidth [Hz]
    double NormalizedEnergy;      // normalized energy []
    double CalibratedEnergy;      // calibrated energy []
    double Amplitude;             // amplitude [Hz^-1/2]
    double Area;                  // measurement time frequency area []
    double sumWeight;             // sum of weight for averages.
    int    nAverageTerms;         // Number of values summed. reset afer svg.
  };

  inline Time 
  wprops::centerTime(void) const {
    return refTime + Interval(TimeOff);
  }

  struct chan_props {
    void dump(std::ostream& out) const;
    wprops peak;   /// peak tile properties
    wprops signal; /// global signal properties.
  };

  class wmeasure {
  public:
    wmeasure(const wtransform& transforms, const wtile& tiling, 
	     const Time& startTime, const Time& referenceTime, 
	     const dble_vect& timeRange, const dble_vect& frequencyRange, 
	     const dble_vect& qRange, int debugLevel=0);
    ~wmeasure(void);

    void dump(std::ostream& out) const;
    const chan_props& operator[](int i) const;
  private:
    typedef std::vector<chan_props> prop_vect;
    prop_vect measurements;
  };

  //====================================  Measurement inline methods.
  inline const chan_props&
  wmeasure::operator[](int i) const {
    return measurements[i];
  }

} // namespace wpipe
