// WCONDITION High pass filter and whiten time series data
//
// WCONDITION high pass filters and whitens time series data prior to analysis
// by the Q transform.  The data is first zero-phase high pass filtered at the
// minimum analysis frequency specified in the tiling structure.  The resulting
// high pass filtered data is then whitened at
// a frequency resolution equal to the minimum analysis bandwidth requested in
// the tiling structure.  Note that the resulting whitened data is returned as
// a frequency series, not a time series.  In addition, the returned frequency
// series extends from zero frequency to the Nyquist frequency.  As a result,
// they are of length N / 2 + 1, where N is the length of the individual input
// time series.
//
// To enable recovery of calibrated amplitude and phase information, WCONDITION
// also returns the effective frequency domain coefficients of the combined high
// pass and whitening filters for each channel.
//
// WCONDITION also permits double whitening of the data to support true matched
// filtering.  Regardless of whether double whitening is requested or not, the
// returned frequency-domain filter coefficients always correspond to the single
// whitened case.
//
// usage:
//
//   [conditionedData, coefficients] = wcondition(rawData, tiling, doubleWhiten);
//
//   rawData               vector of input time series
//   tiling                discrete Q transform tiling structure from WTILE
//   doubleWhiten          bool for double whitening
//
//   conditionedData       cell array of conditioned output frequency series
//   coefficients          cell array of frequency domain filter coefficients
//   doubleWhiten          boolean flag to enable double whitening
//
// There is also an alternative output syntax, which provides access to the
// intermediate raw and high pass filtered data for use by WSCAN.
//
//   [rawData, highPassedData, whitenedData, coefficients] = 
//       wcondition(rawData, tiling, doubleWhiten);
//
//   rawData               cell array of unconditioned frequency series
//   highPassedData        cell array of high pass filtered frequency series
//   whitenedData          cell array of whitened frequency series data
//   coefficients          cell array of frequency domain filter coefficients
//
// See also WTILE, WTRANSFORM, WTHRESHOLD, WSELECT, WSEARCH, WEVENT, WSCAN,
// and WEXAMPLE.

// Shourov K. Chatterji <shourov@ligo.mit.edu>
// Antony Searle <antony.searle@anu.edu.au>
// Jameson Rollins <jrollins@phys.columbia.edu>

#include "wtypes.hh"

class IIRFilter;

namespace wpipe {

  class wtile;

  class wcondition {
  public:
    wcondition(const tser_vect& rawData, const wtile& tiling, 
	       bool doubleWhiten=false);
    ~wcondition(void);
    /** Condition a single channel.
     */
    void single_chan(const TSeries& rawData, const wtile& tiling, 
		     TSeries& hpData, containers::DFT& hpDft, 
		     containers::DFT& whiteDft,
		     containers::DFT& coeffs, bool doubleWhiten=false);
    void coefficientDFT(dft_vect& dftvec) const;
    TSeries highPassedData(void) const;
    const containers::DFT& highPassedDFT(void) const;
    const containers::DFT& rawDFT(void);
    TSeries whitenedData(void) const;
    void whitenedData(tser_vect& dftvec) const;
    const containers::DFT& whitenedDFT(void) const;
    void whitenedDFT(dft_vect& dftvec) const;

  private:
    tser_vect  raw;
    dft_vect   rawDft;
    dft_vect   whitenedDft;
    tser_vect  highPassData;
    dft_vect   highPassedDft;
    dft_vect   coefficients;
    IIRFilter* highPassFilter;
    containers::fSeries hpfResponse;
  };

  inline const containers::DFT&
  wcondition::highPassedDFT(void) const {
    return highPassedDft[0];
  }

  inline const containers::DFT&
  wcondition::whitenedDFT(void) const {
    return whitenedDft[0];
  }

} // namespace wpipe
