#include "matlab_fcs.hh"
#include "scandir.hh"
#include <stdexcept>
#include <cstdlib>
#include <iostream>
#include <time.h>
#include <unistd.h>

using namespace std;

namespace wpipe {

  //====================================  Skip pointer past spaces and tabs.
  inline const char*
  skip_spaces(const char* p) {
    while (*p == ' ' || *p == '\t') p++;
    return p;
  }

  //====================================  Convert date to string
  std::string 
  datestr(unsigned long t, int len) {
    struct tm tstruct;
    time_t t0 = t;
    if (!t0) t0 = time(0);
    string s = asctime(localtime_r(&t0, &tstruct));
    return s.substr(0, len);
  }

  //====================================  Remove blanks from start/end of string
  std::string
  deblank(const std::string& str) {
    std::string::size_type ifrst = str.find_first_not_of(" \t\n");
    if (ifrst == std::string::npos) return "";
    std::string::size_type ilast = str.find_last_not_of(" \t\n");
    if (ilast != std::string::npos) ilast++;
    return str.substr(ifrst, ilast-ifrst);
  }

  //====================================  Diaplay a numeric vector parameter.
  std::ostream& 
  display(const dble_vect& dv, std::ostream& out) {
    size_t N = dv.size();
    out << "[";
    for (size_t i=0; i<N; i++) {
      if (i) out << ", ";
      out << dv[i];
    }
    return out << "]";
  }

  //====================================  Diaplay a numeric vector parameter.
  std::ostream& 
  display(const str_vect& sv, std::ostream& out) {
    size_t N = sv.size();
    out << "{";
    for (size_t i=0; i<N; i++) {
      if (i) out << ", ";
      out << "\"" << sv[i] << "\"";
    }
    return out << "}";
  }

  //====================================  Throw a run-time error exception.
  void
  error(const std::string& err) {
    throw std::runtime_error(err);
  }

  //====================================  Inline function to test for "Inf"
  inline int
  test_inf(const char* p) {
    int rc = 1;
    if (*p == '-') {
      rc = 2;
      p++;
    }
    if (*p != 'i' && *p != 'I') return 0;
    if (*(++p) != 'n') return 0;
    if (*(++p) != 'f') rc = 0;
    return rc;
  }

  //====================================  Parse a list of numbers (allow inf).
  dble_vect
  eval(const std::string& str) {
    dble_vect r;
    const char* ptr = str.c_str();
    if (*ptr != '[') {
      char* pLast = 0;
      r.push_back(strtod(ptr, &pLast));
      if (*pLast) error(string("Unparseable numeric vector: ")+str);
    } else {
      ptr++;
      while (*ptr && *ptr != ']') {
	ptr = skip_spaces(ptr);
	const char* svptr = ptr;
	char* eptr = const_cast<char*>(ptr);
	switch (test_inf(ptr)) {
	case 0:
	  r.push_back(strtod(ptr, &eptr));
	  break;
	case 1:
	  r.push_back(1.0/0.0);
	  eptr += 3;
	  break;
	case 2:
	  r.push_back(-1.0/0.0);
	  eptr += 4;
	  break;
	}
	ptr = skip_spaces(eptr);
	if (*ptr == ',' || *ptr == ';') ptr = skip_spaces(++ptr);
	if (ptr == svptr) error(string("Unparseable numeric vector: ")+str);
      }
    }
    return r;
  }

  
  //====================================  Parse a list of strings.
  str_vect
  eval_str(const std::string& str) {
    str_vect r;
    string s;
    const char* ptr = str.c_str();
    bool instr = false;
    while (*ptr && *ptr != '}') {
      switch (*ptr) {
      case '\'':
      case '"':
	if (instr) {
	  r.push_back(s);
	  s.clear();
	}
	instr = !instr;
	break;
      case '{':
      case ' ':
      case ',':
	if (!instr) break;
      default:
	if (instr) {
	  s += *ptr;
	} else {
	  error("Unexpected character");
	}
      }
      ptr++;
    }
    if (instr) error("Unterminated string");
    //cout << "eval_str output = [";
    //for (size_t i=0; i<r.size(); i++) cout << " " << r[i];
    //cout << " ]" << endl;
    return r;
  }

  //====================================  Test the exiistence of a file.
  bool
  exist(const std::string& path, const std::string& type) {
    bool rc = false;
    if (type == "file") {
      rc = (access(path.c_str(), F_OK) == 0);
    }
    else if (type == "dir") {
      rc = scandir::test_directory(path);
    }
    else {
      rc = false;
    }
    return rc;
  }

  //====================================  Greatest common divisor
  //                                      Use Euler's algorithm
  long
  gcd(long a, long b) {
    if (!a) return b;
    if (b > a) {
      long t = b;
      b = a;
      a = t;
    }
    while (b) {
      long t = b;
      b = a % b;
      a = t;
    }
    return a;
  }

  //====================================  Exponent of 2 for 2^N >= x
  long
  nextpow2(double x) {
    double powval(1.0);
    long ipow(0);
    while (powval < x) {
      powval *= 2.0;
      ipow++;
    }
    return ipow;
  }

  //====================================  Polynomial value.
  double 
  polyval(const double* coef, int N, double x) {
    double sum = coef[0];
    for (int i=1; i<N; i++) {
      sum *= x;
      sum += coef[i];
    }
    return sum;
  }

  //====================================  Replace a string.
  std::string 
  strrep(const std::string& str, const std::string& from, const std::string& to) {
    std::string temp(str);
    string::size_type frlen = from.size();
    if (!frlen) return temp;
    while (str.find(from) != string::npos) {
      temp.replace(str.find(from), frlen, to);
    }
    return temp;
  }

  //====================================  Split a string at specified delimiters
  str_vect 
  strsplit(const std::string& str, const std::string& delim) {
    str_vect ret;
    string temp = str;
    while (!temp.empty()) {
      string::size_type inx = temp.find_first_of(delim);
      ret.push_back(temp.substr(0,inx));
      if (inx == string::npos) temp.clear();
      else                     temp.erase(0,inx+1);
    }
    return ret;
  }

  //====================================  Convert string to lower case
  std::string 
  tolower(const std::string& str) {
    std::string temp(str);
    string::size_type strlen = temp.size();
    if (!strlen) return temp;

    for (string::size_type i=0; i<strlen; i++) {
      temp[i] = ::tolower(temp[i]);
    }
    return temp;
  }

  //====================================  Convert string to upper case
  std::string 
  toupper(const std::string& str) {
    std::string temp(str);
    string::size_type strlen = temp.size();
    if (!strlen) return temp;

    for (string::size_type i=0; i<strlen; i++) {
      temp[i] = ::toupper(temp[i]);
    }
    return temp;
  }

  //====================================  Remove leading and/or trailing quotes
  std::string 
  unquote(const std::string& str) {
    if (str.empty()) return str;
    string::size_type front = 0;
    string::size_type last  = str.size();
    if (str[front]  == '\'' || str[front]  == '"') front++;
    if (str[last-1] == '\'' || str[last-1] == '"') last--;
    if (last <= front) return "";
    return str.substr(front, last-front);
  }

  //====================================  Conditionally log a message.
  void 
  wlog(int thresh, int level, const std::string& txt) {
    if (thresh >= level) cout << txt << endl;
  }
}
