/***************************************************************************
    File        : LowPassFilter.h
    Description : Generates a low-pass filter
 ---------------------------------------------------------------------------
    Begin       : Mon Sep 3 2001
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __LOWPASSFILTER_H
#define __LOWPASSFILTER_H

/*!
 * \file LowPassFilter.h
 *  Computes the filter coefficients for a windwed-sinc low-pass filter,
 *  see e.g.
 * `The Scientist and Engineer's Guide to Digital Signal Processing
 *  Chapter 16.
 * \brief Computes a windowed-sinc low-pass filter.
 */



// Libs
#include <cstdlib>
#include <cmath>
#include <limits>
#include <vector>

// Project
#include "Types.h"
#include "Numeric.h"

namespace gwd {

  //! Computes a windowed-sinc low-pass filter.
  class LowPassFilter {
  public:
    /** Default constructor */
    LowPassFilter()  {}

    /** Destructor */
    ~LowPassFilter() {}

    // Methods
    /**
     * This method computes a windowed-sinc low-pass filter. You need to specifiy
     * the signal sampling rate, the cutoff frequency and the filter order.
     * @param SamplingRate signal sampling rate
     * @param cutoffFrq cutoff frequency
     * @param order filter order
     * @param filter std::vector containing the filter coefficients
     */
    inline void ComputeFilter(const double SamplingRate,const double cutoffFrq,const unsigned int order,Vector& filter);

  private:
    /** The sinc function sinc(x) = sin(x)/x */
    inline double sinc(const double x,const int y);
  };


  inline double LowPassFilter::sinc(const double x,const int y)
  {
    double res = double(0);
    if (y == 0)
      res = x;
    else
    {
      const double val = static_cast<double>(y);
      res = std::sin(x*val)/val;
    }
    return res;
  }

  inline void LowPassFilter::ComputeFilter(const double SamplingRate,const double cutoffFrq,const unsigned int order,Vector& h)
  {
    const double zweipi = 2.*gwd::PI;
    const int fltSize = static_cast<int>(order) + 1;  // no. of taps = filter order + 
    // set filter
    h.resize(fltSize);
    const double x = 2.*gwd::PI*cutoffFrq/SamplingRate;
    double KK = double(0);
    for (int nn = 0; nn < fltSize; nn++)
    {
      double z = zweipi*static_cast<double>(nn)/static_cast<double>(fltSize);
      const int y = nn - fltSize/2;
      h[nn] = LowPassFilter::sinc(x,y)*(0.42 - 0.5*std::cos(z) + 0.08*std::cos(2.*z)); // Blackman window
      //h[nn] = LowPassFilter::sinc(x,y)*(0.54 - 0.46*std::cos(z)); // Hamming window
      KK += h[nn];
    }
    // normalize
    gwd::Multiply<double>(h,1./KK);
  }
} // namespace gwd
#endif // __LOWPASSFILTER_H

