/***************************************************************************
    File        : HighPassFilter.h
    Description : Generates a high-pass filter
 ---------------------------------------------------------------------------
    Begin       : Mon Sep 3 2001
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __HIGHPASSFILTER_H
#define __HIGHPASSFILTER_H

/*!
 * \file HighPassFilter.h
 *  Computes the filter coefficients for a windwed-sinc high-pass filter,
 *  see e.g.
 * `The Scientist and Engineer's Guide to Digital Signal Processing
 *  Chapter 16.
 * \brief Computes a windowed-sinc high-pass filter.
 */


// Libs
#include <cstdlib>
#include <cmath>
#include <limits>
#include <vector>


// Project
#include "Types.h"
#include "Numeric.h"

namespace gwd {

  //! Computes a windowed-sinc high-pass filter.
  class HighPassFilter {
  public:
    /** Default Constructor */
    HighPassFilter()  {}

    /** Destructor */
    ~HighPassFilter() {}

    // Methods
    /**
     * This method computes a windowed-sinc high-pass filter. You need to specifiy
     * the signal sampling rate, the cutoff frequency and the filter order.
     * @param SamplingRate signal sampling rate
     * @param cutoffFrq cutoff frequency
     * @param order filter order
     * @param filter std::vector containing the filter coefficients
     */
    inline void ComputeFilter(const double SamplingRate,const double fc,const unsigned int order,Vector& filter);

  private:
    /** The sinc function sinc(x) = sin(x)/x */
    inline double sinc(const double x,const int y);
  };


  inline double HighPassFilter::sinc(const double x,const int y)
  {
    double res = double(0);
    if (y == 0)
      res = x;
    else
    {
      const double val = static_cast<double>(y);
      res = std::sin(x*val)/val;
    }
    return res;
  }

  inline void HighPassFilter::ComputeFilter(const double SamplingRate,const double fc,const unsigned int order,Vector& h)
  {
    const double zweipi = 2.*gwd::PI;
    const int fltSize = static_cast<int>(order) + 1;  // no. of taps = filter order + 
    // set filter
    if ((int)h.size() != fltSize)
      h.resize(fltSize);

    const double x2 = 2.*gwd::PI*fc/SamplingRate;
    double KK = double(0);
    for (int nn = 0; nn < fltSize; nn++)
    {
      double z = zweipi*static_cast<double>(nn)/static_cast<double>(fltSize);
      //double y = static_cast<double>(nn - fltOrder/2);
      double val = (0.42 - 0.5*std::cos(z) + 0.08*std::cos(2.*z));
      h[nn] = HighPassFilter::sinc(x2,nn - fltSize/2)*val;
      KK += h[nn];
    }
    // normalize
    gwd::Multiply<double>(h,1./KK);

    // Change low-pass filter h2 into high pass filter
    for (int nn = 0; nn < fltSize; nn++) h[nn] = -h[nn];
    h[fltSize/2] += 1.;
  }
} // namespace gwd
#endif // __HIGHPASSFILTER_H

