/***************************************************************************
    File        : BandPassFilter.h
    Description : Generates a band-pass filter
 ---------------------------------------------------------------------------
    Begin       : Tue June 14 2005
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __BANDPASSFILTER_H
#define __BANDPASSFILTER_H

/*!
 * \file BandPassFilter.h
 *  Computes the filter coefficients of a windwed-sinc band-pass filter,
 *  see e.g.
 * `The Scientist and Engineer's Guide to Digital Signal Processing
 *  Chapter 16.
 * \brief Computes a windowed-sinc band-pass filter.
 */


// Libs
#include <cstdlib>
#include <cmath>
#include <limits>
#include <vector>


// Project
#include "Types.h"
#include "Numeric.h"

namespace gwd {

  //! Computes a windowed-sinc band-pass filter.
  class BandPassFilter {
  public:
    /** Default contructor */
    BandPassFilter()  {}

    /** Destructor */
    virtual ~BandPassFilter() {}

    // Methods
    /**
     * This method computes a windowed-sinc band-pass filter. You need to specifiy
     * the signal sampling rate, the lower and upper cutoff frequencies and
     * the filter order.
     * @param SamplingRate signal sampling rate
     * @param fc1 lower cutoff frequency
     * @param fc2 upper cutoff frequency
     * @param order filter order
     * @param filter is a std::vector containing the filter coefficients
     */
    inline void ComputeFilter(const double SamplingRate,const double fc1,const double fc2,const unsigned int order,Vector& filter);

  private:
    /** The sinc function sinc(x) = sin(x)/x */
    inline double sinc(const double x,const int y);
  };


  inline double BandPassFilter::sinc(const double x,const int y)
  {
    double res = double(0);
    if (y == 0)
      res = x;
    else
    {
      const double val = static_cast<double>(y);
      res = std::sin(x*val)/val;
    }
    return res;
  }

  inline void BandPassFilter::ComputeFilter(const double SamplingRate,const double fc1,const double fc2,const unsigned int order,Vector& h)
  { 
    const double zweipi = 2.*gwd::PI;
    const int fltSize = static_cast<int>(order) + 1; // no. of taps = filter order + 1
    Vector h1(fltSize);
    Vector h2(fltSize);
    // set filter
    h.resize(fltSize);
    const double x1 = 2.*gwd::PI*fc1/SamplingRate;
    const double x2 = 2.*gwd::PI*fc2/SamplingRate;
    double KK1 = double(0);
    double KK2 = double(0);
    for (int nn = 0; nn < fltSize; nn++)
    {
      double z = zweipi*static_cast<double>(nn)/static_cast<double>(fltSize);
      //double y = static_cast<double>(nn - fltOrder/2);
      double val = (0.42 - 0.5*std::cos(z) + 0.08*std::cos(2.*z));
      h1[nn] = BandPassFilter::sinc(x1,nn - fltSize/2)*val;
      h2[nn] = BandPassFilter::sinc(x2,nn - fltSize/2)*val;
      KK1 += h1[nn];
      KK2 += h2[nn];
    }
    // normalize
    gwd::Multiply<double>(h1,1./KK1);
    gwd::Multiply<double>(h2,1./KK2);

    // Change low-pass filter h2 into high pass filter
    for (int nn = 0; nn < fltSize; nn++) h2[nn] = -h2[nn];
    h2[fltSize/2] += 1.;

    // Create band-pass
    h.resize(fltSize);
    for (int nn = 0; nn < fltSize; nn++) h[nn] = -(h1[nn] + h2[nn]);
    h[fltSize/2] += 1.;
  }
} // namespace gwd
#endif // __BANDPASSFILTER_H

