#ifndef __CINT__
#include <sys/types.h>
#include <fcntl.h>
#include <iostream>
#include <fstream>
#include <unistd.h>
#include <cstdlib>
#include "MultiVolt.hh"
#include "VoltWatcher.hh"
#include "VoltWriter.hh"
#include "VoltBase.hh"

using namespace std;

const char* const monname = "MultiVolt";

/// Create main program
   EXECDAT (MultiVolt);
#endif

   MultiVolt::MultiVolt (int argc, const char* argv[])
   : DatEnv (argc, argv), MonServer (monname)
   {
      for (int vw=0; vw < maxChannels; vw++) 
         array[vw] = 0;
   
      if (argc<2) { 
         HelpMessage();		// Program needs at least a filename
         finish();
         return;
      }
   
      // Command line argument parsing
      int 		c;		/* option */
      extern char*	optarg;		/* option argument */
      errflag = false;
      quiet = false;
      ismonserv = true;
      usetrend = notrend;
      useharmonics = true;
      makeindex = false;
      duration = 0;		// VW default is 0
      bufSize = 10;		// VW default is 10
      avg = 6;			// VW default is 6
   
      while ((c = getopt (argc, (char**)argv, "hmoiqc:f:Tt:w:d:b:a:")) != EOF) {
         switch (c) {
            /* configuration file */
            case 'c':
               {
                  configfile = string (optarg);
                  break;
               }
            /* filename */
            case 'f':
               {
                  filename = string (optarg);
                  break;
               }
            /* trendfile */
            case 't':
               {
                  trendfile = string (optarg);
                  usetrend = usrtrend;
                  break;
               }
            /* default trendfile */
            case 'T':
               {
                  trendfile = monname;
                  usetrend = deftrend;
                  break;
               }
            /* html file */
            case 'w':
               {
                  HTMLfile = string (optarg);
                  break;
               }
            /* duration */
            case 'd':
               {
                  duration = atoi (optarg);
                  break;
               }
            /* buffer size */
            case 'b':
               {
                  bufSize = atoi (optarg);
                  break;
               }
            /* number of averages */
            case 'a':
               {
                  avg = atoi (optarg);
                  break;
               }
            /* is a monitor server */
            case 'm':
               {
                  ismonserv = false;
                  break;
               }
            /* suppress harmonics */
            case 'o':
               {
                  useharmonics = false;
                  break;
               }
            /* make index */
            case 'i':
               {
                  makeindex = true;
                  break;
               }
            /* quiet */
            case 'q':
               {
                  quiet = true;
                  break;
               }
            /* help */
            case 'h':
            case '?':
               {
                  errflag = true;
                  break;
               }
         }
      }
      /* help */
      if (errflag || (configfile.size() == 0) ||
         ((filename.size() == 0) && (trendfile.size() == 0))) {
         HelpMessage();		// Print help
         finish();
      }
   
      if(bufSize < 2) {
         cout << "Invalid buffer size" << endl; 
         finish();
         return;
      }
   
      if (quiet) {
         int i = ::open ("/dev/null", 2);
         (void) dup2(i, 1);
         (void) dup2(i, 2);   
      }
   
      cout << "new voltwriter" << endl;
      VWriter = new VoltWriter (usetrend != notrend? trendfile.c_str() : 
                           filename.c_str(), 
                           HTMLfile.c_str(), avg, avg * bufSize, useharmonics, 
                           usetrend, ismonserv ? this : 0);
      cout << "new voltwriter done" << endl;
      ID = -1;
      count = 0;
      fillTime = 0;
      oldTime = 0;
      whosTurn = 0;
   
      // All ID#s start at -1
      for(int id=0; id<maxChannels; id++) 
         IDnums[id]=-1;
      // Assigns # of channels to ARRAY_SIZE
      array_size = GetNChannels ();
   
      // Constructs and tests VWs
      MakeWatchers ();
      for(int vw = 0; vw < array_size; vw++)
         // Assigns each VW made to VWriter
         VWriter->addSlot(*array[vw]);
      // check if we just make the index
      if (makeindex) {
         VWriter->writeIndex();
         finish();
	 return;
      }
      // Makes a file w/ header for each GOOD VW it has
      VWriter->makeHeaders();
   
      Reset();
   }


   MultiVolt::~MultiVolt ()
   {
      for (int vw=0; vw < maxChannels; vw++) 
         delete array[vw];
   }


   void MultiVolt::Attention ()
   {
      MonServer::Attention();
   }


   void MultiVolt::ProcessData ()
   {
      // check count
      count++;
      if ((duration != 0) && (count > duration * bufSize * (u_long)avg)) {
         finish();
         return;
      }
      fillTime = getDacc().getCurrentTime().getS() - 1;
   
      // wait for minute break
      if (countdown < 0) {
         if (getDacc().getCurrentTime().getS() % 60 == 0) {
            countdown = bufSize;
         }
      }
      
      // check if continuous
      else if (fillTime - oldTime != 1 && oldTime != 0) {
         Reset();
      }
      
      // fill a buffer before starting with analysis
      else {
         // fill buffer
         oldTime = fillTime;
         for(int vw=0; vw < array_size; vw++)
            array[vw]->AppendBuffers();
      
         // analysis only after countdown (first buffer loaded)
         if (countdown > 0) {
            countdown--;
         }
         else {
            // spread analysis over multiple seconds
	    int spread = bufSize - 1;
            for (whosTurn = fillTime%spread; whosTurn<array_size; whosTurn+=spread) {
               ID = array[whosTurn]->GetID();
               array[whosTurn]->Analyze();
            
               if (Debug()) VWriter->print2Screen(ID);
               VWriter->print2File(ID);
               VWriter->print2Web();
            }   
         }
      }
   }


   void MultiVolt::Reset ()
   {
      oldTime=0;
      for(int vw=0; vw < array_size; vw++) 
         array[vw]->Reset();
      VWriter->reset();
      if (Debug()) VWriter->printHeader();    // Puts eye-catching header on screen
      countdown = -1;
   }


   void MultiVolt::HelpMessage() 
   {
      cout <<
         "usage: MultiVolt -c conf -f out [-w hfile] [-d dt] [-b len] [-a avg] [-moq]" << endl <<
         "       MultiVolt -c conf -t trend [-w hfile] [-d dt] [-b len] [-a avg] [-moiq]" << endl <<
         "       MultiVolt -c conf -T [-w hfile] [-d dt] [-b len] [-a avg] [-moiq]" << endl <<
         "         -c conf - configuration file" << endl <<
         "         -f out - ASCII output file, channels are stored in out.xx" << endl <<
         "         -t trend - trend file for output" << endl << 
	 "         -T - write trend to default output location" << endl <<
         "         -w hfile - web page for summary information" << endl <<
         "         -d dt - number of measurements (default 0 for running indefinitely)" << endl <<
         "         -b len - number of seconds per average (default 10)" << endl <<
         "         -a avg - number of averages per measurement (default 6)" << endl <<
         "         -m - no monitor server" << endl <<
         "         -o - suppress harmonics in output" << endl <<
         "         -i - make index of trend channels only" << endl <<
         "         -q - quiet" << endl;
   }

   int MultiVolt::GetNChannels () 
   {
      bool MAX=false;
      int cCount=0;					// # of channels ('H's) found
      ifstream getNums(configfile.c_str());			// Open configuration file
   
      while(!getNums.eof() && cCount<maxChannels)
      {
         getNums.ignore(9999, '#');			// Find ID number
         getNums >> IDnums[cCount];			// Put number in ID array
         cCount++;
         if(cCount==maxChannels) 
            MAX = true;
      }
   
      if(MAX==false) 
         cCount--;	// The last search adds 1 w/o finding anything
      cout <<cCount<<" channels read"<<(MAX==true?" (maximum!)\n":"\n");
      getNums.close();
   
      return cCount;
   }

   void MultiVolt::MakeWatchers ()
   {
   
      for(int vw=0, info=0; vw<array_size && info<maxChannels; vw++, info++)
      {
      
      // Test validity of the channel ID
      
         if(IDnums[info]<0 || IDnums[info] >= maxChannels) {
            cout <<"ERROR: "<<IDnums[info]<<" is an invalid channel ID#\n";
            array_size--; 
            vw--; 
            continue;
         }
      
      // Test for a duplicate of the channel ID
      
         bool inUse=false;
         for(int id=0; id<info; id++)
            if(IDnums[info]==IDnums[id]) 
               inUse = true;
      
         if(inUse == true) {
            cout <<"ERROR: ID# "<<IDnums[info]<<" is already in use\n";
            array_size--; 
            vw--; 
            continue;
         }
      
      // If the ID is ok, construct a VoltWatcher
      
         array[vw] = new VoltWatcher (getDacc(), IDnums[info], configfile.c_str(), bufSize);
      
      // If the VW has an error, deconstruct it and don't count it
      
         if(array[vw]->GetErrorStatus() == true) {
            delete array[vw]; 
            array[vw] = 0;
            cout << " (deleted)\n";
            array_size--; 
            vw--; 
            continue;
         }
      
      // If nothing went wrong, you will get here
      
      } // end of for() loop
   
      cout << array_size << " channels passed\n";
   
   } // end of function
