//  -*- C++ -*-
//    File: LockLoss.hh
// $Id: LockLoss.hh 6527 2011-11-19 04:05:17Z john.zweizig@LIGO.ORG $
#ifndef LOCKLOSS_HH
#define LOCKLOSS_HH

#include <iosfwd>
#include <iostream>
#include <fstream>
// #include <sstream> // bloody gcc-2.95.1 doesn't have sstream
#include <stdexcept>
#include <vector>
#include <deque>
#include <string>
#include <map>
#include <list>

#include "OperStateCondList.hh"
#include "DatEnv.hh"
#include "MonServer.hh"
#include "TrigRslt.hh"
#include "TrigClient.hh"
#include "Interval.hh"
#include "TSWindow.hh"
#include "Histogram1.hh"
#include "Trend.hh"
#include "AlarmClient.hh"
#include "AlarmData.hh"

typedef unsigned long ulong_t;

/** @name LockLoss Monitor
 *
 *  @memo {\tt LockLoss} monitors for lock acquisition and lock loss.
 * It relies on an
 * \URL[OperStateCondList]{../../../dmtlib/osc3/oscdoc/index.html}
 * object with conditions defined in the configuration file {\tt
 * LockLoss.conf} (see an example \URL[here]{../LockLoss.conf}).  The
 * configuration file defines the channels of interest, and
 * appropriate thresholds.
 *
 *  @version 2.1
 *  @author  David Chin <dwchin@umich.edu>; K. Riles <kriles@umich.edu>
 *   
 *
 */

using namespace std;

//
// The following templates are for computing statistics
//
template <class T> static double sum(const std::vector<T> &data)
{
    double retval = 0.;

    typename std::vector<T>::const_iterator iter = data.begin();
    for (; iter != data.end(); ++iter)
        retval += (double)*iter;

    return retval;
}


template <class T> static double mean(const std::vector<T> &data)
{
    return (sum(data) / (double)(data.size()));
}

template <class T> static double variance(const std::vector<T> &data)
{
    double avg = mean(data);
    std::vector<T> delta = data;
    typename std::vector<T>::iterator iter = delta.begin();
    for (; iter != delta.end(); ++iter)
        *iter -= avg;
    return (sum(delta) / (double)(data.size() - 1));
}

template <class T> static double stdDev(const std::vector<T> &data)
{
    return sqrt(variance(data));
}

//
// This class Summary is for Szabi's summary status page
//   We want a summary line every 1000 seconds.  The summary
//   line consists of:
//
//       GPS_time,state
//
//       where GPS_time is the GPS time of the latest update
//             state is 1 if IFO has >51% uptime for last 1000 secs
//                      0 if not
//                     -1 if LockLoss monitor was not running for
//                        that interval
//
//   We want to store 2 weeks worth => 1210 lines
//  
//
struct lockstatus_t {
    ulong_t gpstime;
    float   livefrac;   // fraction of time locked
};

class LockSummary
{
public:
    /// constructor
    LockSummary(void) {};
    
    void init(const std::string &htmldir,
              const Time &now,
              int dbg = 0);

    /// destructor
    ~LockSummary(void);

    // append data point -- 1 = locked >= 51%,
    //                      0 = locked < 51%,
    //                     -1 = no data
    void append(lockstatus_t state);

    /// dump data to file
    void               dumpList(void);

private:
    /// length of list
    static const ulong_t mLength = 1210;

    /// Summary "sampling" interval
    static const ulong_t mDT = 1000;
    
    /// list of lock status for last 2 weeks
    std::list<lockstatus_t> mLockList;

    /// status file name
    std::string        mFilename;

    /// status file stream
    std::fstream       mFile;

    /// time this object was created
    Time               mCreationTime;

    /// debug level
    int                mDebug;
};





//
//    LockLoss monitor class
//
class LockLoss
    : public DatEnv,
      public MonServer,
      private TrigClient
{
public:
    /// Constructor
    LockLoss(int argc, const char *argv[]);

    /// Destructor
    ~LockLoss();

    /** Main routine: processes data to determine if lock has been
     * acquired or lost, computes livetime ratio, generates triggers.
     */
    void ProcessData();

    /** Interrupt for DMT viewer
     */
    void Attention() { MonServer::Attention(); }

private:
    /// Debug level
    int               mDebug;

    /// Max no. of frames to process
    size_t            mMaxFrame;

    /// How many frames already processed
    size_t            mFrameCount;

    /// Predicates for acquisition and loss of lock
    bool              mXacquiredP;
    bool              mXlostP;
    bool              mYacquiredP;
    bool              mYlostP;
    bool              mBacquiredP;
    bool              mBlostP;
    bool              mTransitionP;

    /// Length of time in lock
    Interval          mTimeXInLock;
    Interval          mTimeYInLock;
    Interval          mTimeBothInLock;

    Interval          mTimeXInLock_10min;
    Interval          mTimeYInLock_10min;
    Interval          mTimeBothInLock_10min;

    Interval          mTimeXInLock_1hr;
    Interval          mTimeYInLock_1hr;
    Interval          mTimeBothInLock_1hr;

    Interval          mTimeXInLock_4hr;
    Interval          mTimeYInLock_4hr;
    Interval          mTimeBothInLock_4hr;

    Interval          mTimeXInLock_8hr;
    Interval          mTimeYInLock_8hr;
    Interval          mTimeBothInLock_8hr;

    Interval          mTimeXInLock_12hr;
    Interval          mTimeYInLock_12hr;
    Interval          mTimeBothInLock_12hr;

    /// Total time this monitor has been running
    Interval          mTotalRunTime;

    /// Ratio of mTimeInLock to mTotalRunTime
    double            mXLiveTimeRatio;
    double            mYLiveTimeRatio;
    double            mBothLiveTimeRatio;

    double            mXLiveTimeRatio_10min;
    double            mYLiveTimeRatio_10min;
    double            mBothLiveTimeRatio_10min;

    double            mXLiveTimeRatio_1hr;
    double            mYLiveTimeRatio_1hr;
    double            mBothLiveTimeRatio_1hr;

    double            mXLiveTimeRatio_4hr;
    double            mYLiveTimeRatio_4hr;
    double            mBothLiveTimeRatio_4hr;

    double            mXLiveTimeRatio_8hr;
    double            mYLiveTimeRatio_8hr;
    double            mBothLiveTimeRatio_8hr;

    double            mXLiveTimeRatio_12hr;
    double            mYLiveTimeRatio_12hr;
    double            mBothLiveTimeRatio_12hr;

    /// Print out HTML header
    void              htmlHead(void) const;

    /// Print out HTML footer
    void              htmlFoot(void) const;

    /// Print out top of HTML status page
    void              htmlTopOfPage(void);

    /// Print out N/A status for given time interval in HTML status page
    void              htmlStatusNA(const char *timeIntervalStr, const int colspan = 1) const;

    /// Print out status for given time interval in HTML status page
    void              htmlStatus(const char *timeIntervalStr, const int colspan = 1) const;

    /// Print unknown error message, and throw exception
    void              unknownError(const size_t ndata) const;

    /// Send a trigger to the Trigger Manager
    void              sendTrig(string trigname, string ifoname, Time now);

    /// List of Operational State Conditions to define when we are in lock
    OperStateCondList mOsclist;

    /// Name of logfile
    std::string   mLogfileName;
    
    /// Log file
    std::ofstream mLogfile;

    /// Log stream
    std::ostream *mLog;

    /// Name of livetime display file
    std::string   mDispfileName;

    /// Livetime display html file
    std::ostream *mDisplay;

    /// Name of index page
    std::string   mIndexPageName;

    /// Index page
    std::ofstream mIndexPage;

    /// Time stamp string
    char mTimestamp[64];

    /// Time stamp format
    std::string mTstampFormat;

    /// Time stride computing live time
    static const int mLiveTimeUpdateInterval = 60;

    /// Update interval for HTML livetime page, and livetime info (in seconds)
    static const int mHTMLupdateInterval = 60;

    /// Length of TSindows below
    static const size_t mMaxLength      = 7200;  // 2hrs @ 1sec stride
    static const size_t mMaxLength6hr   = 4320;  // 6hrs, @ 5sec stride
    static const size_t mMaxLength12hr5 = 7200;  // 12hrs, @ 6sec stride
    static const size_t mMaxLength10min = 600;   // 10min @ 1sec
    static const size_t mMaxLength1hr   = 3600;  // 1hr @ 1sec
    static const size_t mMaxLength4hr   = 14400/5; // 4hr @ 5sec
    static const size_t mMaxLength8hr   = 28800/5; // 8hr @ 5sec
    static const size_t mMaxLength12hr  = 43200/5; // 12hr @ 5sec
    static const size_t mMaxLength1min  = 60;      // 1min @ 1sec
    static const size_t mMaxLength1000s = 1000;    // 1000sec @ 1sec

    /// Name of IFO
    std::string mIFOname;

    /// Name of monitor
    std::string mName;

    /// Time series of lock state
    TSeries mXts;
    TSeries mYts;
    TSeries mBts;

    /// 6hr time series of lock state to line up with Ed Daw's
    /// seismic monitor
    TSeries mXts6hr;
    TSeries mYts6hr;
    TSeries mBts6hr;

    /// 12hr time series of lock state for shift summary
    TSeries mXts12hr;
    TSeries mYts12hr;
    TSeries mBts12hr;
    TSeries mBts12hrExp;

    //
    // Windows on the time series
    //

    // Cumulative window (updated every second)
    TSWindow<int> mXLock;
    TSWindow<int> mYLock;
    TSWindow<int> mBothLock;

    // 1 minute window (for trending only)
    TSWindow<int> mXLock1min;
    TSWindow<int> mYLock1min;
    TSWindow<int> mBothLock1min;

    TSWindow<int> mLockState;

    // The 6hr windows will be updated at 5 second intervals
    TSWindow<int> mXLock6hr;
    TSWindow<int> mYLock6hr;
    TSWindow<int> mBothLock6hr;

    // The 12hr windows will be updated at 6 second intervals
    TSWindow<int> mXLock12hr;
    TSWindow<int> mYLock12hr;
    TSWindow<int> mBothLock12hr;

    // The following two sets will be at 1 second intervals
    TSWindow<double> mTimeXInLock_10minTW;
    TSWindow<double> mTimeYInLock_10minTW;
    TSWindow<double> mTimeBothInLock_10minTW;

    TSWindow<double> mTimeXInLock_1hrTW;
    TSWindow<double> mTimeYInLock_1hrTW;
    TSWindow<double> mTimeBothInLock_1hrTW;

    // The following will be updated at 5 second intervals
    TSWindow<double> mTimeXInLock_4hrTW;
    TSWindow<double> mTimeYInLock_4hrTW;
    TSWindow<double> mTimeBothInLock_4hrTW;

    TSWindow<double> mTimeXInLock_8hrTW;
    TSWindow<double> mTimeYInLock_8hrTW;
    TSWindow<double> mTimeBothInLock_8hrTW;

    TSWindow<double> mTimeXInLock_12hrTW;
    TSWindow<double> mTimeYInLock_12hrTW;
    TSWindow<double> mTimeBothInLock_12hrTW;

    TSWindow<double> mTimeBothInLock_1000sTW;

    /// Array to store temporary livetime data
    std::vector<double>  mLiveTimeData;

    /// Histogram of lock stretches
    Histogram1 mXLockHist;
    Histogram1 mYLockHist;
    Histogram1 mBothLockHist;

    /// Length of current lock stretch (if in lock)
    Interval   mXTimeInLock;
    Interval   mYTimeInLock;
    Interval   mBothTimeInLock;

    /// Start times of lock stretches
    Time       mXLockStartTime;
    Time       mYLockStartTime;
    Time       mBothLockStartTime;

    /// Minute trends
    Trend      mTrend;

    // /// Vectors to contain data for trends
    // std::vector<int>    mXLockData;
    // std::vector<int>    mYLockData;
    // std::vector<int>    mBothLockData;
    // std::vector<int>    mLockStateData;

    /// Trend channel names
    map<std::string, std::string> mTrendChanNames;

    /// Lock summary data for Szabi's status web page
    LockSummary    mLockSumm;

    /// Directory for HTML files
    std::string    mHtmlDir;

    /// Alarm client -- send alarms when lock is lost
    AlarmClient    mAlarmClient;

    /// Locations of files where description of alarms are contained
    std::string    mLockLoss_alarmDesc;

    /// Alarm settings
    AlarmData      mLockLoss_ad;

    /// Alarm handle
    AlarmHandle    mLockLoss_ah;

    /// Actions to take when X-arm is in lock
    void XArmLockedActions(const Time &now);

    /// Actions to take when X-arm is not in lock
    void XArmNotLockedActions(const Time &now);

    /// Actions to take when Y-arm is in lock
    void YArmLockedActions(const Time &now);

    /// Actions to take when Y-arm is not in lock
    void YArmNotLockedActions(const Time &now);

    /// Actions to take when Both arms are in lock
    void BothArmsLockedActions(const Time &now);

    /// Actions to take when Both arms are not in lock
    void BothArmsNotLockedActions(const Time &now);

    /// Print out index page
    void writeIndexPage(void);

    /// Returns smaller of two objects
    template<class T> T min(T a, T b)
        {
            return (a < b) ? a : b;
        };

    /// Returns larger of two objects
    template<class T> T max(T a, T b)
        {
            return (a > b) ? a : b;
        };

    void setTSWindowStuff(void);

    void initTSWindows(const Time &now);

    void saveAllTSWindows(void);

    void restoreAllTSWindows(void);
};


#endif     //  LOCKLOSS_HH

