/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File BitTest.cc
//
//    Bit test monitor class. Test that all the low order bits of a 
//    data series are seen in both '1' and '0' states. Count the longest
//    sequence of repeated data for each channel.
//
#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "BitTest.hh"
#include "Dacc.hh"
#include "Time.hh"
#include "TrigRslt.hh"
#include "framecpp/FrameH.hh"
#include "html/document.hh"
#include "html/table.hh"
#include "html/text.hh"
#include "html/color.hh"
#include "html/writer.hh"
#include "ParseLine.hh"

using namespace std;

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/BitTest/BitTest.cc 6619 2012-03-12 23:58:12Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Channel Data Monitor"
#include "ProcIdent.hh"

//======================================  Generate the main function.
EXECDMT(BitTest)

//======================================  Least significant set bit
inline int
lssb(int x) {
    return (x & -x);
}

//======================================  Constructor
const int daysec = 3600*24;
BitTest::BitTest(int argc, const char *argv[])
  : DMTBase(argc, argv), mAlarm("BitTest"), mLastProcessed(0)
{
    char* cptr;

    //----------------------------------  Parse the arguments
    mConfig  = "BitTest.conf";
    mOutput  = "BitTest.junk";
    mReset   = 0;
    mGenTrig = false;
    mSynch   = 0;
    for (int i=1 ; i<argc ; i++) {
      string argi = argv[i];
        if (argi == "cfile") {
	    mConfig = argv[++i];
        } else if (argi == "ofile") {
	    mOutput = argv[++i];
        } else if (argi == "reset") {
	    mReset = strtol(argv[++i], 0, 0);
        } else if (argi == "+trigger" || argi == "+trig") {
	    mGenTrig = true;
        } else if (argi == "synch") {
	    mSynch = int(getUTC(Now())/daysec)*daysec;   // start of UTC day
	    mSynch += strtol(argv[++i], &cptr, 0)*3600;
	    if (*cptr == ':') {
	        mSynch += strtol(cptr+1, &cptr, 0)*60;
		if (*cptr == ':') mSynch += strtol(cptr+1, &cptr, 0);
	    }
	    if (*cptr) {
	        cerr << "Error in synch specifier: " << argv[i] << endl;
		cerr << "Syntax: " << argv[0] << " ... synch hh[:mm[:ss]]" 
		     << endl;
		finish();
		return;
	    }
	} else if (argi == "-toc") {
	    getDacc().setTOCMode(false);
	} else if (isEnvArg(argi.c_str())) {
	    i++;
	} else {
	    cerr << "Invalid argument: " << argi << endl;
	    cerr << "Command syntax is :" << endl;
	    cerr << *argv << " [cfile <config>] [reset <time>] "
		 << "[ofile <ouput>] [synch <print-time>]" << endl;
	    cerr << "        [+trig[ger]] [-toc]" << endl;
	    finish();
	    return;
	}
    }
    getDacc().setIgnoreMissingChannel(true);

    //----------------------------------  Read the configuration file
    if (ReadConfig()) {
        finish();
	return;
    }

    //----------------------------------  Define a few alarms
    const char* stuckDesc="Channel $1 has bits $2 stuck on, bits $3 stuck off";
    const char* webFile="Monitors/BitTest/index.html#Alarms";
    AlarmData alm_stuck("BitTest", "Bit_is_Stuck", Interval(mReset+10.0), 
			4, stuckDesc);
    alm_stuck.setWebFile(webFile);
    alm_stuck.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(alm_stuck);

    const char* ovfloDesc="Channel $1 has $2 data overflows";
    AlarmData alm_ovflo("BitTest", "Overflow", Interval(mReset+10.0), 4, 
			ovfloDesc);
    alm_ovflo.setWebFile(webFile);
    alm_ovflo.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(alm_ovflo);

    const char* roErrDesc="Channel $1 has a $2 readout errors";
    AlarmData alm_roErr("BitTest", "ReadOutError", Interval(mReset+10.0), 
			7, roErrDesc);
    alm_roErr.setWebFile(webFile);
    alm_roErr.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(alm_roErr);

    //----------------------------------  Starting status printout.
    if (Debug()) {
        bool phdr = true;
	for (channel_iter itr=mChannel.begin() ; itr!=mChannel.end() ; itr++){
	    itr->Status(cout, phdr);
	    phdr = false;
	}
    }
    mProcessed = 0;
    if (mSynch && mReset) {
        Time::ulong_t now = getUTC(Now());
        if (now > mSynch) mSynch += int((now - mSynch)/mReset)*mReset;
    }
}

//======================================  Read in the configuration
bool
BitTest::ReadConfig(void) {

    //----------------------------------  Open the configuration file
    ParseLine fd(mConfig.c_str());
    if (!fd.isOpen()) {
        cerr << "Unable to open BitTest configuration file: " 
		  << mConfig << endl;
        finish();
        return true;
    }

    //----------------------------------  Read in the configuration file
    while (fd.getLine() > 0) {
        const char*        name = fd.getArg(0);
	ChanBit::mask_t    mask = 0;
	if (fd.getCount() > 1) mask = fd.getHex(1);
	ChanBit::counter_t rMax = 0;
	if (fd.getCount() > 2) rMax = fd.getInt(2);
	if (rMax < 0) rMax = 0;
        mChannel.push_back(ChanBit(name, mask, rMax));
    }
    return false;
}

//======================================  Process one frame (FrameCPP).
void
BitTest::ProcessFrame(DMTBase::frame_ptr_type frame)
{
    Time tFrame(frame->GetGTime().getSec(), frame->GetGTime().getNSec());

    //----------------------------------  Allocate local data storage
    for (channel_iter itr=mChannel.begin() ; itr!=mChannel.end() ; itr++) {
        const char* ChName = itr->getChannel();
	channel_iter tent = itr;

        //------------------------------  Get the adc pointer.
	if (Debug() > 2) cout << "Look for channel: " << ChName << endl;
	fradcdata_pointer adc;
	int order = getDacc().findAdcOrdered(ChName, adc);
	if (order < 0 && itr != mChannel.begin()) {
	    channel_iter save = tent;
	    --itr;
	    if (Debug() > 2) cout << "Reorder channels: " 
				  << itr->getChannel() << " <-> "
				  << save->getChannel() << endl;
	    tent = mChannel.insert(itr, *save);
	    mChannel.erase(save);
	}

	if (!order) {
	    if (Debug()) cout << "Channel: " << ChName 
			      << " not found." << endl;
	    continue;
        }

	if (adc->GetDataValid()) {
	    if (Debug()) {
	        cout << "Channel: " << ChName << " Status: " 
		     << adc->GetDataValid() << endl;
	    }
	    itr->incReadOutError();
	    continue;
	}

	if (adc->RefData().size() == 0) continue;
        frvect_pointer vect(adc->RefData().front());
	if (!vect) continue;

	vect->Uncompress();
	ChanBit::uint_t nw = vect->GetNData();
	if (nw <= 0) continue;

#ifndef FCPP_SHARED_PTRS
        const void* data =vect->GetData();
#else
        const void* data =vect->GetData().get();
#endif
	if (!data) continue;

	//------------------------------  Process depending on type
	int vtype = vect->GetType();
        switch (vtype) {
	case FrameCPP::FrVect::FR_VECT_2S:
	case FrameCPP::FrVect::FR_VECT_2U:
	    tent->setDebug(Debug());
	    tent->Scan(nw, reinterpret_cast<const short*>(data));
	    break;
	case FrameCPP::FrVect::FR_VECT_4R:
	    tent->setDebug(Debug());
	    tent->Scan(nw, reinterpret_cast<const float*>(data));
	    break;
	default:
	    break;
	}
    }

    //--------------------------------  Check results every mReset seconds
    Interval dT = frame->GetDt();
    mProcessed += long(dT);
    mLastProcessed = tFrame + dT;
    Time::ulong_t now = getUTC(mLastProcessed);
    if (!mReset) {
        return;
    } else if (mSynch) {
        if (now >= mSynch + mReset) mSynch += int((now-mSynch)/mReset)*mReset;
	else                        return;
    } else if (mProcessed < mReset) {
        return;
    }

    //----------------------------------  Generate alarms
    lmsg::error_t rc = 0;
    AlarmHandle han;
    for (channel_iter itr=mChannel.begin() ; itr!=mChannel.end() ; itr++){
        if (itr->getNReadOutError()) {
	    ostringstream param;
	    param << itr->getChannel() << " " << itr->getNReadOutError();
	    AlarmData al("BitTest", "ReadOutError", 0, 6, "", param.str());
	    rc = mAlarm.setAlarm(al, han);
	} else if (itr->BitError()) {
	    ostringstream param;
	    ChanBit::mask_t hi  = ~itr->getHiBits();
	    ChanBit::mask_t On  =  itr->getStuckOn()  & hi;
	    ChanBit::mask_t Off =  itr->getStuckOff() & hi;
	    param << itr->getChannel() 
		  << " 0x" << hex << setw(4) << setfill('0') << On
		  << " 0x" << hex << setw(4) << setfill('0') << Off;
	    AlarmData al("BitTest", "Bit_is_Stuck", 0, 4, "", param.str());
	    rc = mAlarm.setAlarm(al, han);
	} else if (itr->OverflowError()) {
	    ostringstream param;
	    param << itr->getChannel() << " " << itr->getNOverFlow();
	    AlarmData al("BitTest", "Overflow", 0, 4, "", param.str());
	    rc = mAlarm.setAlarm(al, han);
	}
	if (rc) cout << "[BitTest] Error sending alarm: " << rc << endl;
    }

    //----------------------------------  Optionally generate a trigger
    if (mGenTrig) {
        for (channel_iter itr=mChannel.begin() ; itr!=mChannel.end() ; itr++){
	    if (itr->Error()) {
	        trig::TrigRslt t("BitTest", itr->getChannel(), 6);
		t[0] = itr->getNWords();
		t[1] = itr->getLongRpt();
		t[2] = itr->getNReadOutError();
		t[3] = itr->getStuckOn();
		t[4] = itr->getStuckOff();
		t[5] = itr->getNOverFlow();
		t.setTime(tFrame-Interval(mProcessed));
		t.setDuration(Interval(mProcessed));
		t.setDisposition(trig::d_metaDB | trig::d_alarm);
		t.setPriority(trig::p_warn);
		t.setIfos(string(itr->getChannel()).substr(0,2).c_str());
		sendTrigger(t);
	    }
	}
    }

    //----------------------------------  Generate an html report
    Report();

    //----------------------------------  Reset the counters.
    Reset();
}

//======================================  Global reset
void 
BitTest::Reset(void) {
    for (channel_iter itr=mChannel.begin() ; itr!=mChannel.end() ; itr++) {
        itr->Reset();
    }
    mProcessed = 0;
}


//======================================  Attention handler.
void 
BitTest::Attention(void) {
    Report();
    mChannel.erase(mChannel.begin(), mChannel.end());
    ReadConfig();
    mProcessed = 0;
}

//======================================  Write a status report
void
BitTest::Report(void) {
    char TimeStamp[64];
    gethostname(TimeStamp, sizeof(TimeStamp));
    string conFile = string(TimeStamp) + ":" + mConfig;
    LocalStr(mLastProcessed, TimeStamp, "%M %d, %Y %H:%N:%S");

    //----------------------------------  Print out Error Channels
    string File = mOutput + ".Errors";
    ofstream fout(File.c_str(), ios::out);
    if (!fout.good()) return;
    fout << "Report produced for " << mProcessed << " seconds to "  
	 << TimeStamp << endl;
    fout << "Configuration file: " << conFile << endl << endl;
    fout << "=============================   Channels with errors" 
	 << endl << endl;
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        if (!itr->getNFrames()) continue;
	ChanBit::mask_t HBits = itr->getHiBits();
	if (itr->getLongRpt()+1 >= itr->getNWords()) continue;
        if (itr->BitError() || (itr->RepeatError() && !itr->getNOverFlow())) {
	    fout << "Error(s) in channel: " << itr->getChannel() << endl;
	    ChanBit::mask_t mask = itr->getStuckOn() & ~HBits;
	    if (mask) fout << "   Bits stuck On:  " << hex << setw(4) 
			   << setfill('0') << mask << dec << setfill(' ')
			   << setw(0) << endl;
	    mask = itr->getStuckOff() & ~HBits;
	    if (mask) fout << "   Bits stuck Off: " << hex << setw(4) 
			   << setfill('0') << mask << dec << setfill(' ')
			   << setw(0) << endl;
	    // mask = itr->getStuckAdj() & ~HBits;
	    // if (mask) fout << "   Adjacent bits:  " << hex << setw(4) 
	    //		      << setfill('0') << mask << dec << setfill(' ')
	    //		      << setw(0) << endl;
	    if (itr->RepeatError()) 
	        fout << "   Longest repeat group (" << itr->getLongRpt() 
		     << ") exceeds limit (" << itr->getMaxRpt() << ")"
		     << endl;
	    fout << endl;
        }
    }

    //----------------------------------  Print stuck channels
    fout << "=============================   Channels with all one value"
	 << endl << endl;
    ChanBit::counter_t count = 0;
    fout.setf(ios::left);
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        if (itr->getNFrames() && (itr->getMinimum() == itr->getMaximum())) {
	    fout << "  " << setw(24) << itr->getChannel() << setw(0);
	    if (!(++count % 3)) fout << endl;
        }
    }
    fout << endl;
    fout.setf(ios::right);

    //----------------------------------  Print Channel with Error flags
    fout << "=============================   Channels with readout errors "
	 << "reported by DAQ" << endl << endl;
    count = 0;
    fout.setf(ios::left);
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        if (itr->getNReadOutError()) {
	    fout << "  " << setw(24) << itr->getChannel() << setw(0);
	    if (!(++count % 3)) fout << endl;
        }
    }
    fout << endl;
    fout.setf(ios::right);

    //----------------------------------  Print Channel with Over/Underflow
    fout << "=============================   Channels with Overflow"
	 << endl << endl;
    fout.setf(ios::left);
    count = 0;
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        if (itr->getNOverFlow()) {
	    fout << "  " << setw(24) << itr->getChannel() << setw(0);
	    if (!(++count % 3)) fout << endl;
        }
    }
    fout << endl;
    fout.setf(ios::right);

    //----------------------------------  Print Channel not read out
    fout << "=============================   Channels not read out"
	 << endl << endl;
    fout.setf(ios::left);
    count = 0;
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        if (!itr->getNFrames()) {
	    fout << "  " << setw(24) << itr->getChannel() << setw(0);
	    if (!(++count % 3)) fout << endl;
        }
    }
    fout << endl;
    fout.close();
    fout.setf(ios::right);

    //----------------------------------  Print out Channel Statistics
    File  = mOutput + ".Statistics";
    fout.open(File.c_str(), ios::out);
    if (!fout.good()) return;
    fout << "Report produced for " << mProcessed << " seconds to " 
	 << TimeStamp << endl;
    fout << "Configuration file: " << conFile << endl << endl;
    bool phdr = true;
    fout << "=============================   Channel Statistics"
	 << endl << endl;
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        itr->Print(fout, phdr);
	phdr = false;
    }
    fout.close();

    //----------------------------------  Send Channel statistics to html
    html::document doc;
    doc.setBackgroundColor(html::color("white"));
    html::text* txt = dynamic_cast<html::text*>(&doc.addObject(html::text()));
    *txt << "Report produced for " << mProcessed << " seconds to " 
	 << TimeStamp;
    doc.lineBreak();
    doc.addObject(html::text(string("Configuration file name: ")+conFile));
    doc.lineBreak();
    html::table* tab = dynamic_cast<html::table*>(&doc.addObject(html::table()));
    tab->setBorder();
    tab->addColumn("Flag");
    tab->addColumn("Channel");
    tab->addColumn("Frames");
    tab->addColumn("On");
    tab->addColumn("Off");
    tab->addColumn("Repeat");
    tab->addColumn("Average");
    tab->addColumn("Sigma");
    tab->addColumn("Minimum");
    tab->addColumn("Maximum");
    for (channel_iter itr=mChannel.begin() ; itr != mChannel.end() ; itr++) {
        int Row = tab->addRow();
	if (itr->Error()) tab->insertData(Row, 0, "****");
	else              tab->insertData(Row, 0, "&nbsp;");
	tab->insertData(Row, 1, itr->getChannel());
	tab->insertData(Row, 2, long(itr->getNFrames()));
	ChanBit::mask_t hi = ~(itr->getHiBits());
	tab->insertData(Row, 3, long(itr->getStuckOn()  & hi), 4, 16);
	tab->insertData(Row, 4, long(itr->getStuckOff() & hi), 4, 16);
	tab->insertData(Row, 5, long(itr->getLongRpt()));
	tab->insertData(Row, 6, itr->getAverage());
	tab->insertData(Row, 7, itr->getSigma());
	tab->insertData(Row, 8, itr->getMinimum());
	tab->insertData(Row, 9, itr->getMaximum());
    }
    File  = mOutput + ".html";
    fout.open(File.c_str(), ios::out);
    if (!fout.good()) return;
    html::writer hw(fout);
    doc.write(hw);
    fout.close();
}

//======================================  Print final Results
BitTest::~BitTest() {
    if (getNFrames()) {
        bool phdr = true;
	for (channel_iter it=mChannel.begin() ; it!=mChannel.end() ; it++) {
	    it->Print(cout, phdr);
	    phdr = false;
	}
    }
    mChannel.erase(mChannel.begin(), mChannel.end());
}

//======================================  Channel constructor
ChanBit::ChanBit(const char *name, mask_t mask, counter_t repeat) {
    mChanName = name;            // Channel name
    mMaxRpt  = repeat;          // decimation factor
    mSelect  = mask;            // bit ignore mask
    mDBits   = (1 << 16) - 1;   // Data mask
    if (mSelect) mDBits &= mSelect;
    setDebug(0);                // set the debug level.
    Reset();                    // Reset all counters, et.
}

//======================================  Reset temporary channels.
void
ChanBit::Reset(void) {
    mNWords    = 0.0;
    mAverage   = 0.0;
    mSigma     = 0.0;
    mMaximum   = 0.0;
    mMinimum   = 0.0;
    mNFrames   = 0;
    mStuckOff  = 0;
    mStuckOn   = ~0;
    for (counter_t i=0 ; i<4 ; i++) mStuckAdj[i] = ~0;
    mValue     = ~0;
    mFValue    = 0.0;
    mRepeat    = 0;
    mLongR     = 0;
    mInteger   = false;
    mOverFlow  = 0;
    mReadOutErr = 0;
}

//======================================  Channel destructor
ChanBit::~ChanBit() {
}

bool
ChanBit::BitError(void) const {
    int stuck = mStuckOn | ~mStuckOff;
    if (mSelect) return (stuck & mDBits) != 0;
    return (((::lssb(stuck) + stuck) & mDBits) != 0);
}

bool
ChanBit::RepeatError(void) const {
    return mMaxRpt && (mLongR > mMaxRpt);
}

bool
ChanBit::Error(void) const {
    return (getNReadOutError() != 0 || getNOverFlow() != 0 || 
	    mMinimum == mMaximum || RepeatError() || BitError());
}

//======================================  Print the results
void
ChanBit::Print(ostream& ostr, bool phdr) const {
    const char* flag;

    if (phdr) { 
        ostr << "Flag Channel                         Frames  On   Off " 
	     << " Repeat   Avg      Sigma    Minimum   Maximum" << endl;
    }

    if (Error()) flag = "****";
    else         flag = "    ";
    if (mInteger) {

        ostr << flag << " ";
	ostr.setf(ios::left);
	ostr << setw(31) << mChanName.c_str();
	ostr.setf(ios::right);
	ostr << setw(7) << mNFrames << " "
	     << hex << setfill('0') << setw(4) << getStuckOn()  << " "
	     << hex << setfill('0') << setw(4) << getStuckOff() << " " 
	     << dec << setfill(' ') << setw(6) << mLongR << " "
	     << setw(9) << getAverage() << " "
	     << setw(9) << getSigma()   << " " 
	     << setw(9) << getMinimum() << " "
	     << setw(9) << getMaximum() << endl;
    } else {
        ostr << flag << " ";
	ostr.setf(ios::left);
	ostr << setw(31) << mChanName.c_str() << setw(0);
	ostr.setf(ios::right);
	ostr << setw(7) << mNFrames << setw(0)
	     << " .... .... "
	     << setw(6) << mLongR << " "
	     << setw(9) << getAverage() << " "
	     << setw(9)<< getSigma()    << " " 
	     << setw(9)<< getMinimum()  << " "
	     << setw(9)<< getMaximum()  << endl;
    }
}

//======================================  Get high-bit mask
ChanBit::mask_t
ChanBit::getHiBits(void) const {
     mask_t hmask, lmask;
     mask_t mStuck = getStuckOn() | getStuckOff();
     for (hmask=1 ; hmask <= mDBits ; hmask += hmask);
     for (lmask = hmask >> 1 ; (lmask & mStuck) ; lmask>>=1);
     if (!lmask) return hmask-1;
     return      hmask - (lmask<<1);
}

//======================================  Print the status
void
ChanBit::Status(ostream& ostr, bool phdr) const {
    if (phdr) { 
        ostr << "Channel                         Max Rpt D.Bits  Frames "
	     << " On   Off  Long" << endl;
    }

    ostr.setf(ios::left);
    ostr << setw(31) << mChanName.c_str() << setw(0);
    ostr.setf(ios::right);
    ostr << setw(8) << mMaxRpt 
	 << hex << setfill('0') << setw(4) << mDBits << dec << setfill(' ')
	 << setw(8) << mNFrames 
	 << hex << setfill('0') << setw(4) << getStuckOn()
	 << getStuckOff()  << dec << setfill(' ') 
	 << setw(8) << mLongR << setw(0) << endl;
}

//======================================  Check everything
void
ChanBit::Scan(uint_t nw, const short *data) {

    //----------------------------------  Check the arguments
    if (nw <= 0 || !data) return;

    //----------------------------------  Loop over all words
    int Val(mValue);
    mask_t on(~0), off(0), Rpt(mRepeat);
    // mask_t Adj0(~0), Adj1(~0), Adj2(~0), Adj3(~0);
    double Avg(0.0), Sig(0.0);
    double Min = data[0];
    double Max = data[0];
    for (uint_t i=0 ; i<nw ; i++) {
        int test = *data++;
	double Stat = test;
	off |= test;
	on  &= test;
	// mask_t rtest = test >> 1;
	// Adj0 &=  test |  rtest;
	// Adj1 &= ~test |  rtest;
	// Adj2 &=  test | ~rtest;
	// Adj3 &= ~test | ~rtest;
	Avg += Stat;
	Sig += Stat*Stat;
	if (Stat < Min) Min = Stat;
	if (Stat > Max) Max = Stat;

	if (Val != test) {
	    Val = test;
	    Rpt = 0;
	} else if (Rpt++ == mLongR) {
	    mLongR = Rpt;
	}
    }
    mAverage += Avg;
    mSigma   += Sig;
    if (Min == -32768) mOverFlow++;
    if (Max ==  32767) mOverFlow++;
    if (mNWords == 0) {
        mMinimum = Min;
	mMaximum = Max;
    } else {
        if (Min < mMinimum) mMinimum = Min;
	if (Max > mMaximum) mMaximum = Max;
    }
    mNWords += nw;
    mValue    = Val;
    mRepeat   = Rpt;

    //---------------------------------  Update global status
    mStuckOn   &= on;
    mStuckOff  |= off;
    // mStuckAdj[0] &= Adj0;
    // mStuckAdj[1] &= Adj1;
    // mStuckAdj[2] &= Adj2;
    // mStuckAdj[3] &= Adj3;
    mNFrames++;
    mInteger = true;
}

//======================================  Check everything
void
ChanBit::Scan(uint_t nw, const float *data) {

    //----------------------------------  Check the arguments
    if (nw <= 0 || !data) return;

    //----------------------------------  Loop over all words
    double Val(mFValue);
    counter_t Rpt(mRepeat);
    double Avg(0.0), Sig(0.0);
    double Min = data[0];
    double Max = data[0];
    for (uint_t i=0 ; i<nw ; i++) {
	double Stat = *data++;
	Avg += Stat;
	Sig += Stat*Stat;
	if (Stat < Min) Min = Stat;
	if (Stat > Max) Max = Stat;
	if (Val != Stat) {
	    Val = Stat;
	    Rpt = 0;
	} else if (Rpt++ == mLongR) {
	    mLongR = Rpt;
	}
    }
    mAverage += Avg;
    mSigma   += Sig;
    if (mNWords == 0) {
        mMinimum = Min;
	mMaximum = Max;
    } else {
        if (Min < mMinimum) mMinimum = Min;
	if (Max > mMaximum) mMaximum = Max;
    }
    mNWords += nw;
    mFValue   = Val;
    mRepeat   = Rpt;
    mNFrames++;
    mStuckOn  =  0;
    mStuckOff = ~0;
}

//======================================  Adjacent stuck bits
ChanBit::mask_t
ChanBit::getStuckAdj(void)  const {
    if (!mInteger) return 0;
    mask_t On   = getStuckOn();
    mask_t Off  = getStuckOff();
    mask_t On1  = On >> 1;
    mask_t Off1 = Off >> 1;
    return ((mStuckAdj[0] & ~On  & ~On1 ) |
            (mStuckAdj[1] & ~Off & ~On1 ) |
            (mStuckAdj[2] & ~On  & ~Off1) |
            (mStuckAdj[3] & ~Off & ~Off1)) & mDBits;
}






