//
//  BicoViewer -- A GUI -based application to display the bicoherence 
//                and other related second order statistical quantities.
//  					
//             	This program uses both the ROOT and DMT libraries.
//             	
//             	For more information:  
//             	http://marble.ligo-wa.caltech.edu/dmt/Monitors/BicoMon/
//             	
//             	Written by Steve Penn
//
//    ------------------------------------------------------------------------
//
//  NOTES:
//  ------
//   - If the channels are of different rates, they are decimated down to the 
//        slowest rate. 
//        
//   - The Bicoherence is displayed as a 2D plot in frequency space.  
//     In general the bicoherence is shown in the full XY plane of 
//     (-f_Nyq, f_Nyq) in each direction, though only half the plane is 
//     unique. In the case of the autobicoherence (1 input channel) the 
//     bicoherence has a unique area entirely in the first quadrant 
//     (a triangle with vertices (0,0), (1,0), and (2/3, 2/3), where 
//     f_Nyquist = 1).  Currently, to save time, this monitor only displays the first 
//     quandrant. Eventially, I will have a cross-bicoherence (2-3 inout 
//     channels) display all quandrants to allow a better intuitive
//     understanding of the results.
//     
//
//  REQUEST:
//  --------
//      This code is still in development.  PLEASE send any suggestions, 
//      comments, questions, or bug notices to Steve Penn (penn@hws.edu).  
//      
//      
//  VERSION HISTORY:
//  ----------------
//  Earlier versions of this code were named BicoMon.  BicoMon is now the name for 
//  the background bicoherence monitor.  I am restarting the code version number
//  with the renaming of the foregroung viewer to BicoViewer.
//  
//  Old BicoMon History
//  -------------------
//  Version 0.1 -- 6 June 2002 -- Steve Penn
//    - First GUI release.  GUI version melds the XBic code from XBS7 with a
//        GUI test monitor code by Daniel Sigg.  Many thanks to Daniel for all
//        his assistance in getting this code into production.
//        
//    - For development information on XBic look in the notes of that code.
//    
//    
//  Version 0.2 -- Summer Solstice Edition -- Steve Penn
//     - Changed program so that the threads pass arrays of data rather than  
//          ROOT histograms.  Daniel advises that ROOT is not thread safe.
//	    Hence the change.
//          
//     - Changed input to allow for greater frequency range and resolution.  I 
//          will soon switch the input to freq. max. and freq. resolution.
//          This is more intuitive for the users.
//          
//     - Found a bug in the Bispectrum routine that initialized the BS between 
//          each slice rather than between calculations.
//          
//     - Corrected error which used x-y symmetry even when performing a  
//          cross-bispectrum, rather than just for the auto-bispectrum.
//          
//     - Histograms now created and filled in BVWindow::DrawPlots
//     
//     - Expanded BVWindow::ReadConf to calculate most parameters used in 
//          calculation. Thus BicoMon::SetParameters will be eliminated soon.
//
//
//  Version 0.3b
//    - Make new window 1130 x 750
//    - New user interface parameters
//       - Frequency Max
//       - Frequency resolution
//       - Averaging/Uncertainty
//       - Overlap in 1D FFT's
//    - Added to display
//       - Bispectrum
//       - PSD's of all channels
//    - Corrected assignment of f1+f2 channel: changed jxy=jx+jy to jxy=ix+iy+1
//    - 
//
//  Version 0.3b8  August 2003
//    - Updated FFTW to version 3.0
//
//
//  BicoViewer History
//  ------------------
//  Version 0.1 -- March 2004 -- Steve Penn
//  	- Changed the Bicoherence calculation so that it uses the BicoDir class
//		- I don't yet understand the need for pointer declaration vs. direct 
//		   declaration in the BVWindow and BicoViewer.
//		   
//	 Version 0.8 -- October 2005 
//	   - Lots of changes and updates in intervening versions.  I need to 
//	     start using CVS.
//	   
//
//---------------------------------------------------------------------------- 
//


#include "BicoViewer.hh"

// SET CONSTANTS
// -------------

// Monitor name
const char* const kMonName = "BicoViewer";

// Heartbeat rate
const Long_t kHeartbeatInterval = 100; // 100 ms

// Minimum Number of Frequency Bins
const Int_t kFreqBinMin = 64 ;

// Default Number of Frequency Bins
const Int_t kFreqBinDef = 256 ;

// Default Span
const Int_t kTimeSpanDef = 64 ;

// Default Frequency Resolution
const Int_t kFreqResDef = 1 ;


// Widget IDs
// ----------
 
const Int_t kButtonStart  = 1;  // Start button
const Int_t kButtonPause  = 2;  // Pause button
// const Int_t kButtonResume = 3;  // Resume button 
const Int_t kButtonQuit   = 3;  // Quit button 

const Int_t kChanSel1 = 10;     // Three Channel selectors
const Int_t kChanSel2 = 11;
const Int_t kChanSel3 = 12; 

const Int_t kFreqMax = 21;       // Max Freq Selector
const Int_t kFreqRes = 22;       // Freq Res Selector
const Int_t kTimeSpan = 23;     		// Span Selector
const Int_t kOverlap = 24;     	// Overlap Selector
const Int_t kWindowTypeBox = 25;     // Window Selector

const Int_t kZScale = 30;       // Z Scale Selector
const Int_t kXLimitMin = 31;    // X Min TextBox
const Int_t kXLimitMax = 32;    // X Max TextBox
const Int_t kYLimitMin = 33;    // Y Min TextBox
const Int_t kYLimitMax = 34;    // Y Max TextBox

const Int_t kWriteBico = 40;    // Write Bicoherence Plot CheckBox
const Int_t kPlotFileFormat = 42;  // Plot File Format Selector
const Int_t kButtonWriteNow = 43;  // Write Current Plot Button


// Widget Labels
// -------------
 
// Control Button Text
const char* const kControlButtonText[] = 
  {"   &Start   ", "   &Pause   ", 
   "   &Quit    ", "   &Resume  ", "  Re&Start  ", };
const char* const kControlButtonHelp[] = 
  {"Start/Restart  with New Parameters",
   "Pause/Resume Calculation", 
   "Quit Program"};

// WriteNow Button Text
const char* const kButtonWriteNowText = "  &Write Current Plot  "; 
const char* const kButtonWriteNowHelp = "Write Current Plot to GIF File";

// Window Check Box Label
const char* const kUseWindowText = "2D Window"; 

// Write Bicoherence Check Box Label
const char* const kWriteBicoText = "Save Bicoherence as"; 

// Frequency Max Entries
const Int_t kFreqMaxSize = 18;
const char* const kFreqMaxText[] = {
  "F_Nyq",
  "8192 ",
  "4096 ",
  "2048 ",
  "1024 ",
  "512  ",
  "256  ",
  "128  ",
  "64   ",
  "32   ",
  "16   ",
  "8    ",
  "4    ",
  "2    ",
  "1    ",
  "0.5  ",
  "0.25 ",
  "0.125"
};


// Frequency Resolution Entries
const Int_t kFreqResSize = 18;
const char* const kFreqResText[] = {
  "Default",
  "64.0   ",  
  "32.0   ",  
  "16.0   ", 
  " 8.0   ",  
  " 4.0   ",  
  " 2.0   ",  
  " 1.0   ",  
  " 0.5   ",  
  " 0.25  ",  
  " 0.125 ",  
  " 0.063 ",  
  " 0.031 ",  
  " 0.016 ",  
  " 0.008 ",  
  " 0.004 ",  
  " 0.002 ",  
  " 0.001 "
};


// Time Span Entries
const Int_t kTimeSpanSize = 12;
const char* const kTimeSpanText[] = {
  "Default",
  "     1 ",
  "     2 ",
  "     4 ",
  "     8 ",
  "    16 ",
  "    32 ",
  "    64 ",
  "   128 ",
  "   256 ",
  "   512 ",
  "  1024 "
};


// Percent Overlap Entries
const Int_t kOverlapSize = 8;
const char* const kOverlapText[] = {
  "Default",
  "   0%  ",
  "  10%  ",
  "  20%  ",
  "  25%  ",
  "  30%  ",
  "  40%  ",
  "  50%  "
};


// Window Method Entries
const Int_t kWindowTypeSize = 4;
const char* const kWindowTypeText[] = {
  "Default",
  "1D Time",
  "2D Time",
  "2D Freq"
};


const char* const kPlottingMessage = "Plotting ...";



// Create Main Program
// -------------------

EXEROOT (BicoViewer);


using namespace std;
using namespace ligogui;

// #ifdef BV_DEBUG
//    std::cout << " Create a ROOT executable ..."<<endl;
// #endif



// *************************************************
// *                                               *
// *   ---  BVOutConf: Output Configuration  ---   *
// *                                               *
// *************************************************

BVOutConf::BVOutConf () 
  : fWriteBico (false), fUseGIFformat (true), fWriteNow (false) 
{
  //    fLimitGIF = 1.0;
}


BVOutConf::~BVOutConf ()   
{
}




// ************************************************
// *                                              *
// *   ---  BVPlotConf: Plot Configuration  ---   *
// *                                              *
// ************************************************

BVPlotConf::BVPlotConf () 
  : fZLogScale (false)
{
  fXLimit[0] = 0.0;    // Don't override the Default
  fXLimit[1] = 0.0; 
  fYLimit[0] = 0.0; 
  fYLimit[1] = 0.0; 
}


BVPlotConf::~BVPlotConf ()   
{
}



// *************************************************
// *                                               *
// *   ---  BVPlotData:  Passed Data Object  ---   *
// *                                               *
// *************************************************

BVPlotData::BVPlotData () 
  : fBic (0), fPSD1 (0), fPSD2 (0), fPSD3 (0)
    //     fNew (false), 
{
}

BVPlotData::~BVPlotData ()   
{
}




// *********************************************************
// *                                                       *
// *   ---  BVWindow:  GUI Window  Constructor  ---   *
// *                                                       *
// *********************************************************
const char* def_font_12 = "-adobe-helvetica-bold-r-*-*-12-*-*-*-*-*-iso8859-1";
const char* def_font_18 = "-adobe-helvetica-bold-r-*-*-18-*-*-*-*-*-iso8859-1";
GContext_t BVWindow::fgButtonGC = 0;
FontStruct_t BVWindow::fgButtonFont = 0;
GContext_t BVWindow::fgMonNameGC = 0;
FontStruct_t BVWindow::fgMonNameFont = 0;

//_____________________________________________________________________
//
// 
BVWindow::BVWindow (const TGWindow *p, UInt_t w, UInt_t h,
		    BicoViewer& monitor, const char* chnnames)
  : TGMainFrame (p, w, h), fMonitor (&monitor), fCalcConf (&monitor.fCalcConf), 
    fPlotDataActive (&monitor.fPlotData1), 
    //     fPlotDataQueue (&monitor.fPlotData2),
    fSkipHeartbeats (0), fRefreshGUItime (0), fPlotTimer (0), firstPlot (true), 
    newPlot (false), fPlotNumber (0)
  
{
#ifdef BV_DEBUG
  cout << " Build the Window ..."<<endl;
#endif

  // CHANNEL & RATE EXTRACTION 
  // _________________________
  // 
  // Form string of channel names and rates for the channel selector widget
  //   - Routine sets up a temporary DACC, extracts a frame, iterates through 
  //       the channels, grabs the names and rates, puts them in a structure.
  //       
  //       This must be done in the constructor so as not to interfere with 
  //       the DACC created by EXEROOT.
  //       
#ifdef BV_DEBUG
  cout << " Get Frame File ..."<<endl;
#endif

  using FrameCPP::FrRawData;
  using FrameCPP::FrAdcData;
  using FrameCPP::FrameH;

  DaccIn tempDacc;
  tempDacc.setTOCMode(false);
  const char* firstFile = fMonitor->getDacc().getFile();
  if (!firstFile || !*firstFile) {
    cerr<< "Unable to find DACC input file." << endl;
    fMonitor->finish();
    return;
  } 
   
  isOnline = (strstr(firstFile,"online") != 0);
   
  if (isOnline || strstr(firstFile,".gwf") != 0 || strstr(firstFile,".F") != 0) {
    tempDacc.addFile(firstFile);
  } else {
    cerr<< "DACC input file has improper name: " << firstFile << endl;
    fMonitor->finish();
    return;
  }
   
	
  if (tempDacc.open() == 0) {
#ifdef BV_DEBUG
    cout << " Opening Frame File ..."<<endl;
#endif
  } else {
    cerr<< "Unable to open data input file." << endl;
    fMonitor->finish();
    return;
  }


#ifdef BV_DEBUG
  cout << " Data input file = "<< firstFile <<endl;
#endif
		
  if (tempDacc.nextFrame() == -1) {
    cerr << "Attempt to get next Frame failed" << endl;
  }
	
  if (!tempDacc.haveFrame()) {
    cerr << "No Frame from which to extract name and rate info" << endl;
    fMonitor->finish();
    return;
  }
  frameh_pointer tempFrame(tempDacc.getFrame());
	
	
	
  // Get the raw frame data 
  // ----------------------
#ifdef BV_DEBUG
  cout << " Extract Frame Data ..."<<endl;
#endif

  frrawdata_pointer raw(tempFrame->GetRawData());

  if (!raw) {
    cout << "No raw data in frame." << endl;
    return;
  }
   
// ****************** Change this *************************************
// ****************** Use TOC - DaccApi????
    size_t chanAlloc = 2000;
    chanNames = new ChannelEntry[chanAlloc];
    chanTotal = 0 ;


    for (FrRawData::const_firstAdc_iterator i = raw->RefFirstAdc().begin() ;
        i != raw->RefFirstAdc().end() ; ++i)
    {
      string name = (*i)->GetName();
      double rate = (*i)->GetSampleRate();

      if (rate > 16) {
         chanNames[chanTotal].SetName(name.c_str());
         chanNames[chanTotal].SetRate(float(rate));
         chanTotal = chanTotal + 1;
      }
    }
    ChannelTree::SortChannelList(chanNames, chanTotal);
   
#ifdef BV_DEBUG
  cout << " Setting Graphics Context ..."<<endl;
#endif

  // SET GRAPHICS CONTEXT
  // --------------------
  if (fgButtonGC == 0) {
    fgButtonFont = gClient->GetFontByName(gEnv->GetValue("Gui.NormalFont",
							 def_font_12));
    GCValues_t   gval;
    gval.fMask = kGCForeground | kGCFont;
    gval.fFont = gVirtualX->GetFontHandle (fgButtonFont);
    gval.fForeground = fgBlackPixel;
    //fgButtonGC = gVirtualX->CreateGC (gClient->GetRoot()->GetId(), &gval);
    fgButtonGC = gClient->GetGC(&gval)->GetGC();
  }
  if (fgMonNameGC == 0) {
    fgMonNameFont = gClient->GetFontByName(gEnv->GetValue("Gui.NormalFont", 
							  def_font_18));
    GCValues_t   gval;
    gval.fMask = kGCForeground | kGCFont;
    gval.fFont = gVirtualX->GetFontHandle (fgMonNameFont);
    gval.fForeground = fgBlackPixel;
    fgMonNameGC = gVirtualX->CreateGC (gClient->GetRoot()->GetId(), &gval);
  }


#ifdef BV_DEBUG
  cout << " Constructing GUI ..."<<endl;
#endif


  // LAYOUT OF THE GUI
  // -----------------
  // The Main Window is 1150 wide by 750 tall.
  // The window is created and sized in the BicoViewer constructor.
  //

  // Main Layout
  // -----------
#ifdef BV_DEBUG
  cout << " Constructing Main frames ..."<<endl;
#endif
  // Main Frame
  fFrame[0] = new TGCompositeFrame (this, w, h, kHorizontalFrame);
  fLayout[0] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandY | kLHintsExpandX, 0, 0, 0, 0);
  AddFrame (fFrame[0], fLayout[0]);

  // Configuration Frame
  fFrame[1] = new TGCompositeFrame (fFrame[0], 250, 750, kVerticalFrame | kLHintsExpandY);
  fLayout[1] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandY, 0, 0, 0, 0);
  fFrame[0]->AddFrame (fFrame[1], fLayout[1]);

  // Main Plot Frame
  fFrame[2] = new TGCompositeFrame (fFrame[0], 900, 750, kVerticalFrame);
  fFrame[2] -> Resize(900,750);
  fLayout[2] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandX | kLHintsExpandY, 0, 0, 0, 0);
  fFrame[0] -> AddFrame (fFrame[2], fLayout[2]);
     
  // Bicoherence Frame
  fFrame[3] = new TGCompositeFrame (fFrame[2], 900, 500, kHorizontalFrame);
  fFrame[3] -> Resize(900,500);
  fLayout[3] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandX | kLHintsExpandY, 0, 0, 0, 0);
  fFrame[2] -> AddFrame (fFrame[3], fLayout[3]);
     
  // Power Spectra Frame
  fFrame[4] = new TGCompositeFrame (fFrame[2], 900, 240, kHorizontalFrame);
  fFrame[4] -> Resize(900,230);
  fLayout[4] = new TGLayoutHints (kLHintsLeft | kLHintsBottom | kLHintsExpandX, 0, 0, 10, 0);
  fFrame[2] -> AddFrame (fFrame[4], fLayout[4]);

  // Bicoherence Canvas
  fEmbedCvsBico = new TRootEmbeddedCanvas (0, fFrame[3], 900, 500);
  fCvsBico = new TCanvas("cvs_bicoherence",10, 10, fEmbedCvsBico->GetCanvasWindowId());
  fEmbedCvsBico -> AdoptCanvas(fCvsBico);
  fLayout[5] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandX | kLHintsExpandY, 0, 0, 0, 0);
  fFrame[3] -> AddFrame (fEmbedCvsBico, fLayout[5]);
    
  // Power Spectra Canvas
  fEmbedCvsPSD1 = new TRootEmbeddedCanvas (0, fFrame[4], 300, 240);
  fCvsPSD1 = new TCanvas("cvs_psd1",10, 10, fEmbedCvsPSD1->GetCanvasWindowId());
  fEmbedCvsPSD1 -> AdoptCanvas(fCvsPSD1);
  fLayout[6] = new TGLayoutHints (kLHintsLeft | kLHintsBottom | kLHintsExpandX, 0, 0, 0, 0);
  fFrame[4] -> AddFrame (fEmbedCvsPSD1, fLayout[6]);

  fEmbedCvsPSD2 = new TRootEmbeddedCanvas (0, fFrame[4], 300, 240);
  fCvsPSD2 = new TCanvas("cvs_psd2",10, 10, fEmbedCvsPSD2->GetCanvasWindowId());
  fEmbedCvsPSD2 -> AdoptCanvas(fCvsPSD2);
  fFrame[4] -> AddFrame (fEmbedCvsPSD2, fLayout[6]);

  fEmbedCvsPSD3 = new TRootEmbeddedCanvas (0, fFrame[4], 300, 240);
  fCvsPSD3 = new TCanvas("cvs_psd3",10, 10, fEmbedCvsPSD3->GetCanvasWindowId());
  fEmbedCvsPSD3 -> AdoptCanvas(fCvsPSD3);
  fFrame[4] -> AddFrame (fEmbedCvsPSD3, fLayout[6]); 


  // Name Panel
  // ----------
#ifdef BV_DEBUG
  cout << " Constructing Name frame ..."<<endl;
#endif
  fFrameName = new TGCompositeFrame (fFrame[1], 250, 70, 
				    kVerticalFrame | kSunkenFrame | kFixedSize);
  fLayoutName[0] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 0);
  fFrame[1]->AddFrame (fFrameName, fLayoutName[0]);
  fLabelName[0] = new TGLabel (fFrameName, "BICOHERENCE VIEWER", 
			       fgMonNameGC, fgMonNameFont);
  fLayoutName[1] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 10, 5);
  fFrameName->AddFrame (fLabelName[0], fLayoutName[1]);
  fLabelName[1] = new TGLabel (fFrameName, "Steve Penn -- v. 0.8, October 05");
  fLayoutName[2] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 0, 5);
  fFrameName->AddFrame (fLabelName[1], fLayoutName[2]);
  fFrameName->DrawBorder();


  // Calculation Parameters Panel
  // ----------------------------
#ifdef BV_DEBUG
  cout << " Constructing Parameter frame ..."<<endl;
#endif
  fFrameCalc[0] = new TGCompositeFrame (fFrame[1], 250, 330, 
				    kVerticalFrame | kRaisedFrame | kFixedSize);
  fLayoutCalc[0] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 0);
  fFrame[1]->AddFrame (fFrameCalc[0], fLayoutCalc[0]);
   
  // Title 
  fLabelCalc[0] = new TGLabel (fFrameCalc[0], "CALCULATION PARAMETERS", 
			       fgButtonGC, fgButtonFont);
  fLayoutCalc[1] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 10, 5);
  fFrameCalc[0]->AddFrame (fLabelCalc[0], fLayoutCalc[1]);

  // Subtitle 
  fLabelCalc[1] = new TGLabel (fFrameCalc[0], "(Select 1--3 channels)");
  fLayoutCalc[2] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 0, 5);
  fFrameCalc[0]->AddFrame (fLabelCalc[1], fLayoutCalc[2]);

  // Channel selection widgets
  Int_t ctype = kChannelTreeShowRate;
  fLayoutCalc[3] = new TGLayoutHints (kLHintsCenterX | kLHintsTop | kLHintsExpandX, 10, 10, 2, 2);  
  for (Int_t i=0; i<3; ++i) {
    fChannelSel[i] = new TLGChannelCombobox (fFrameCalc[0], kChanSel1+i, 
					     chanNames, chanTotal, kTRUE, 
					     ctype, kTRUE);
    fChannelSel[i]->Associate (this);
    fChannelSel[i]->Resize(180, 22);
    fChannelSel[i]->SetPopupHeight (650);
    fFrameCalc[0]->AddFrame (fChannelSel[i], fLayoutCalc[3]);
  }

  // Table form for selecting and posting parameters
  // 
  fLayoutCalc[4] = new TGLayoutHints (kLHintsLeft | kLHintsTop , 10, 0, 0, 0);
  fLayoutCalc[5] = new TGLayoutHints (kLHintsLeft | kLHintsBottom , 0, 0, 0, 2);
  fLayoutCalc[6] = new TGLayoutHints (kLHintsLeft | kLHintsTop , 0, 0, 0, 0);
  // Column Label Frames
  fFrameCalc[1] = new TGCompositeFrame (fFrameCalc[0], 230, 35, 
					kHorizontalFrame | kFixedSize);
  fFrameCalc[0]->AddFrame (fFrameCalc[1], fLayoutCalc[4]);
  fFrameCalc[2] = new TGCompositeFrame (fFrameCalc[1], 100, 25, 
				 kHorizontalFrame | kFixedWidth | kRaisedFrame);
  fFrameCalc[1]->AddFrame (fFrameCalc[2], fLayoutCalc[5]);
  fFrameCalc[3] = new TGCompositeFrame (fFrameCalc[1], 65, 25, 
					kHorizontalFrame | kFixedWidth | kRaisedFrame);
  fFrameCalc[1]->AddFrame (fFrameCalc[3], fLayoutCalc[5]);
  fFrameCalc[4] = new TGCompositeFrame (fFrameCalc[1], 65, 25, 
					kHorizontalFrame | kFixedWidth | kRaisedFrame );
  fFrameCalc[1]->AddFrame (fFrameCalc[4], fLayoutCalc[5]);
  // Frequency Range Selector
  fFrameCalc[5] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[5], fLayoutCalc[4]);
  fFrameCalc[6] = new TGCompositeFrame (fFrameCalc[5], 100, 25, 
					kHorizontalFrame | kFixedWidth);
  fFrameCalc[5]->AddFrame (fFrameCalc[6], fLayoutCalc[6]);
  fFrameCalc[7] = new TGCompositeFrame (fFrameCalc[5], 65, 25, 
					kHorizontalFrame | kFixedWidth);
  fFrameCalc[5]->AddFrame (fFrameCalc[7], fLayoutCalc[6]);
  fFrameCalc[8] = new TGCompositeFrame (fFrameCalc[5], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[5]->AddFrame (fFrameCalc[8], fLayoutCalc[6]);
  // Frequency Resolution Selector
  fFrameCalc[9] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[9], fLayoutCalc[4]);
  fFrameCalc[10] = new TGCompositeFrame (fFrameCalc[9], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[9]->AddFrame (fFrameCalc[10], fLayoutCalc[6]);
  fFrameCalc[11] = new TGCompositeFrame (fFrameCalc[9], 65, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[9]->AddFrame (fFrameCalc[11], fLayoutCalc[6]);
  fFrameCalc[12] = new TGCompositeFrame (fFrameCalc[9], 65, 25, 
			      	 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[9]->AddFrame (fFrameCalc[12], fLayoutCalc[6]);
  // Time Span Selector
  fFrameCalc[13] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					 kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[13], fLayoutCalc[4]);
  fFrameCalc[14] = new TGCompositeFrame (fFrameCalc[13], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[13]->AddFrame (fFrameCalc[14], fLayoutCalc[6]);
  fFrameCalc[15] = new TGCompositeFrame (fFrameCalc[13], 65, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[13]->AddFrame (fFrameCalc[15], fLayoutCalc[6]);
  fFrameCalc[16] = new TGCompositeFrame (fFrameCalc[13], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[13]->AddFrame (fFrameCalc[16], fLayoutCalc[6]);
  // Overlap Selector
  fFrameCalc[17] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					 kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[17], fLayoutCalc[4]);
  fFrameCalc[18] = new TGCompositeFrame (fFrameCalc[17], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[17]->AddFrame (fFrameCalc[18], fLayoutCalc[6]);
  fFrameCalc[19] = new TGCompositeFrame (fFrameCalc[17], 65, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[17]->AddFrame (fFrameCalc[19], fLayoutCalc[6]);
  fFrameCalc[20] = new TGCompositeFrame (fFrameCalc[17], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[17]->AddFrame (fFrameCalc[20], fLayoutCalc[6]);
  // Windowing Selector
  fFrameCalc[21] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					 kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[21], fLayoutCalc[4]);
  fFrameCalc[22] = new TGCompositeFrame (fFrameCalc[21], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[21]->AddFrame (fFrameCalc[22], fLayoutCalc[6]);
  fFrameCalc[23] = new TGCompositeFrame (fFrameCalc[21], 65, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[21]->AddFrame (fFrameCalc[23], fLayoutCalc[6]);
  fFrameCalc[24] = new TGCompositeFrame (fFrameCalc[21], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[21]->AddFrame (fFrameCalc[24], fLayoutCalc[6]);
  // Uncertainty listing
  fFrameCalc[25] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					 kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[25], fLayoutCalc[4]);
  fFrameCalc[26] = new TGCompositeFrame (fFrameCalc[25], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[25]->AddFrame (fFrameCalc[26], fLayoutCalc[6]);
  fFrameCalc[27] = new TGCompositeFrame (fFrameCalc[25], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[25]->AddFrame (fFrameCalc[27], fLayoutCalc[6]);
  fFrameCalc[28] = new TGCompositeFrame (fFrameCalc[25], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[25]->AddFrame (fFrameCalc[28], fLayoutCalc[6]);
  // Refresh Time listing
  fFrameCalc[29] = new TGCompositeFrame (fFrameCalc[0], 230, 25, 
					 kHorizontalFrame);
  fFrameCalc[0]->AddFrame (fFrameCalc[29], fLayoutCalc[4]);
  fFrameCalc[30] = new TGCompositeFrame (fFrameCalc[29], 100, 25, 
					 kHorizontalFrame | kFixedWidth);
  fFrameCalc[29]->AddFrame (fFrameCalc[30], fLayoutCalc[6]);
  fFrameCalc[31] = new TGCompositeFrame (fFrameCalc[29], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[29]->AddFrame (fFrameCalc[31], fLayoutCalc[6]);
  fFrameCalc[32] = new TGCompositeFrame (fFrameCalc[29], 65, 25, 
				 kHorizontalFrame | kFixedWidth | kSunkenFrame);
  fFrameCalc[29]->AddFrame (fFrameCalc[32], fLayoutCalc[6]);
   
  // Table Layouts   
  fLayoutCalc[7] = new TGLayoutHints (kLHintsLeft | kLHintsCenterY, 2, 2, 2, 2);
  fLayoutCalc[8] = new TGLayoutHints (kLHintsCenterX | kLHintsCenterY, 
				      2, 2, 2, 2);
  fLayoutCalc[9] = new TGLayoutHints (kLHintsCenterX | kLHintsCenterY, 
				      0, 0, 0, 0);
   
  // Table Column Labels
  fLabelCalc[2] = new TGLabel (fFrameCalc[2], "PARAMETER");
  fFrameCalc[2]->AddFrame (fLabelCalc[2], fLayoutCalc[7]);
  fLabelCalc[3] = new TGLabel (fFrameCalc[3], "NEW");
  fFrameCalc[3]->AddFrame (fLabelCalc[3], fLayoutCalc[8]);
  fLabelCalc[4] = new TGLabel (fFrameCalc[4], "CURRENT");
  fFrameCalc[4]->AddFrame (fLabelCalc[4], fLayoutCalc[8]);
      
  // Frequency Range Selector
  fLabelCalc[5] = new TGLabel (fFrameCalc[6], "Freq Max (Hz)");
  fFrameCalc[6]->AddFrame (fLabelCalc[5], fLayoutCalc[7]);
  fFreqMaxBox = new TGComboBox (fFrameCalc[7], kFreqMax); 
  for (Int_t iEntry=0; iEntry<kFreqMaxSize; ++iEntry) {
    fFreqMaxBox -> AddEntry(kFreqMaxText[iEntry], iEntry);
  }   
  fFreqMaxBox -> Select(0);
  fFreqMaxBox->Associate (this);
  fFreqMaxBox -> Resize(65,20);
  fFrameCalc[7]->AddFrame (fFreqMaxBox, fLayoutCalc[9]);
  fLabelCalc[6] = new TGLabel (fFrameCalc[8], "F_Nyq");
  fFrameCalc[8]->AddFrame (fLabelCalc[6], fLayoutCalc[8]);

  // Frequency Resolution Selector
  fLabelCalc[7] = new TGLabel (fFrameCalc[10], "Freq Resol (Hz)");
  fFrameCalc[10]->AddFrame (fLabelCalc[7], fLayoutCalc[7]);
  fFreqResBox = new TGComboBox (fFrameCalc[11], kFreqRes);
  for (Int_t iEntry=0; iEntry<kFreqResSize; ++iEntry) {
    fFreqResBox -> AddEntry(kFreqResText[iEntry], iEntry);
  }   
  fFreqResBox -> Select(0);
  fFreqResBox -> Associate (this);
  fFreqResBox -> Resize(65,20);
  fFrameCalc[11]->AddFrame (fFreqResBox, fLayoutCalc[9]);
  fLabelCalc[8] = new TGLabel (fFrameCalc[12], "Deflt");
  fFrameCalc[12]->AddFrame (fLabelCalc[8], fLayoutCalc[8]);
  
  // Span Selector
  fLabelCalc[9] = new TGLabel (fFrameCalc[14], "Data Span (s)  ");
  fFrameCalc[14]->AddFrame (fLabelCalc[9], fLayoutCalc[7]);
  fTimeSpanBox = new TGComboBox (fFrameCalc[15], kTimeSpan);
  for (Int_t iEntry=0; iEntry<kTimeSpanSize; ++iEntry) {
    fTimeSpanBox -> AddEntry(kTimeSpanText[iEntry], iEntry);
  }   
  fTimeSpanBox -> Select(0);
  fTimeSpanBox -> Associate (this);
  fTimeSpanBox -> Resize(65,20);
  fFrameCalc[15]->AddFrame (fTimeSpanBox, fLayoutCalc[9]);
  fLabelCalc[10] = new TGLabel (fFrameCalc[16], "Deflt");
  fFrameCalc[16]->AddFrame (fLabelCalc[10], fLayoutCalc[8]);
   
  // Overlap Selector
  fLabelCalc[11] = new TGLabel (fFrameCalc[18], "Data Overlap (%)");
  fFrameCalc[18]->AddFrame (fLabelCalc[11], fLayoutCalc[7]);
  fOverlapBox = new TGComboBox (fFrameCalc[19], kOverlap);
  for (Int_t iEntry=0; iEntry<kOverlapSize; ++iEntry) {
    fOverlapBox -> AddEntry(kOverlapText[iEntry], iEntry);
  }   
  fOverlapBox -> Select(0);
  fOverlapBox -> Associate (this);
  fOverlapBox -> Resize(65,20);
  fFrameCalc[19]->AddFrame (fOverlapBox, fLayoutCalc[9]);
  fLabelCalc[12] = new TGLabel (fFrameCalc[20], "Deflt");
  fFrameCalc[20]->AddFrame (fLabelCalc[12], fLayoutCalc[8]);
   
  // Windowing Selector
  fLabelCalc[13] = new TGLabel (fFrameCalc[22], "Window Method");
  fFrameCalc[22]->AddFrame (fLabelCalc[13], fLayoutCalc[7]);
  fWindowTypeBox = new TGComboBox (fFrameCalc[23], kWindowTypeBox);
  for (Int_t iEntry=0; iEntry<kWindowTypeSize; ++iEntry) {
    fWindowTypeBox -> AddEntry(kWindowTypeText[iEntry], iEntry);
  }   
  fWindowTypeBox -> Select(0);
  fWindowTypeBox -> Associate (this);
  fWindowTypeBox -> Resize(65,20);
  fFrameCalc[23]->AddFrame (fWindowTypeBox, fLayoutCalc[9]);
  fLabelCalc[14] = new TGLabel (fFrameCalc[24], "Deflt");
  fFrameCalc[24]->AddFrame (fLabelCalc[14], fLayoutCalc[8]);


  // Uncertainty listing
  fLabelCalc[15] = new TGLabel (fFrameCalc[26], "Uncertainty (%)");
  fFrameCalc[26]->AddFrame (fLabelCalc[15], fLayoutCalc[7]);
  fLabelCalc[16] = new TGLabel (fFrameCalc[27], "Est Acc 1");
  fFrameCalc[27]->AddFrame (fLabelCalc[16], fLayoutCalc[8]);
  fLabelCalc[17] = new TGLabel (fFrameCalc[28], "Est Acc 2");
  fFrameCalc[28]->AddFrame (fLabelCalc[17], fLayoutCalc[8]);


  // Refresh Time listing
  fLabelCalc[18] = new TGLabel (fFrameCalc[30], "Calculation Time");
  fFrameCalc[30]->AddFrame (fLabelCalc[18], fLayoutCalc[7]);
  fLabelCalc[19] = new TGLabel (fFrameCalc[31], "Time 1");
  fFrameCalc[31]->AddFrame (fLabelCalc[19], fLayoutCalc[8]);
  fLabelCalc[20] = new TGLabel (fFrameCalc[32], "Time 2");
  fFrameCalc[32]->AddFrame (fLabelCalc[20], fLayoutCalc[8]);




  // SAVE PANEL
  // ----------
#ifdef BV_DEBUG
  cout << " Constructing Save frame ..."<<endl;
#endif
  fFrameSave[0] = new TGCompositeFrame(fFrame[1], 250, 100, 
				       kVerticalFrame|kRaisedFrame|kFixedSize);
  fLayoutSave[0] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 0);
  fFrame[1]->AddFrame (fFrameSave[0], fLayoutSave[0]);
   
  // Title 
  fLabelSave = new TGLabel (fFrameSave[0], "SAVE BICOHERENCE PLOT", fgButtonGC,
			    fgButtonFont);
  fLayoutSave[1] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 10, 5);
  fFrameSave[0]->AddFrame (fLabelSave, fLayoutSave[1]);

  // Write Bicoherence Files Check Box
  fFrameSave[1] = new TGCompositeFrame (fFrameSave[0], 250, 25, 
					kHorizontalFrame);
  fLayoutSave[2] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandX,
				      0, 0, 0, 0);  
  fFrameSave[0]->AddFrame (fFrameSave[1], fLayoutSave[2]);
  fWriteBicoCheck = new TGCheckButton(fFrameSave[1], 
				   new TGHotString(kWriteBicoText), kWriteBico);
  //    fWriteBicoCheck = new TGCheckButton (fFrameSave[1], new TGHotString (kWriteBicoText), 
  //                                         kWriteBico, fgButtonGC, fgButtonFont);
  fWriteBicoCheck->Associate (this);
  fLayoutSave[3] = new TGLayoutHints (kLHintsLeft | kLHintsTop, 10, 0, 2, 2);
  fFrameSave[1]->AddFrame (fWriteBicoCheck, fLayoutSave[3]);
  fFileFormatBox = new TGComboBox (fFrameSave[1], kOverlap);
  fFileFormatBox -> AddEntry(" GIF  file", 0);
  fFileFormatBox -> AddEntry("ASCII file", 1);
  fFileFormatBox -> Select(0);
  fFileFormatBox -> Associate (this);
  fFileFormatBox -> Resize(70,20);
  fLayoutSave[4] = new TGLayoutHints (kLHintsRight | kLHintsTop, 0, 10, 2, 2);
  fFrameSave[1]->AddFrame (fFileFormatBox, fLayoutSave[4]);


#ifdef BV_DEBUG
  cout << " Constructing Save Button ..."<<endl;
#endif
   
  // Write Current Button
  fButtonWriteNow = new TGTextButton (fFrameSave[0], 
				      new TGHotString (kButtonWriteNowText), 
				      kButtonWriteNow, fgButtonGC, fgButtonFont);
  fButtonWriteNow->Associate (this);
  fButtonWriteNow->SetToolTipText (kButtonWriteNowHelp, 500);
  fLayoutSave[5] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 5, 5, 5, 5);
  fFrameSave[0]->AddFrame (fButtonWriteNow, fLayoutSave[5]);
   
   
   
  // PLOT LIMITS PANEL
  // -----------------
#ifdef BV_DEBUG
  cout << " Constructing Plot Limits frame ..."<<endl;
#endif
  fFramePlot[0] = new TGCompositeFrame(fFrame[1], 250, 125, 
				       kVerticalFrame | kRaisedFrame | kFixedSize);
  fLayoutPlot[0] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 0);
  fFrame[1]->AddFrame (fFramePlot[0], fLayoutPlot[0]);
	
  // Title 
  fLabelPlot[0] = new TGLabel (fFramePlot[0], "PLOT LIMITS", fgButtonGC, 
			       fgButtonFont);
  fLayoutPlot[1] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 
				      5, 5, 10, 10);
  fFramePlot[0]->AddFrame (fLabelPlot[0], fLayoutPlot[1]);

  // Z Scale Selector
  fFramePlot[1] = new TGCompositeFrame (fFramePlot[0], 250, 25, 
					kHorizontalFrame);
  fLayoutPlot[2] = new TGLayoutHints (kLHintsLeft | kLHintsTop | kLHintsExpandX,
				      0, 0, 0, 10);  
  fFramePlot[0]->AddFrame (fFramePlot[1], fLayoutPlot[2]);
  fLabelPlot[1] = new TGLabel (fFramePlot[1], "Z Scale");
  fLayoutPlot[3] = new TGLayoutHints (kLHintsLeft | kLHintsTop, 25, 5, 2, 2);
  fFramePlot[1]->AddFrame (fLabelPlot[1], fLayoutPlot[3]);
  fZScaleBox = new TGComboBox (fFramePlot[1], kZScale);
  fZScaleBox -> AddEntry("   Linear    ", 0);
  fZScaleBox -> AddEntry(" Logarithmic ", 1);
  fZScaleBox -> Select(0);
  fZScaleBox -> Associate (this);
  fZScaleBox -> Resize(110,20);
  fLayoutPlot[4] = new TGLayoutHints (kLHintsLeft | kLHintsTop, 0, 0, 0, 0);
  fFramePlot[1]->AddFrame (fZScaleBox, fLayoutPlot[4]);
   
  // Y Limits Selector
  fFramePlot[2] = new TGCompositeFrame (fFramePlot[0], 250, 25, 
					kHorizontalFrame | kFixedWidth);
  fLayoutPlot[5] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 3, 3);
  fFramePlot[0]->AddFrame (fFramePlot[2], fLayoutPlot[5]);
  fFramePlot[3] = new TGCompositeFrame (fFramePlot[2], 50, 25, 
					kHorizontalFrame | kFixedWidth);
  fLayoutPlot[6] = new TGLayoutHints (kLHintsLeft | kLHintsTop, 0, 0, 0, 0);
  fFramePlot[2]->AddFrame (fFramePlot[3], fLayoutPlot[6]);
  fFramePlot[4] = new TGCompositeFrame (fFramePlot[2], 75, 25, 
					kHorizontalFrame | kFixedWidth);
  fFramePlot[2]->AddFrame (fFramePlot[4], fLayoutPlot[6]);
  fFramePlot[5] = new TGCompositeFrame (fFramePlot[2], 75, 25, 
					kHorizontalFrame | kFixedWidth);
  fFramePlot[2]->AddFrame (fFramePlot[5], fLayoutPlot[6]);
  fFramePlot[6] = new TGCompositeFrame (fFramePlot[2], 50, 25, 
					kHorizontalFrame | kFixedWidth);
  fFramePlot[2]->AddFrame (fFramePlot[6], fLayoutPlot[6]);
  fLabelPlot[2] = new TGLabel (fFramePlot[3], "Ymin");
  fLayoutPlot[7] = new TGLayoutHints (kLHintsRight | kLHintsCenterY, 
				      0, 0, 0, 0);
  fFramePlot[3]->AddFrame (fLabelPlot[2], fLayoutPlot[7]);
  fYLimitBox[0] = new TLGNumericControlBox (fFramePlot[4], fPlotConf.fYLimit[0],
					    12, kYLimitMin);
  fYLimitBox[0]->Associate (this);
  fYLimitBox[0]->Resize(65,20);
  fLayoutPlot[8] = new TGLayoutHints (kLHintsCenterX | kLHintsCenterY, 
				      0, 0, 0, 0);
  fFramePlot[4]->AddFrame (fYLimitBox[0], fLayoutPlot[8]);
  fYLimitBox[1] = new TLGNumericControlBox (fFramePlot[5], fPlotConf.fYLimit[1],
					    12, kYLimitMax);
  fYLimitBox[1]->Associate (this);
  fYLimitBox[1]->Resize(65,20);  
  fFramePlot[5]->AddFrame (fYLimitBox[1], fLayoutPlot[8]);
  fLabelPlot[3] = new TGLabel (fFramePlot[6], "Ymax");
  fLayoutPlot[9] = new TGLayoutHints (kLHintsLeft | kLHintsCenterY, 0, 0, 0, 0);
  fFramePlot[6]->AddFrame (fLabelPlot[3], fLayoutPlot[9]);


  // X Limits Selector
  fFramePlot[7] = new TGCompositeFrame (fFramePlot[0], 250, 25,
					kHorizontalFrame | kFixedWidth);
  fFramePlot[0]->AddFrame (fFramePlot[7], fLayoutPlot[5]);
  fFramePlot[8] = new TGCompositeFrame (fFramePlot[7], 50, 25,
					kHorizontalFrame | kFixedWidth);
  fFramePlot[7]->AddFrame (fFramePlot[8], fLayoutPlot[6]);
  fFramePlot[9] = new TGCompositeFrame (fFramePlot[7], 75, 25,
					kHorizontalFrame | kFixedWidth);
  fFramePlot[7]->AddFrame (fFramePlot[9], fLayoutPlot[6]);
  fFramePlot[10] = new TGCompositeFrame (fFramePlot[7], 75, 25,
					 kHorizontalFrame | kFixedWidth);
  fFramePlot[7]->AddFrame (fFramePlot[10], fLayoutPlot[6]);
  fFramePlot[11] = new TGCompositeFrame (fFramePlot[7], 50, 25,
					 kHorizontalFrame | kFixedWidth);
  fFramePlot[7]->AddFrame (fFramePlot[11], fLayoutPlot[6]);
  fLabelPlot[4] = new TGLabel (fFramePlot[8], "Xmin");
  fFramePlot[8]->AddFrame (fLabelPlot[4], fLayoutPlot[7]);
  fXLimitBox[0] = new TLGNumericControlBox (fFramePlot[9], fPlotConf.fXLimit[0],
					    12, kXLimitMin);
  fXLimitBox[0]->Associate (this);
  fXLimitBox[0]->Resize(65,20);
  fFramePlot[9]->AddFrame (fXLimitBox[0], fLayoutPlot[8]);
  fXLimitBox[1] = new TLGNumericControlBox (fFramePlot[10], 
					    fPlotConf.fXLimit[1], 
					    12, kXLimitMax);
  fXLimitBox[1]->Associate (this);
  fXLimitBox[1]->Resize(65,20);  
  fFramePlot[10]->AddFrame (fXLimitBox[1], fLayoutPlot[8]);
  fLabelPlot[5] = new TGLabel (fFramePlot[11], "Xmax");
  fFramePlot[11]->AddFrame (fLabelPlot[5], fLayoutPlot[9]);

  // CONTROL BUTTONS PANEL
  // ---------------------
#ifdef BV_DEBUG
  cout << " Constructing Control Buttons frame ..."<<endl;
#endif
  fFrameCtrl[0] = new TGCompositeFrame(fFrame[1], 250, 125, 
				    kVerticalFrame | kRaisedFrame | kFixedSize);
  fLayoutCtrl[0] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 0);
  fFrame[1]->AddFrame (fFrameCtrl[0], fLayoutCtrl[0]);

  // Title 
  fLabelCtrl[0] = new TGLabel (fFrameCtrl[0], "MONITOR CONTROL", fgButtonGC, 
			       fgButtonFont);
  fLayoutCtrl[1] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 
				      5, 5, 10, 10);
  fFrameCtrl[0]->AddFrame (fLabelCtrl[0], fLayoutCtrl[1]);

  // Calculation Time Label 
  fFrameCtrl[1] = new TGCompositeFrame (fFrameCtrl[0], 250, 25, 
					kHorizontalFrame | kFixedWidth);
  fLayoutCtrl[2] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 0, 0, 0, 10);
  fFrameCtrl[0]->AddFrame (fFrameCtrl[1], fLayoutCtrl[2]);
  fLabelCtrl[1] = new TGLabel(fFrameCtrl[1], 
			      "Time to next plot (s) = " );   //kCalcEstText[0]
  fLayoutCtrl[3] = new TGLayoutHints (kLHintsLeft | kLHintsCenterY, 
				      30, 0, 0, 0);
  fFrameCtrl[1]->AddFrame (fLabelCtrl[1], fLayoutCtrl[3]);
  fLabelCtrl[2] = new TGLabel (fFrameCtrl[1], 
			       "               " ); //kCalcEstText[0]
  fLayoutCtrl[4] = new TGLayoutHints (kLHintsLeft | kLHintsCenterY, 3, 0, 0, 0);
  fFrameCtrl[1]->AddFrame (fLabelCtrl[2], fLayoutCtrl[4]);

      
#ifdef BV_DEBUG
  cout << " Constructing Control Button ..."<<endl;
#endif

  // Control Buttons
  fFrameCtrl[2] = new TGCompositeFrame (fFrameCtrl[0], 240, 25, 
					kHorizontalFrame | kFixedSize);
  fLayoutCtrl[5] = new TGLayoutHints (kLHintsCenterX | kLHintsTop, 
				      5, 5, 0, 5);
  fFrameCtrl[0] -> AddFrame (fFrameCtrl[2], fLayoutCtrl[5]);
  fLayoutCtrl[6] = new TGLayoutHints (kLHintsCenterX | kLHintsCenterY, 
				      5, 5, 0, 0);
  for (int i = 0; i < 3; ++i) {
    fControlButton[i] = new TGTextButton(fFrameCtrl[2], 
					 new TGHotString(kControlButtonText[i]),
					 kButtonStart+i, fgButtonGC, 
					 fgButtonFont);
    fControlButton[i]->SetToolTipText (kControlButtonHelp[i]);
    fControlButton[i]->Associate (this);
    fFrameCtrl[2]->AddFrame (fControlButton[i], fLayoutCtrl[6]);
  }
  //  fFrameCtrl[3] = new TGCompositeFrame (fFrameCtrl[0], 240, 20, 
  //                                        kHorizontalFrame | kFixedSize);
  //  fFrameCtrl[0] -> AddFrame (fFrameCtrl[3], fLayoutCtrl[5]);
  //  for (int i = 2; i < 4; ++i) {
  //     fControlButton[i] = new TGTextButton (fFrameCtrl[3], 
  //                                           new TGHotString(kControlButtonText[i]), 
  //                                           kButtonStart + i, fgButtonGC,
  //                                           fgButtonFont);
  //     fControlButton[i]->Associate (this);
  //     fControlButton[i]->SetToolTipText (kControlButtonHelp[i]);
  // 	  fFrameCtrl[3]->AddFrame (fControlButton[i], fLayoutCtrl[6]);
  //  }

#ifdef BV_DEBUG
  cout << " Initializing the Window ..."<<endl;
#endif

  // SET WINDOW NAME
  // ---------------
  SetWindowName (kMonName);
  SetIconName (kMonName);
  SetClassHints (kMonName, kMonName);
   
  // RESIZE & MAP WINDOW
  // -------------------
  SetWMPosition (0,0);
  MapSubwindows ();
  Resize (GetDefaultSize());
  MapWindow ();
   
  // INITIALIZE HEARTBEAT TIMER
  // --------------------------
  fHeartbeat = new TTimer (this, kHeartbeatInterval, kTRUE);
  fHeartbeat->TurnOn();

  // SET BUTTON INITIAL STATE
  // ------------------------
  fCalcConf->fPause = true;
  fPaused = true;

  fControlButton[0]->SetState (kButtonUp);       // START
   
  fControlButton[1]->SetState (kButtonDown);       // PAUSE/RESUME
  fControlButton[1]->SetState (kButtonDisabled);
  
  fControlButton[2]->SetState (kButtonUp);       // QUIT

}

// ********************************************************
// *                                                      *
// *   ---  BVWindow:  GUI Window  Destructor  ---   *
// *                                                      *
// ********************************************************
BVWindow::~BVWindow()
{
  // Deleting the GUI Widget poiter arrays causes an error:
  // "Deallocation of a pointer not malloced"
  // 

#ifdef BV_DEBUG
  cout << "BVWindow Destructor " << endl;
#endif
   
    
  // Delete Timer
  // ------------
  delete fHeartbeat;

  // Delete Widgets
  // --------------
  delete fFreqMaxBox;
  delete fFreqResBox;
  delete fTimeSpanBox;
  delete fOverlapBox;
  //    delete fWindow2DCheck;
  delete fWindowTypeBox;
  delete fWriteBicoCheck;
  //    delete fLimitGIFBox;
  delete fFileFormatBox;
  delete fButtonWriteNow;
  delete fZScaleBox;
  //    delete[] fXLimitBox;
  //    delete[] fYLimitBox;
  delete fEmbedCvsBico;
  delete fEmbedCvsPSD1;
  delete fEmbedCvsPSD2;
  delete fEmbedCvsPSD3;
  //    delete[] fChannelSel;     	// Channel Selector
  //    delete[] fControlButton;  	// Control Buttons
  //    delete[] fFrame;           // Frames
  //    delete[] fLayout;          // Layout Hints
  delete   fFrameName;       // Frames
  // 	delete[] fLayoutName;      // Layout Hints
  // 	delete[] fLabelName;       // Labels
  // 	delete[] fFrameCalc;       // Frames
  // 	delete[] fLayoutCalc;      // Layout Hints
  // 	delete[] fLabelCalc;       // Labels
  // 	delete[] fFrameSave;       // Frames
  // 	delete[] fLayoutSave;      // Layout Hints
  delete   fLabelSave;       // Labels
  // 	delete[] fFramePlot;       // Frames
  // 	delete[] fLayoutPlot;      // Layout Hints
  // 	delete[] fLabelPlot;       // Labels
  // 	delete[] fFrameCtrl;       // Frames
  // 	delete[] fLayoutCtrl;      // Layout Hints
  // 	delete[] fLabelCtrl;       // Labels
}




// *******************************************************
// *                                                     *
// *   ---  BVWindow:  Close Window Function  ---        *
// *                                                     *
// *******************************************************
void BVWindow::CloseWindow()
{
  // Exit by deleting each process & thread
  //    fMonitor->fMainWindow = 0;
  //    fMonitor->finish();
  //    TGMainFrame::CloseWindow();
  //    delete this;
  //    pthread_exit(0);
       
  // Forced exit out of everything  
  ::exit(0);

}




// ***********************************************************
// *                                                         *
// *   ---  BVWindow:  Read Calculation Configuration  ---   *
// *                                                         *
// ***********************************************************
Bool_t BVWindow::ReadCalcConf()  
{
  //    Int_t fWindow2DRange = 64 ;  // 1 / Range = fraction of conf.fNFFT bins covered by Window 
   
  fCalcConf->lock();
   
  fCalcConf->fNChans = 0;
  fCalcConf->fChanName1 = "";
  fCalcConf->fChanName2 = "";
  fCalcConf->fChanName3 = "";
   
   
  for (int i=0; i<3; ++i) {
    std::string tempName = fChannelSel[i]->GetChannel();
    float tempRate = 0;
    if (tempName != "") {
      int j=0;
      while (tempName != chanNames[j].Name() && j<chanTotal) ++j;
      tempRate = chanNames[j].Rate();
      if (j < chanTotal && tempRate > 0) {
	fCalcConf->fNChans += 1;
	switch (fCalcConf->fNChans) {
	case 1:
	  fCalcConf->fChanName1 = tempName;
	  fCalcConf->fRate1 = tempRate;
	  fCalcConf->fRate = tempRate;
	  break;
	case 2:
	  fCalcConf->fChanName2 = tempName;
	  fCalcConf->fRate2 = tempRate;
	  fCalcConf->fRate = (tempRate < fCalcConf->fRate) ? tempRate : fCalcConf->fRate ;
	  break;
	case 3:
	  fCalcConf->fChanName3 = tempName;
	  fCalcConf->fRate3 = tempRate;
	  fCalcConf->fRate = (tempRate < fCalcConf->fRate) ? tempRate : fCalcConf->fRate ;
	  break;
	}
      } 
    } 
  }
   
  // If No Channels Selected or Rate not found then set InValid and return
  if (fCalcConf->fNChans < 1 || fCalcConf->fRate < 16) {
    fCalcConf->fValid = false;
    cout << "ERROR - Configuration Invalid.  No Channels Selected." << endl;
    fCalcConf->unlock();
    return kTRUE;
  }
      
  float fFreqNyq = 0.5 * fCalcConf->fRate;
      
  int tempFreqMax = fFreqMaxBox->GetSelected();
  if (tempFreqMax > 0) {
    float ftemp = pow(2.0,14-tempFreqMax);
    //       float ftemp = 1 << (14-tempFreqMax) ;
    fCalcConf->fFreqMax = (ftemp > fFreqNyq) ? fFreqNyq : ftemp ;
  } else {
    fCalcConf->fFreqMax = fFreqNyq;    // use Default
  }
  tempFreqMax = 14 - int(log(fCalcConf->fFreqMax) / log(2.0)) ;
  fFreqMaxBox -> Select(tempFreqMax);

   
   
  int tempFreqRes = fFreqResBox->GetSelected();
  if (tempFreqRes != 0) {
    fCalcConf->fFreqRes = pow(2.0, 7-tempFreqRes) ;   // Frequency Resolution
    //       fCalcConf->fFreqRes = 1 << (7-tempFreqRes) ;   // Frequency Resolution
    fCalcConf->fNBins = int(fCalcConf->fFreqMax/fCalcConf->fFreqRes);
    if (fCalcConf->fNBins < kFreqBinMin) {
      fCalcConf->fNBins = kFreqBinMin;
      fCalcConf->fFreqRes = fCalcConf->fFreqMax / kFreqBinMin ;
    } 
  } else {
    fCalcConf->fNBins = kFreqBinDef ;     // Default bins
    fCalcConf->fFreqRes = fCalcConf->fFreqMax / fCalcConf->fNBins ;    
  }   
  tempFreqRes = 7 - int(floor(0.5 + log(fCalcConf->fFreqRes)/log(2.0)));
  fFreqResBox -> Select(tempFreqRes);


   
  int tempSpan = fTimeSpanBox->GetSelected();
  if (tempSpan > 0) {
    //   fCalcConf->fSpan = 1 << (tempSpan-1) ;
    fCalcConf->fSpan = pow(2.0,tempSpan-1) ;
  } else {
    fCalcConf->fSpan = kTimeSpanDef ;
  }
  tempSpan = 1 + int(log(fCalcConf->fSpan) / log(2.0));
  fTimeSpanBox -> Select(tempSpan);

   
   
  int tempOverlap = fOverlapBox->GetSelected();
  switch (tempOverlap) {
  case 0:
    fCalcConf->fOverlap = 0.5;
    break;
  case 1:
    fCalcConf->fOverlap = 0.0;
    break;
  case 2:
    fCalcConf->fOverlap = 0.1;
    break;
  case 3:
    fCalcConf->fOverlap = 0.2;
    break;
  case 4:
    fCalcConf->fOverlap = 0.25;
    break;
  case 5:
    fCalcConf->fOverlap = 0.3;
    break;
  case 6:
    fCalcConf->fOverlap = 0.4;
    break;
  case 7:
    fCalcConf->fOverlap = 0.5;
    break;
  default:
    fCalcConf->fOverlap = 0.5;
  }
  if (fCalcConf->fOverlap == 0.5) {
    tempOverlap = 7;
  }
  fOverlapBox -> Select(tempOverlap);


  // Set the Estimated Time
  // Int_t tempBufferTime = 16;
  // Int_t tempCalcTime = int(pow(fCalcConf->fNBins/98.0 ,2));
  // Int_t tempTimeEst = int(fCalcConf->fSpan) - tempBufferTime + tempCalcTime;
  Int_t tempTimeEst = int(4.3e-6 * fCalcConf->fSpan * fCalcConf->fFreqMax * 
			  fCalcConf->fFreqMax / fCalcConf->fFreqRes);
  if (isOnline) {
    tempTimeEst = tempTimeEst + int(fCalcConf->fSpan);
  }
  fLabelCalc[19]->SetText(tempTimeEst);   
   
  // Set the percentage uncertainty
  Int_t tempPerError = 100;
  Float_t tempVar = fCalcConf->fSpan/(1.0-fCalcConf->fOverlap);
  //    Float_t tempVar = (fCalcConf->fSpan*fCalcConf->fRate - 2*fCalcConf->fNBins*fCalcConf->fOverlap)/(1.0-fCalcConf->fOverlap);
  if (tempVar > 0.0) {
    tempPerError = int(100.0 / sqrt(tempVar));
  }
  fLabelCalc[16]->SetText(tempPerError);
   

  // Set Valid Flags
  if (fCalcConf->fNChans > 0 && fCalcConf->fNChans < 4) {
    fCalcConf->fValid = true;
  } else {
    fCalcConf->fValid = false;
    cout << "ERROR - Configuration Invalid.  No Channels Selected." << endl;
  }

  // If START then Update "Current parameters"
  if (fCalcConf->fNew) {
    fLabelCalc[6] -> SetText(kFreqMaxText[tempFreqMax]);
    fLabelCalc[8] -> SetText(kFreqResText[tempFreqRes]);
    fLabelCalc[10] -> SetText(kTimeSpanText[tempSpan]);
    fLabelCalc[12] -> SetText(kOverlapText[tempOverlap]);
    fLabelCalc[17] -> SetText(tempPerError);
    fLabelCalc[20] -> SetText(tempTimeEst);
    fEstPlotTime = tempTimeEst;

    // Restart the Plot Timer
    fPlotTimer = fEstPlotTime;
  }
   
   
  fCalcConf->unlock();
  return kTRUE;
}




// ****************************************************
// *                                                  *
// *   ---  BVWindow:  Read Plot Configuration  ---   *
// *                                                  *
// ****************************************************
Bool_t BVWindow::ReadPlotConf()  
{
  fPlotConf.lock();

  fPlotConf.fZLogScale = (fZScaleBox->GetSelected() == 1);

  fPlotConf.fXLimit[0] = fXLimitBox[0]->GetNumber();
  fPlotConf.fXLimit[1] = fXLimitBox[1]->GetNumber();
  fPlotConf.fYLimit[0] = fYLimitBox[0]->GetNumber();
  fPlotConf.fYLimit[1] = fYLimitBox[1]->GetNumber();

  // ReInitialize Plot
  // newPlot = true;

  // Set New Axis Scale and Limits
  fCvsBico -> cd();
  gPad->SetLogz(1);
  //    gPad->RangeAxis(fPlotConf.fXLimit[0], fPlotConf.fYLimit[0], fPlotConf.fXLimit[1], fPlotConf.fYLimit[1]);
   

  fPlotConf.unlock();
  return kTRUE;
}




// ******************************************************
// *                                                    *
// *   ---  BVWindow:  Read Output Configuration  ---   *
// *                                                    *
// ******************************************************
Bool_t BVWindow::ReadOutConf()  
{
  fOutConf.lock();
  fOutConf.fUseGIFformat = (fFileFormatBox->GetSelected() == 0);
  //    fOutConf.fLimitGIF = fLimitGIFBox->GetNumber();

  fOutConf.unlock();
  return kTRUE;
}


 

// **************************************
// *                                    *
// *   ---  BVWindow:  Draw Plot  ---   *
// *                                    *
// **************************************
Bool_t BVWindow::DrawPlots()
{  
  fCvsBico -> cd();
   
  TimeStr(fPlotDataActive->fFrameStart, frameTimeString, "%s"); 
    
   
#ifdef BV_DEBUG
  cout << " Begin Draw Plot ... "<< endl;
  cout << " Frame Time = "<< fPlotDataActive->fFrameStart << ", Label = " << frameTimeString <<endl;
#endif

   
   
  // Perform Initial Setup & Create New Histograms
  // ---------------------------------------------
  if (newPlot) {
      
    if (firstPlot) {
      firstPlot = false;
    } else {
      delete h_bicoh;
      delete h_psd1;
      delete h_psd2;
      delete h_psd3;
    }
      
      
    //  Read in new config info 
    //  ------------------------
    fCalcConf->lock(); 
    fmin = fCalcConf->fFreqRes;
    fmax = fCalcConf->fFreqMax;
    Nbs = fCalcConf->fNBins;
    NChans = fCalcConf->fNChans;
    sChanNames[1] = fCalcConf->fChanName1;
    sChanNames[2] = fCalcConf->fChanName2;
    sChanNames[3] = fCalcConf->fChanName3;
    fCalcConf->unlock();
      
    // Make the Titles and Axis Labels
    // -------------------------------
    sChanNames[0].erase();        // clear the combined name string
    switch (NChans) {
    case 3: {
      sChanNames[0] = sChanNames[1] + std::string(", ") + sChanNames[2] + std::string(", ") + sChanNames[3] + std::string(":  Fr:");
      sPSD3Title = std::string("PSD: ") + sChanNames[3] + std::string(" .");
      sPSD2Title = std::string("PSD: ") + sChanNames[2] + std::string(" .");
      sPSD1Title = std::string("PSD: ") + sChanNames[1] + std::string(" .");
      sYAxisLabel = sChanNames[2] + std::string(" -- Frequency (Hz)      .");
      sXAxisLabel = sChanNames[1] + std::string(" -- Frequency (Hz)");
      break;
    }
    case 2: {
      sChanNames[0] = sChanNames[1] + std::string(", ") + sChanNames[2] + std::string(":  Fr:");
      sPSD3Title = std::string("PSD: ") + std::string(" .");
      sPSD2Title = std::string("PSD: ") + sChanNames[2] + std::string(" .");
      sPSD1Title = std::string("PSD: ") + sChanNames[1] + std::string(" .");
      sYAxisLabel = sChanNames[2] + std::string(" -- Frequency (Hz)      .");
      sXAxisLabel = sChanNames[1] + std::string(" -- Frequency (Hz)");
      break;
    }
    case 1: {
      sChanNames[0] = sChanNames[1] + std::string(":  Fr:");
      sPSD3Title = std::string("PSD: ") + std::string(" .");
      sPSD2Title = std::string("PSD: ") + std::string(" .");
      sPSD1Title = std::string("PSD: ") + sChanNames[1] + std::string(" .");
      sYAxisLabel = sChanNames[1] + std::string(" -- Frequency (Hz)      .");
      sXAxisLabel = sChanNames[1] + std::string(" -- Frequency (Hz)");
      break;
    }
    default: {
      sChanNames[0] = std::string("No Channels:  Fr:");
      sPSD3Title = std::string("PSD: ") + std::string(" .");
      sPSD2Title = std::string("PSD: ") + std::string(" .");
      sPSD1Title = std::string("PSD: ") + std::string(" .");
      sYAxisLabel = std::string(" -- Frequency (Hz)      .");
      sXAxisLabel = std::string(" -- Frequency (Hz)");
    }
    }

                          
    sBicTitle = std::string("Bico: ") + sChanNames[0] + frameTimeString + std::string(" .");

    sXAxisPSD = std::string("Frequency (Hz)");
    sYAxisPSD = std::string("Power (1/Hz)      .");
      
    h_bicoh  = new TH2F("h_bicoh",sBicTitle.c_str(),Nbs,fmin,fmax,Nbs,fmin,fmax);
    h_psd1   = new TH1F("h_psd1",sPSD1Title.c_str(),Nbs,fmin,fmax);
    h_psd2   = new TH1F("h_psd2",sPSD2Title.c_str(),Nbs,fmin,fmax);
    h_psd3   = new TH1F("h_psd3",sPSD3Title.c_str(),Nbs,fmin,fmax);
      
    h_bicoh->GetXaxis()->SetTitle(sXAxisLabel.c_str());
    h_bicoh->GetYaxis()->SetTitle(sYAxisLabel.c_str());
    h_bicoh->SetLabelFont(132, "X");
    h_bicoh->SetLabelFont(132, "Y");
    h_bicoh->SetLabelFont(132, "Z");
    h_bicoh->SetLabelSize(0.03, "X");
    h_bicoh->SetLabelSize(0.03, "Y");
    h_bicoh->SetLabelSize(0.03, "Z");
    h_bicoh->SetTitleSize(0.03,"X");
    h_bicoh->SetTitleSize(0.03,"Y");
    h_bicoh->SetTitleSize(0.03,"Z");
         
    h_psd1->GetXaxis()->SetTitle(sXAxisPSD.c_str());
    h_psd1->GetYaxis()->SetTitle(sYAxisPSD.c_str());
    h_psd1->SetLabelFont(132, "X");
    h_psd1->SetLabelFont(132, "Y");
    h_psd1->SetLabelSize(0.06, "X");
    h_psd1->SetLabelSize(0.06, "Y");
    h_psd1->SetTitleSize(0.05,"X");
    h_psd1->SetTitleSize(0.05,"Y");
      
    h_psd2->GetXaxis()->SetTitle(sXAxisPSD.c_str());
    h_psd2->GetYaxis()->SetTitle(sYAxisPSD.c_str());
    h_psd2->SetLabelFont(132, "X");
    h_psd2->SetLabelFont(132, "Y");
    h_psd2->SetLabelSize(0.06, "X");
    h_psd2->SetLabelSize(0.06, "Y");
    h_psd2->SetTitleSize(0.05,"X");
    h_psd2->SetTitleSize(0.05,"Y");
      
    h_psd3->GetXaxis()->SetTitle(sXAxisPSD.c_str());
    h_psd3->GetYaxis()->SetTitle(sYAxisPSD.c_str());
    h_psd3->SetLabelFont(132, "X");
    h_psd3->SetLabelFont(132, "Y");
    h_psd3->SetLabelSize(0.06, "X");
    h_psd3->SetLabelSize(0.06, "Y");
    h_psd3->SetTitleSize(0.05,"X");
    h_psd3->SetTitleSize(0.05,"Y");

      
    h_bicoh->SetMaximum(0.0);       //fix the scale
    h_bicoh->SetMinimum(1.0);       //fix the scale
      
    gStyle->SetPalette(1); 
    gStyle->SetOptStat(0);         // no stats box 
    //  gStyle->SetOptStat(11);    // stats box has hist name and # entries
      
    //   If the plots are old, just change the title,
  } else {
    sBicTitle = std::string("Bico: ") + sChanNames[0] + frameTimeString + std::string("   .");
    h_bicoh->SetNameTitle("h_bicoh",sBicTitle.c_str());
  }
   
         
   
  // Fill the Histograms
  // -------------------
#ifdef BV_DEBUG
  cout << " Filling Histograms ..."<<endl;
#endif

  //    float max_z_bc = h_bicoh->GetMaximum() ;  // Preserve old value
  float max_z_bc = 1.0 ;  
  float min_z_bc = h_bicoh->GetMinimum();  // Preserve old value

   
#ifdef BV_DEBUG
  for (Int_t ix=0; ix<5; ++ix) {    
    cout << "fPSD1[" << ix << "] = " << fPlotDataActive->fPSD1[ix]<<endl;
    cout << "fPSD2[" << ix << "] = " << fPlotDataActive->fPSD2[ix]<<endl;
    cout << "fPSD3[" << ix << "] = " << fPlotDataActive->fPSD3[ix]<<endl;

    cout << endl;
    cout <<  "fBic[" << ix << ", 1-5] = ";

    for (Int_t iy=0; iy<5; ++iy) {
      Int_t ixy = Nbs*ix + iy;
      cout << fPlotDataActive->fBic[ixy]<<"  ";
    }
      
    cout << endl;
      
  }
#endif

   
   
  for (Int_t ix=0; ix<Nbs; ++ix) {    
    Int_t jx = ix + 1;
    Int_t xbin = h_psd1->GetBin(jx);
    h_psd1->SetBinContent(xbin, fPlotDataActive->fPSD1[ix]) ;
    h_psd2->SetBinContent(xbin, fPlotDataActive->fPSD2[ix]) ;
    h_psd3->SetBinContent(xbin, fPlotDataActive->fPSD3[ix]) ;
    for (Int_t iy=0; iy<Nbs; ++iy) {
      Int_t ixy = Nbs*ix + iy;
      Int_t jy = iy + 1;
      Int_t xybin = h_bicoh->GetBin(jx,jy);
      h_bicoh->SetBinContent(xybin, fPlotDataActive->fBic[ixy]) ;
      max_z_bc = (fPlotDataActive->fBic[ixy] > max_z_bc) ? fPlotDataActive->fBic[ixy] : max_z_bc ;
      min_z_bc = (fPlotDataActive->fBic[ixy] < min_z_bc) ? fPlotDataActive->fBic[ixy] : min_z_bc ;
    }
  }

     
  // Set Plot Limits
  // ---------------
  float max_lim_bc = (max_z_bc > 1.0) ? max_z_bc : 1.0 ;
  float min_lim_bc ;
   
  if (fPlotConf.fZLogScale) {
    min_lim_bc = (min_z_bc > 0.0) ? 0.8*min_z_bc : 0.001*max_lim_bc ;
  } else {
    min_lim_bc = 0.0;
  }
   
  // Temporary fix
  max_lim_bc = 1.0;
   
  h_bicoh->SetMaximum(max_lim_bc);  //fix the scale
  h_bicoh->SetMinimum(min_lim_bc);  //fix the scale
   
  if (newPlot) {
    if (max_z_bc > 1.0) {
      cout << "WARNING: Bicoherence Range = ("<<min_z_bc<<", "<<max_z_bc<<")"<<endl;
    }
      
    if (fPlotConf.fZLogScale) {
      fCvsBico -> cd();
      gPad->SetLogz(1);
    } else {
      fCvsBico -> cd();
      gPad->SetLogz(0);
    }
    fCvsPSD1 -> cd();
    gPad->SetLogy(1);
    fCvsPSD2 -> cd();
    gPad->SetLogy(1);
    fCvsPSD3 -> cd();
    gPad->SetLogy(1);
    //       newPlot = false;  // Reset flag after initial pass
  }
   
#ifdef BV_DEBUG
  cout << " Update Plots ..."<<endl;
#endif

  // Update the Bicoherence Plot
  // ---------------------------
  fCvsBico -> cd();
  h_bicoh->Draw("COLZ");
  fCvsBico->Update();


  // Update the Power Spectra Plots
  // ------------------------------
  switch (NChans) {
  case 3:
    fCvsPSD3 -> cd();
    h_psd3->Draw("C");
    fCvsPSD3->Update();
  case 2:
    fCvsPSD2 -> cd();
    h_psd2->Draw("C");
    fCvsPSD2->Update();
  case 1:
    fCvsPSD1 -> cd();
    h_psd1->Draw("C");
    fCvsPSD1->Update();
  }
   
   
  // Write the Plots to Disk
  // -----------------------
  if (fOutConf.fWriteBico) {
      
    Int_t fError = 0;
      
      
    if (newPlot) {
      fPlotNumber = 0;
         
      TimeStr(fPlotDataActive -> fFrameStart, frameTimeString, "%s"); 
      sOutDirName = std::string("BV_") + frameTimeString + std::string("/");

#ifdef BV_DEBUG
      cout << " Plot Directory = "<< sOutDirName << endl;
#endif
         
      fError = mkdir(sOutDirName.c_str(), S_IRWXU|S_IRWXG|S_IRWXO);
    }
      
         
    if (fError < 0) {
      cerr << "Error creating directory " << sOutDirName.c_str() << endl;
      sOutDirName = std::string("BV_");
    }
      
      
    fPlotNumber++ ;
    TGString sPlotNumber(fPlotNumber);
    sOutFileName = sOutDirName + std::string("BV_") + sPlotNumber.GetString() + std::string(".gif");

#ifdef BV_DEBUG
    cout << " Plot File = "<< sOutFileName.c_str() << endl;
#endif

    fCvsBico -> Print(sOutFileName.c_str());
  }
   
  // Unset NewPlot flag    
  newPlot = false;  
   
  return kTRUE;
}




// ******************************************************
// *                                                    *
// *   ---  BVWindow:  Process GUI Messages  ---   *
// *                                                    *
// ******************************************************
// 
// ADD BUTTON TO UPDATE CONFIG AND ONE TO REPLOT
// 
Bool_t BVWindow::ProcessMessage (Long_t msg, Long_t parm1, Long_t parm2)
{
  // BUTTONS
  // -------
  // EButtonState {kButtonUp, kButtonDown, kButtonEngaged, kButtonDisabled}

  if ((GET_MSG (msg) == kC_COMMAND) && (GET_SUBMSG (msg) == kCM_BUTTON)) {
    switch (parm1) {
    case kButtonStart:   // Start Button
      {
	// fControlButton[0]->SetText(new TGHotString (kControlButtonText[3]));  // Not Written
            
            
	// Change Button State
	fControlButton[0]->SetState (kButtonDisabled);       // Disable START temporarily
	fControlButton[1]->SetState (kButtonUp);             // Enable PAUSE 
	fControlButton[0]->SetText (kControlButtonText[1]);  // Change label to PAUSE
            
	// Set "New" Flag to indicate START
	fCalcConf->lock();
	fCalcConf->fNew = true; 
	fCalcConf->unlock();
            
	// Read Configuration Panel
	ReadCalcConf();
	ReadPlotConf();
	ReadOutConf();
            
	// Valid Configuration ==> START
	fCalcConf->lock();
	if (fCalcConf->fValid) {
	  fCalcConf->fNew = true; 
	  fCalcConf->fPause = false;
	  newPlot = true;
	  fControlButton[0]->SetText (kControlButtonText[4]);  // Change label to RESTART
	} else { 
	  fCalcConf->fNew = false; 
	  fCalcConf->fPause = true;
	  newPlot = false;
	  fControlButton[0]->SetText (kControlButtonText[1]);  // Change label back to START
	  fControlButton[1]->SetState (kButtonDisabled);       // Disable PAUSE 
	}
	fControlButton[0]->SetState (kButtonUp);             // Enable RESTART 
	//             fPaused = fCalcConf->fPause;
	fCalcConf->unlock();
	break;
      }
         
    case kButtonPause:   // Pause Button
      {
	fCalcConf->lock();
	if (fCalcConf->fPause) {
	  fControlButton[1]->SetText (kControlButtonText[1]);  // Change Label RESUME => PAUSE
	  fCalcConf->fPause = false;
	} else {
	  fControlButton[1]->SetText (kControlButtonText[3]);  // Change Label PAUSE => RESUME
	  fCalcConf->fPause = true;
	}
	//  fCalcConf->fPause = !fCalcConf->fPause;   //  fCalcConf->fPause = true;
	//            fPaused = fCalcConf->fPause;
	fCalcConf->unlock();
	break;
      }
    case kButtonQuit:    // Quit Button
      {
	CloseWindow();
	break;
      }
    case kButtonWriteNow:   // WriteNow Button
      {
	fButtonWriteNow -> SetState (kButtonDown);
	fButtonWriteNow -> SetState (kButtonDisabled);
            
	// Time frameStartTime = tDataDec1->getStartTime();
	char frameTimeString[32];
	TimeStr(fPlotDataActive -> fFrameStart, frameTimeString, "%s"); 
	std::string gifFileName = std::string("BV_") + frameTimeString + std::string(".gif");
	fCvsBico->Print(gifFileName.c_str());
	break;
            
	fButtonWriteNow -> SetState (kButtonUp);
	fButtonWriteNow -> SetState (kButtonEngaged);
      }
    }
  }
   
    
  // COMBOBOXES
  // ----------
  else if ((GET_MSG (msg) == kC_COMMAND) && (GET_SUBMSG (msg) == kCM_COMBOBOX)) {
    switch (parm1) {
    case kChanSel1:      // Channel Selectors
      {
	ReadCalcConf();
	break;
      }
    case kChanSel2:
      {
	ReadCalcConf();
	break;
      }
    case kChanSel3:
      {
	ReadCalcConf();
	break;
      }
    case kFreqMax:        // Maximum Frequency 
      {
	ReadCalcConf();
	break;
      }
    case kFreqRes:       // Frequency Resolution
      {
	ReadCalcConf();
	break;
      }
    case kTimeSpan:       // Data time span
      {
	ReadCalcConf();
	break;
      }
    case kOverlap:       // Percentage overlap (No_Window ==> 0% overlap, Window ==> 50% overlap)
      {
	ReadCalcConf();
	break;
      }
    case kWindowTypeBox:       // Types of Windowing
      {
	ReadCalcConf();
	break;
      }
    case kZScale:         // Z Axis Scaling
      {
	ReadOutConf();
	break;
      }
    }
  }
   

  // CHECK BUTTONS
  // -------------
  else if ((GET_MSG (msg) == kC_COMMAND) && (GET_SUBMSG (msg) == kCM_CHECKBUTTON)) {
    switch (parm1) {
      //        case kWindowCheck:
      //          {
      //             fCalcConf->fWindow2D = ! fCalcConf->fWindow2D ;
      //             // ReadConf();
      //             break;
      //          }
    case kWriteBico:
      {
	fOutConf.fWriteBico = ! fOutConf.fWriteBico;
	ReadOutConf();
	break;
      }
    }
  }
                       
  // TEXT BOXES
  // ----------
  else if ((GET_MSG (msg) == kC_TEXTENTRY) && (GET_SUBMSG (msg) == kTE_TEXTUPDATED)) {
    switch (parm1) {
      //        case kLimitGIF:
      //          {
      //             ReadOutConf();
      //             break;
      //          }
    case kYLimitMin:
      {
	if (!firstPlot) {
	  ReadPlotConf();
	}
	break;
      }
    case kYLimitMax:
      {
	if (!firstPlot) {
	  ReadPlotConf();
	}
	break;
      }
    case kXLimitMin:
      {
	if (!firstPlot) {
	  ReadPlotConf();
	}
	break;
      }
    case kXLimitMax:
      {
	if (!firstPlot) {
	  ReadPlotConf();
	}
	break;
      }
    }
  }
  return kTRUE;
}




// ****************************************************************
// *                                                              *
// *   ---  BVWindow:  Handle Timer and Data Exchange  ---   *
// *                                                              *
// ****************************************************************
Bool_t BVWindow::HandleTimer (TTimer* timer)
{
   
   
  fCalcConf->lock();
  bool testPause = fCalcConf->fPause;
  fCalcConf->unlock();
      
  if (testPause) {
    if (!fPaused) {
      fLabelCtrl[2] -> SetText("PAUSED");
    }
  } else {
    if (fPaused) {
      fRefreshGUItime = Now().getS();
    }
    Int_t fCountdown = fPlotTimer - (Now().getS() - fRefreshGUItime) ;
    fLabelCtrl[2] -> SetText(fCountdown);
  }
  fPaused = testPause;

   
  // sometimes skip a heartbeat to avoid overloading the GUI
  if (fSkipHeartbeats) {
    fSkipHeartbeats--;
    timer->Reset();
    return kTRUE;
  }
   
   
  // Check if there is New Plot Data
  // -------------------------------
   
  if (fMonitor->fNewPlot) {
    // Alert User Plot in Progress
    fLabelCtrl[2] -> SetText(kPlottingMessage);
    //       fLabelCtrl[2] -> DoRedraw();
      
#ifdef BV_DEBUG
    cout << " New Plot: Switch Array pointers"<<endl;
#endif
      
    // Lock Data Arrays
    fPlotDataActive->lock(); 
    //       fPlotDataQueue->lock();
    
    // Switch Active and Queue Arrays
    //       if (fMonitor->fActivePlotIsData1) {
    //          fPlotDataActive = &fMonitor->fPlotData2;
    //          fPlotDataQueue = &fMonitor->fPlotData1;
    //       } else {
    //          fPlotDataActive = &fMonitor->fPlotData1;
    //          fPlotDataQueue = &fMonitor->fPlotData2;
    //       }

      
#ifdef BV_DEBUG
    cout << " Before Plot ..."<<endl;
#endif
     
      
    BVWindow::DrawPlots ();
      
#ifdef BV_DEBUG
    cout << " After Plot ..."<<endl;
#endif

    //       fMonitor->fActivePlotIsData1 = !fMonitor->fActivePlotIsData1 ;
    fMonitor->fNewPlot = false;

    fPlotDataActive->unlock(); 
    //       fPlotDataQueue->unlock();
      
    // Restart the Plot Timer
    fPlotTimer = fEstPlotTime;
    fRefreshGUItime = Now().getS();

  }
   
   
  // reset timer and return
  fSkipHeartbeats = 3;
  timer->Reset();
  return kTRUE;
}



// ************************************************
// *                                              *
// *   ---  GUI Thread:  Initiates the GUI  ---   *
// *                                              *
// ************************************************

extern "C" {
  static void* guithread (void*)
  {
#ifdef BV_DEBUG
    cout << " Initiate New Thread ..."<<endl;
#endif
    gApplication->Run();
    return 0;
  }
}

// *************************************************
// *                                               *
// *   ---  BicoViewer:  Viewer Constructor  ---   *
// *                                               *
// *************************************************
BicoViewer::BicoViewer (int argc, const char* argv[])
  : DatEnv (argc, argv), fMainWindow (0), 
    //         fCalcData (&fPlotData2),
    //         fBic_Data1 (0), fBic_Data2 (0), 
    //         fPSD1_Data1 (0), fPSD1_Data2 (0), 
    //         fPSD2_Data1 (0), fPSD2_Data2 (0), 
    //         fPSD3_Data1 (0), fPSD3_Data2 (0), 
    //         fActivePlotIsData1 (true), fCalcDataIsData1 (false),
    fNewPlot (false),
    fFirstTime (true), iFrame (0),
    tData1 (0), tData2 (0), tData3 (0)
{
#ifdef BV_DEBUG
  cout << " Create BicoViewer ..."<<endl;
#endif
  iFrame = 0 ;
  std::string BVWindowTitle = "Bicoherence Viewer";
   
      
  // Create Main Window
  // ------------------
#ifdef BV_DEBUG
  cout << " Create BVWindow ..."<<endl;
#endif
  fMainWindow = new BVWindow (gClient->GetRoot(), 1150, 750,
			      *this, BVWindowTitle.c_str());
  fMainWindow->Resize (TGDimension (1152, 752));
      
      
  // Create GUI Thread
  // -----------------
  pthread_t      		tid;
  pthread_attr_t    	tattr;
  struct sched_param	schedprm;
  int         			status;

  // Set Thread Parameters: Detached & Process Scope
  if (pthread_attr_init (&tattr) != 0) {
    cerr << "Unable to create GUI thread" << endl;
    exit (0);
  }
  pthread_attr_setdetachstate (&tattr, PTHREAD_CREATE_DETACHED);
  pthread_attr_setscope (&tattr, PTHREAD_SCOPE_PROCESS);

  // Set Low Priority
  pthread_attr_getschedparam (&tattr, &schedprm);
  schedprm.sched_priority = 10;
  pthread_attr_setschedparam (&tattr, &schedprm);

  // Create Thread
  status = pthread_create (&tid, &tattr, guithread, 0);
  pthread_attr_destroy (&tattr);
  if (status != 0) {
    cerr << "Unable to create GUI thread" << endl;
    exit (0);
  }
   
   
   
}






// ************************************************
// *                                              *
// *   ---  BicoViewer:  Viewer Destructor  ---   *
// *                                              *
// ************************************************
BicoViewer::~BicoViewer()
{
#ifdef BV_DEBUG
  cout << "BicoViewer Destructor " << endl;
#endif

  // Delete the Data Arrays
  // ----------------------
  fPlotData1.DeleteArrays();
  //    fPlotData2.DeleteArrays();
   

  //getDacc().close();
  gApplication->Terminate(0);
   
}




// ********************************************
// *                                          *
// *   ---  BVPlotData:  Delete Arrays  ---   *
// *                                          *
// ********************************************
void BVPlotData::DeleteArrays()
{   
  // Delete the Bicoherence and PSD Data Arrays
  // ------------------------------------------
  delete[] fBic;

  delete[] fPSD1;
  delete[] fPSD2;
  delete[] fPSD3;
}




// ********************************************************
// *                                                      *
// *   ---  BicoViewer:  Frame Processing Function  ---   *
// *                                                      *
// ********************************************************
void BicoViewer::ProcessData()
{  
  iFrame += 1 ;  // update the frame counter
  //    cout << " Entering ProcessData, Frame = "<<iFrame<<endl;
   
  //fOnline = getDacc().isOnline();     // Test if Dacc input is Online  
  fOnline = true;

#ifdef BV_DEBUG
  if (fOnline) {
    cout << " Processing Online Data ..."<<endl;
  } else {
    cout << "Processing File Data ..."<<endl;
  }     
#endif
   
   
    
  // TEST FOR VALID CONFIGURATION
  // ----------------------------
  fBicoConf.fNew = false;
  fCalcConf.lock();
   
  if (fOnline) {
    if (fCalcConf.fPause) {         // Process Paused ... skip to next frame if online
      fCalcConf.unlock();
      fBicoConf.fPause = true;
      return;
    } else if (!fCalcConf.fValid) {
      if (fCalcConf.fNew) {
	//       #ifdef BV_DEBUG
	cout << "ERROR - Current configuration is Not Valid." << endl;
	//       #endif
	fCalcConf.fNew = false;
      }
      fCalcConf.unlock();
      fBicoConf.fValid = false;
      return;
    }
      
  } else {
    while (fCalcConf.fPause) {      // Process Paused ... wait for START if not online
      fCalcConf.unlock();
      sleep(1);                  // Allow time for the Configuration to change
      fCalcConf.lock();
    }
    fBicoConf.fPause = false;
      
    while (!fCalcConf.fValid) {      // Configuration Invalid ... wait for valid config
      if (fCalcConf.fNew) {
	//       #ifdef BV_DEBUG
	cout << "ERROR - Current configuration is Not Valid." << endl;
	//       #endif
	fCalcConf.fNew = false;
      }
      fCalcConf.unlock();
      sleep(1);                   // Allow time for the Configuration to change
      fCalcConf.lock();
    }
    fBicoConf.fValid = true;
  }
                  
  if (fCalcConf.fNew) {               // New Configuration
    fBicoDir.SetConfig(fCalcConf);		// Send new settings to BicoDir
    fBicoConf = fBicoDir.GetConfig();	// Get full configuration from BicoDir
    fCalcConf.fNew = false;					// Reset window "new config" flag
  }
  fCalcConf.unlock();
   
 
   
  // NEW CONFIG ==> REALLOCATE DATA OBJECTS & CHANGE DATA CHANNELS
  // -------------------------------------------------------------
  if (fBicoConf.fNew) { 
      
    iFrame = 0 ;   // restart frame counter
      
    // Remove Old Data Objects, Unless First Pass
    // ------------------------------------------
    if (fFirstTime) {         
      fFirstTime = false ;
    } else {        
      // Remove the Old Channels
      switch (fBicoConf.fNChans) {
      case 3:
	getDacc().rmChannel(tData3->getName());
      case 2:
	getDacc().rmChannel(tData2->getName());
      case 1:
	getDacc().rmChannel(tData1->getName());
	break;
      default:
	cout<<"ERROR: Old Channel Number not 1-3"<<endl;
	finish(); 
	return;
      }
      // Delete the Data Arrays         
      fPlotData1.DeleteArrays();
      // fPlotData2.DeleteArrays();       
    }    
     
          
    // Add Data Channels and Set the Stride
    // ------------------------------------
    // 
    //    NOTE:  After the channels are added, ProcessData must be allowed
    //    to cycle to the next frame because the new FFT & PSD containers are 
    //    created from the new time series.
    //      BicoMon::SetParameters();

    // Add Channels to Dacc with Decimation
    // ---------------------------------------------
    switch (fBicoConf.fNChans) {
    case 3:  
      getDacc().addChannel(fBicoConf.fChanName3.c_str(), 1, &tData3);
#ifdef BV_DEBUG
      cout<<"AddChannel:  Ch 3 = : "<<fBicoConf.fChanName3<<endl;
#endif
    case 2:
      getDacc().addChannel(fBicoConf.fChanName2.c_str(), 1, &tData2);
#ifdef BV_DEBUG
      cout<<"AddChannel:  Ch 2 = : "<<fBicoConf.fChanName2<<endl;
#endif
    case 1:
      getDacc().addChannel(fBicoConf.fChanName1.c_str(), 1, &tData1);
#ifdef BV_DEBUG 
      cout<<"AddChannel:  Ch 1 = : "<<fBicoConf.fChanName1<<endl;
#endif
      break;
    default: 
      cout<<"ERROR: Channel Number not 1-3"<<endl;
      finish(); 
      return;
    }


    // Set the Stride
    // --------------
    getDacc().setStride(fBicoConf.fStride);
    return ;
  }
   
   
   
  // Create the Data Containers and Arrays
  // ------------------------------------
  if (iFrame == 1) {
    fPlotData1.CreateArrays(fBicoConf.fNBins) ;     
    //       fPlotData2.CreateArrays(fBicoConf.fNBins) ;           
    return ;  // Next Frame starts standard process
  }  
      
   
   
  // If Plot Queue is full, skip frame
  // ---------------------------------
  if (fNewPlot && fOnline) {
    return;
  }
   
   

  // STANDARD PROCESS
  // ----------------
  // Calculate the bispectrum & bicoherence for each group of frames 

#ifdef BV_DEBUG
  Time_t start_of_calc;
  Time_t end_of_calc;
  start_of_calc = time(NULL);
  cout<<"  Start Clock  ... "<< start_of_calc << endl;
#endif
   

      
  // Test Data Validity for Bicoherence Class
  // ----------------------------------------
  //    // Test if Dacc input is Online  
  //    // ----------------------------
  //    fOnline = getDacc().isOnline();
  if (!fOnline) {
    if (!fBicoDir.TestData(fBicoConf.fNChans, tData1, tData2, tData3)) {
      cout << "ERROR: Data is not Valid. Skipping to Next Data Block." << endl;
      return;
    }
  }


   
  // Send Data to Bicoherence Class
  // ------------------------------
  switch (fBicoConf.fNChans) {
  case 3:  
    fBicoDir.SetData(3, tData3, fOnline);
#ifdef BV_DEBUG
    cout<<"SetData:  Ch 3 = : "<<fBicoConf.fChanName3<<endl;
#endif
  case 2:
    fBicoDir.SetData(2, tData2, fOnline);
#ifdef BV_DEBUG
    cout<<"SetData:  Ch 2 = : "<<fBicoConf.fChanName3<<endl;
#endif
  case 1:
    fBicoDir.SetData(1, tData1, fOnline);
#ifdef BV_DEBUG
    cout<<"SetData:  Ch 1 = : "<<fBicoConf.fChanName3<<endl;
#endif
    break;
  default: 
    cout<<"ERROR: Channel Number not 1-3"<<endl;
    finish(); 
    return;
  }

   

  // Bicoherence Calculation
  // -----------------------
  fBicoDir.Calculate();
   
   
  // Check full buffer, swap pointers if needed, and fill arrays
  // -----------------------------------------------------------
  fPlotData1.lock();
   
  if (!fNewPlot) {
    //       if (fCalcDataIsData1) {
    //          fCalcData = &fPlotData1;
    //       } else {
    //          fCalcData = &fPlotData2;
    //       }
    //       
    fBicoDir.CopyBicoData(fPlotData1.fBic);
    //       fBicoDir.CopyStartTime(fPlotData1.fFrameStart);
    fPlotData1.fFrameStart = fBicoDir.GetStartTime();
      
      
#ifdef BV_DEBUG
    //          cout<<"Frame Start Time = "<< fBicoDir.fFrameStartTime << endl;
    cout<<"Frame Start Time = "<< fPlotData1.fFrameStart << endl;
#endif
      
    switch (fBicoConf.fNChans) {
    case 3:
      fBicoDir.CopyPSDData(3,fPlotData1.fPSD3);	
          
    case 2:
      fBicoDir.CopyPSDData(2,fPlotData1.fPSD2);	
         
    case 1:
      fBicoDir.CopyPSDData(1,fPlotData1.fPSD1);	
      break;
         
    default:
      cout<<"ERROR: Channel Number not 1-3"<<endl;
      finish(); 
      return;
      
    }
      
    //       fCalcDataIsData1 = !fCalcDataIsData1;
    fNewPlot = true;
  }
  fPlotData1.unlock();
   
  
  // Check Calculation Time
  // ----------------------
#ifdef BV_DEBUG
  end_of_calc = time(NULL);
  float calc_time = difftime(end_of_calc, start_of_calc);
  cout<<"Elapsed Calc Time = "<<end_of_calc<<" - "<<start_of_calc<<" = "<<calc_time<<endl;
#endif
   
   
}





// *************************************************
// *                                               *
// *   ---  BVPlotData:  Create Data Arrays  ---   *
// *                                               *
// *************************************************
void BVPlotData::CreateArrays(const int& fNBins)
{
  // To create the data arrays we require information from the requested data 
  // channels. Rather than create and delete these objects with each frame, 
  // we sample the data channels and create thse objects once.
  // 

   
  // Create the Bicoherence Data Arrays
  // ----------------------------------
  fBic = new float[fNBins*fNBins];


  // Create the Power Spectrum Data Arrays
  // -------------------------------------
  fPSD1 = new float[fNBins];
  fPSD2 = new float[fNBins];
  fPSD3 = new float[fNBins];
  
}
