/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_XOBJ_HH
#define XSIL_XOBJ_HH
#include <time.h>
#include <string>

namespace xsil {
    class Xwriter;

    /**  The %xobj class contains the name and type fields common to (almost) 
     *  all of the XSil data objects. It also provides a standard interface 
     *  to put objects of different types into tree structures, scan the trees
     *  and print the XML document defined by the tree.
     *  @memo %XSIL object base class
     *  @author J. Zweizig
     *  @version 1.1; Modified December 13, 1999
     *  @ingroup IO_xsil
     */
    class xobj {
    public:

	/**  Default (null) constructor.
	 *  @memo Null constructor.
	 */
	xobj(void);

	/**  Data constructor
	 *  @memo Data constructor
	 *  @param Name Pointer to the object name string
	 *  @param Type Pointer to the object type string
	 */
	xobj(const char* Name, const char* Type=0);

	/**  Object destructor.
	 *  @memo Object destructor.
	 */
	virtual ~xobj(void) {}

	/**  Write the Object and any embedded object to an XML file.
	 *  @memo Write the object to an XML file
	 *  @param ostr XML output file descriptor.
	 */
	virtual void Spew(xsil::Xwriter& ostr) const=0;

	/**  Get the object type
	 *  @memo Get the object type
	 *  @return pointer to the object type string.
	 */
	virtual const char* getObjType(void) const=0;

	/**  Create an identical copy(clone) of the current object.
	 *  @memo Clone an object
	 *  @return Pointer to the newly created clone
	 */
	virtual xobj* Clone(void) const=0;

	/**  Test whether object is a container for generic object API. The
	  *  default %xobj code returns false. This must be overridden by
	  *  container classes.
	  *  \brief Test if this is a container.
	  *  \return True if the object is a container.
	  */
	virtual bool container(void) const;

	/**  Find object with the specified name and type inside a container. 
	  *  The first object in the container matching the specified type and
	  *  name is returned. If the type or name are specified as an empty 
	  *  string, any type or name is considered to match. Find returns a 
	  *  null pointer if the object is not a container or if the specified
	  *  name isn't found. The default %xobj implementation returns a NULL
	  *  for all names. This should be overridden by any container object.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Constant pointer to the requested object or NULL.
	  */
	virtual const xobj* find(const std::string& nm, 
				 const std::string& ty="") const;

	/**  Find object with the specified name and type inside a container.
	  *  The first object in the container matching the specified type and
	  *  name is returned. If the type or name are specified as an empty 
	  *  string, any type or name is considered to match. Find returns a 
	  *  null pointer if the object is not a container or if the specified
	  *  name isn't found. The default %xobj implementation returns a NULL
	  *  for all names. This should be overridden by any container object.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	virtual xobj* find(const std::string& nm, 
			   const std::string& ty="");

	/**  Find object recursively in a container.
	  *  @memo find an object recursively.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	virtual const xobj* findr(const std::string& nm, 
				  const std::string& ty="") const;

	/**  Find object recursively in a container.
	  *  @memo find an object recursively.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	virtual xobj* findr(const std::string& nm, 
			    const std::string& ty="");

	/**  Get the object name
	  *  @memo Get the object name
	  *  @return Pointer to the object type name
	  */
	const char* getName(void) const;

	/**  Get the object type field
	  *  @memo Get the object type field
	  *  @return Pointer to the object type string.
	  */
	const char* getType(void) const;

	/**  Get the object name
	 *  @memo Get the object name
	 *  @return Reference to the object name string.
	 */
	const std::string& refName(void) const;

	/**  Get the object type field
	 *  @memo Get the object type field
	 *  @return Reference to the object type string.
	 */
	const std::string& refType(void) const;

	/**  Set the object name
	 *  @memo Set the object name
	 *  @param Name Pointer to the object type name
	 */
	void setName(const char* Name);

	/**  Set the object type field.
	 *  @memo Set the object type
	 *  @param Type Pointer to the object type string.
	 */
	void setType(const char* Type);

    private:
	std::string mName;
	std::string mType;
    };

    //--------------------------------------  Inline functions
    inline const std::string&
    xsil::xobj::refName(void) const {
	return mName;
    }

    inline const std::string&
    xsil::xobj::refType(void) const {
	return mType;
    }

}

#endif  //  XSIL_XOBJ_HH
