/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "xsil/Xwriter.hh"
#include "xsil/array.hh"
#include <stdexcept>

using namespace std;

namespace xsil {

   //===================================  Default constructor
   array::array(void) {
   }

   //===================================  Data constructor
   array::array(const char *Name, const char* Type, const char* Unit) 
     : xobj(Name, Type) 
   {
      if (Unit) setUnit(Unit);
   }

   //===================================  Destructor
   array::~array(void) {
      mDim.erase(mDim.begin(), mDim.end());
   }

   //-----------------------------------  Spew out the array object
   void
   array::Spew(Xwriter& xout) const {
      const int   nattr=3;
      const char* attnames[]={"Name","Type","Unit"};
      const char* attstrings[]={getName(), getType(), mUnit.c_str()};

      xout.Tag(getObjType(), nattr, attnames, attstrings);
      xout.endLine();
      for (int i=0 ; i<getNDim() ; i++) {
        mDim[i].Spew(xout);
      }
      mData.Spew(xout);
      xout.endTag(getObjType());
   }

   //===================================  Return object name
   array* 
   array::Clone(void) const {
      array* ptr = new array(getName(), getType(), getUnit());
      for (int i=0 ; i<getNDim() ; i++) {
         ptr->addDim(mDim[i]);
      }
      ptr->setStream(mData);
      return ptr;
   }

   //===================================  Return object name
   const char* 
   xsil::array::getObjType(void) const {
      return "Array";
   }

   //===================================  Get double data
   void
   array::getData(vector<double>& vf) {
      int N= getTotLen();
      vf.resize(N);
      if (mData.read(&vf[0], N)) throw runtime_error("array::getData: EOF");
   }

   //===================================  Get float data
   void
   array::getData(vector<float>& vf) {
      int N= getTotLen();
      vf.resize(N);
      if (mData.read(&vf[0], N)) throw runtime_error("array::getData: EOF");
   }

   //===================================  Get int data
   void
   array::getData(vector<int>& vf) {
      int N= getTotLen();
      vf.resize(N);
      if (mData.read(&vf[0], N)) throw runtime_error("array::getData: EOF");
   }

   //===================================  Get string data
   void
   array::getData(vector<string>& vf) {
      int N= getTotLen();
      vf.resize(N);
      if (mData.read(&vf[0], N)) throw runtime_error("array::getData: EOF");
   }

   //===================================  Get string data
   int
   array::getTotLen(void) const {
      int N=getNDim();
      int tot(1);
      for (int i=0; i<N; ++i) {
         tot *= mDim[i].getDim();
      }
      return tot;
   }

   //===================================  Set the units field
   void 
   xsil::array::setUnit(const char* unit) {
     mUnit = unit;
   }

   //===================================  Add dimension
   int 
   array::addDim(const char* Name, int N) {
      return addDim(dim(Name, N));
   }

   //===================================  Add dimension
   int 
   array::addDim(const dim& x) {
      mDim.push_back(x);
      return getNDim();
   }

   //====================================  Set the data stream
   void 
   array::setStream(const Stream& istr) {
      mData = istr;
   }

   //====================================  Make a remote data stream
   void 
   array::setRemote(const char* File) {
      mData.setRemote(File);
   }

   //===================================  Fill stream with N ints
   void 
   array::FillData(int N, int* Data) {
      mData.Fill(N, Data, getDim(getNDim()-1));
   }

   //===================================  Fill data stream with N floats
   void 
   array::FillData(int N, float* Data) {
      mData.Fill(N, Data, getDim(getNDim()-1));
   }

   //===================================  Fill data stream with N doubles
   void 
   array::FillData(int N, double* Data) {
      mData.Fill(N, Data, getDim(getNDim()-1));
   }

   //====================================  Fill data stream with N strings
   void 
   array::FillData(int N, std::string* Data) {
      mData.Fill(N, Data, getDim(getNDim()-1));
   }

}  //   namespace xsil
