#ifndef XSIL_XREADER_HH
#define XSIL_XREADER_HH

#include <iosfwd>
#include <string>
#include "lxr.hh"
#include "xsil/xsil.hh"
#include "xsil/attrList.hh"

namespace xsil {

  class array;
  class ligolw;
  class XSIL;
  class param;
  class Stream;
  class table;
  class xsil_time;

/**  The %Xreader class reads an XSil file and encodes the contents as a tree
  *  of xsil:: class objects. In general, the user would read an entire 
  *  document, and then search it for the objects of interest. This can be 
  *  done as follows:
  *
  *  \verbatim
     std::ifstream in("myxsil.xml");
     xsil::Xreader xs(in);
     xsil::xobj*  xo = in.readDoc();
     xsil::table* xt = reinterpret_cast<>(xo->find("xmltable", "table"));
     \endverbatim
  *
  *  @memo XSil reader class.
  *  @author J. Zweizig
  *  @version 1.1; September 22, 2004
  *  @ingroup IO_xsil 
  */
class Xreader {
public:

  /**  Xsil input lexicographical tokens.
   */
  enum LxrTokens {
    tkEnd,       // end of file
    tkWord,      // word
    tkString,    // quoted string
    tkNumber,    // number
    tkEqual,     // =
    tkOpenTag,   // <
    tkOpenETag,  // </
    tkOpenCTag,  // <!
    tkOpenHead,  // <?
    tkEndTag,    // >
    tkEndSTag,   // />
    tkCloseHead, // ?>
    tkEscape,    // &opr;
    tkQMark
  };

private:
  enum CharGroups {
    kInvChar,
    kAlpha,
    kNumer,
    kSpace
  };

  enum LxrStates {
    stStart,     // start state
    stInWord,    // in word
    stInString,  // quoted string
    stNumInt,    // number nnnn
    stNumFix,    // nnn.[nnn]
    stNumExp ,   // nnn[.nnn]e
    stNumSExp,   // nnn[.nnn]e[snnn]
    stTag,       // <[]
    stQMark,     // ?
    stETag,      // /
    stEscape,    // &[opr]
    stNStates
  };

public:
    /**  Construct an XML file reader to Parse an input XML stream.
      *  @memo XML reader constructor.
      *  @param istr Input stream containing formatted XML.
      */
    Xreader(std::istream& istr);

    /**  Destroy an XML reader and free all allocated storage.
      *  @memo %Xreader destructor.
      */
    ~Xreader(void);

    /**  Read an xsil array object.
      *  @memo Read an xsil array object.
      *  @param a       Attribute list read from the array tag.
      *  @param tagonly Boolean value indicating that the tag has no data.
      *  @return Pointer to an xsil array object.
      */
    array* readArray(const attr_list& a, int tagonly);

    /**  The first/next document in the stream associated with 
      *  the %Xreader is ingested into memory as a tree of xsil class objects.
      *  the returned pointer must be deleted with a C++ delete command when
      *  it will no longer be used to avoid memory leakage.
      *  \brief Read a document.
      *  \return Pointer to the parsed document.
      */
    xobj* readDoc(void);

    /**  Read an end tag and compare the tag name to the specified tag
      *  value. Pop the tag context off the stack.
      *  \brief Read and check end tag.
      *  \exception "Mismatched end tag" exception is thrown if the end tag 
      *             name does not match the current tag.
      *  \param endtag Expected tag name.
      */
    void readEndTag(const std::string& endtag);

    /**  Read the xml header from the input file.
      *  \brief Read the file header
      */
    void readHeader(void);

    /**  Read a ligolw structure object.
      *  \brief Read a ligolw structure object.
      *  \param a       Attribute list read from the ligolw tag.
      *  \param tagonly Boolean value indicating that the tag has no data.
      *  \return Pointer to an xsil ligolw object.
      */
    ligolw* readLigoLW(const attr_list& a, int tagonly);

    /**  Read an xsil object by calling the appropriate read function for the
      *  named structure type. 
      *  \brief Read an xsil object.
      *  \return Pointer to the object.
      */
    xobj* readObject(void);

    /**  Read an xsil parameter object from the input stream..
      *  \brief Read an xsil parameter object.
      *  \param a       Attribute list read from the param tag.
      *  \param tagonly Boolean value indicating that the tag has no data.
      *  \return Pointer to an xsil param object.
      */
    param* readParam(const attr_list& a, int tagonly);

    /** Read an xsil stream object from the input file.
      *  \brief Read an xsil stream object.
      *  \param a       Attribute list read from the stream tag.
      *  \param tagonly Boolean value indicating that the tag has no data.
      *  \return Pointer to an xsil stream object.
     */
    Stream* readStream(const attr_list& a, int tagonly);

    /** Read an xsil table object.
      *  \brief Read an xsil table object.
      *  \param a       Attribute list read from the table tag.
      *  \param tagonly Boolean value indicating that the tag has no data.
      *  \return Pointer to an xsil table object.
     */
    table* readTable(const attr_list& a, int tagonly);

    /**  Read an xsil time specifier.
      *  \brief Read an xsil_time instance.
      *  \param a       Attribute list read from the time tag.
      *  \param tagonly Boolean value indicating that the tag has no data.
      *  \return Pointer to an xsil_time instance.
      */
    xsil_time* readTime(const attr_list& a, int tagonly);

    /**  Read an xml object header tag. The tag type is returned in the
      *  string references by name and any attributes are copied to the 
      *  attribute list.
      *  \brief Read a tag.
      *  \param name  Reference to string that will receive the tag type.
      *  \param attrs reference to an attribute list to receive attributes
      *               specified in the tag.
      *  \return Tag type: standard=0, no data=1.
      */
    int readTag(std::string& name, attr_list& attrs);

    /**  Set the level of debug printout.
      *  @memo Set the debug level.
      *  @param lvl level of detail in debug outputs.
      */
    void setDebug(int lvl);

private:
    //----------------------------------  Set up lexicographic analysis
    void lxSetup(void);

    //----------------------------------  Get a syntactic token
    LxrTokens getToken(std::string& token);

private:
    std::istream& mStream;
    lxr           mLexr;
    int           mDebug;
};     // xsil::Xreader

}      // namespace xsil

#endif  //   XSIL_XREADER_HH
