/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xsilTSeries     					*/
/*                                                         		*/
/* Module Description: Classes for LIGO-LW input/output to DMT objects.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* (testing-- R. Rahkola)       Initial revision                        */
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: none 						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Rauha Rahkola (509) 372-8141  (509) 372-8137  rrahkola@darkwing.     */
/*                                                        uoregon.edu   */
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 2001.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _GDS_XSILTSERIES_HH
#define _GDS_XSILTSERIES_HH

#include <complex>
#include <string>
#include <iosfwd>
//=============  xml common header file
#include "xml/Xsil.hh"
//=============  DMT C++ header files
#include "TSeries.hh"
#include "Complex.hh"

namespace xml {

  /** \ingroup IO_xml
    * \{ 
    */

/**  \defgroup xml_handlers XML handlers for DMT data
  *  @memo XML handlers for parsing the LIGO_LW format of data into
  *  DMT objects following the xml representation described in
  *  LIGO-T000067. This group of classes contains the following
  *  - Classes to convert DMT data classes to xml
  *    -# xsilTSeries  - %Stream a TSeries to xml
  *    -# xsilFSpectrum - %Stream an FSpectrum to xml
  *  - Classes to handle DMT classes in an xml reader
  *    -# xsilHandlerTSeries
  *    -# xsilHandlerFSpectrum
  *  - Classes to request DMT class handlers
  *    -# xsilHandlerQueryTSeries
  *    -# xsilHandlerQueryFSpectrum
  *
  *  @author Written September 2001 by Rauha Rahkola
  *  @version 0.1
  *  \{
  *************************************************************************/


/** xsilTSeries converts a DMT TSeries object to an output stream in LIGO_LW
    format as described in LIGO-T000067.
    @memo Writes TSeries in LIGO_LW format as described in LIGO-TOOOO67
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
  *************************************************************************/
   class xsilTSeries {
   protected:
      /// Indent level
      int            fLevel;
      /// Pointer to TSeries object
      const TSeries *fData;
   public:
      /// Constructor (no TSeries)
      xsilTSeries (int level = 1) : fLevel(level), fData (0) {
      }
      /// Constructor for TSeries
      xsilTSeries(const TSeries *ts, int level = 1):
      fLevel(level),fData(ts) {
      }
      /// Constructor for TSeries
      xsilTSeries(const TSeries& ts, int level = 1):
      fLevel(level),fData(&ts) {
      }
      /// Write the TSeries data to output stream
      std::ostream& write (std::ostream &os) const;
   };


/** Write TSeries to an output stream.
    @param os output stream
    @param ts xsilTSeries object
    @memo XML output of TSeries
    @return Output stream reference
  *************************************************************************/
   inline std::ostream& operator<< (std::ostream &os, const xsilTSeries &ts) {
      return ts.write(os); }

/** Handler for TSeries data types
    @memo TSeries datatype handler
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
  *************************************************************************/
   class xsilHandlerTSeries: public xsilHandler {
   protected:
    /// Ldas variety?
      bool 			  fLdasIHaveToBeDifferent;
    /// Pointer to DMT Object vector class
      std::vector<TSeries>       *fObjects;
    /// Name of data object
      std::string                 fName;
    /// Data subtype
      int                         fSubtype;
    /// Time (sec)
      unsigned long               fSec;
    /// Time (nsec)
      unsigned long               fNsec;
    /// Data spacing
      double                      fDt;
    /// Modulation/heterodyne frequency (Hz)
      float                       fF0;
    /// Status of TSeries object
      unsigned int                fStatus;
    /// Flag for complex data
      bool                        fComplex;
    /// Data pointer
      float                      *fData;
    /// Data dimension 1 (length of each column)
      int                         fDim1;
    /// Data dimension 2 (number of columns)
      int                         fDim2;
   
   
   public:
    /// Constructor
      explicit xsilHandlerTSeries (std::vector<TSeries> *objects, 
                        const attrlist *attr=0, bool ignore=false,
                        bool ldas = false);
    /// Destructor
      virtual ~xsilHandlerTSeries();
    /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
    /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
    /// short parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
    /// int parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const int& p, int N=1);
   #ifndef __CINT__
    /// long parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const long long& p, int N=1);
   #endif //__CINT__
    /// float parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, 
                        const float& p, int N=1);
    /// double parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, 
                        const double& p, int N=1);
    /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
    /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
    /// string parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, const attrlist& attr,
                        const std::string& p);
    /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name, const attrlist& attr,
                        unsigned long sec, unsigned long nsec);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, float* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, std::complex<float>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, double* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, std::complex<double>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
      /// handler for nested data objects (used for ldas)
      virtual xsilHandler* GetHandler (const attrlist& attr);
   };

/** Xsil TSeries handler query class.
    The query will return a handler if the data object type is TSeries.
    @memo Xsil TSeries handler query
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
  *************************************************************************/
   class xsilHandlerQueryTSeries: public xsilHandlerQuery {
   protected:
    /// TSeries vector pointer
      std::vector<TSeries>       *fObjects;
   
   public:
    /// Constructor
      explicit xsilHandlerQueryTSeries (std::vector<TSeries> &vts)
      : fObjects(&vts) {
      }
    /// returns a handler for the specified object (or 0 if not)
      virtual xsilHandler* GetHandler(const attrlist& attr);
   };

  /** \} */

  /** \} */

} // namespace xml

#endif // define GDS_XSILTSERIES_HH
