#ifndef _LIGO_XSILSTD_H
#define _LIGO_XSILSTD_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xsilStd							*/
/*                                                         		*/
/* Module Description: Utility classes for LIGO-LW input/output.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 10Jun00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: xsilStd.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "xml/Xsil.hh"

   class BasicDataDescriptor;

namespace xml {


/** Utility functions for dealing with data objects which 
    follow the xml representation described in LIGO-T000067.
   
    @memo Utility functions for XML standard objects
    @author Written June 2000 by Daniel Sigg
    @version 1.0
    @ingroup xml_utils
 ************************************************************************/
   class xsilStd {
   public:
      /// xml data type
      enum DataType {
      /// Time series
      kTimeSeries = 0,
      /// Spectrum 
      kSpectrum = 1,
      /// Transfer function
      kTransferFunction = 2,
      /// Coefficients
      kCoefficients = 3,
      /// Histogram
      kHistogram = 4
      };
   
      /// Data information structure
      class datainfo {
      public:
         /// channel list type
         typedef std::vector<std::string> chnlist;
         /// Object type
         DataType	fType;
         /// subtype
         int		fSubtype;
         /// A channel list
         chnlist	fAChn;
         /// B channel list
         chnlist	fBChn;
         /// Length of data array
         int		fN;
         /// Number of data arrays
         int		fM;
         /// GPS sec
         unsigned long	fSec;
         /// GPS nsec
         unsigned long	fNSec;
         /// dt
         double		fDt;
         /// f0
         double		fF0;
         /// df
         double		fDf;
         /// BW
         double		fBW;
         /// Averages
         int		fAverages;
         /// Measurement number
         int		fMeasurementNumber;
	 /// Constructor
         datainfo () {
            init(); }
	 /// Initialization method
         void init ();
      };
   
      /** Get the data type and subtype associated with the graph type.
          The returned type assumes Y data only. Use DataTypeXY to
          translate to XY data arrays.
          @memo Get type/subtype.
          @param graphtype Graph type
          @param type XML data object type (return)
          @param subtype XML data object subtype (return)
          @return true if successful
       ******************************************************************/
      static bool GetDataType (const char* graphtype, DataType& type,
                        int& subtype);
      /** Get the graph type associated with the type and subtype.
          @memo Get graph type.
          @param type XML data object type
          @param subtype XML data object subtype
          @param graphtype Graph type (return)
          @return true if successful
       ******************************************************************/
      static bool GetGraphType (DataType type,
                        int subtype, std::string& graphtype);
   
      /** Returns the corresponding type name.
          @memo Get type name.
          @param type XML data object type
          @return type name 
       ******************************************************************/
      static std::string Typename (DataType type);
      /** Returns the corresponding type id.
          @memo Get type id.
          @param type XML data object type
          @return type name 
       ******************************************************************/
      static DataType Typeid (const char* type);
   
      /** Returns the subtype for XY data.
          @memo Get XY subtype.
          @param type XML data object type
          @param subtype XML data object subtype
          @return subtype
       ******************************************************************/
      static int DataSubtypeXY (DataType type, int subtype);
   
      /** Separates a name into character string and indices.
          @memo Analyze name.
          @param name Name to analyze
          @param n character string (return)
          @param index1 first index or -1 (return)
          @param index2 second index or -1 (return)
          @return true if successful
       ******************************************************************/
      static bool analyzeName (const std::string& name,
                        std::string& n, int& index1, int& index2);
   
      /** Makes a name from a character string and indices.
          @memo Make name.
          @param Name character string
          @param index1 first index or -1
          @param index2 second index or -1
          @return Name with indices
       ******************************************************************/
      static std::string makeName (const std::string& Name, 
                        int index1 = -1, int index2 = -1);
   
      /** Returns information about a data object based on an
          xml stream describing the object.
          @memo Get data descriptor information.
          @param xml xml data stream
          @param info data info structure (return)
          @return true if successful
       ******************************************************************/
      static bool GetDataInfo (const char* xml, datainfo& info);
   
      /** Returns information about a data descriptor. The second
          dimension is the number of data points, whereas the first
          dimension describes the number of arrays.
          @memo Get data descriptor information.
          @param desc Data descriptor
          @param dim1 Dimension 1 (return)
          @param dim2 Dimension 2 (return)
          @param complex true if complex data (return)
          @param XYdata true if XY data format (return)
          @return true if successful
       ******************************************************************/
      static bool GetDataInfo (const BasicDataDescriptor* desc,
                        int& dim1, int& dim2, bool& complex,
                        bool& XYdata);
   
      /** Returns information about a data descriptor. The second
          dimension is the number of data points, whereas the first
          dimension describes the number of arrays. If copy is true,
          the returned data pointer is a copy of the data and must
          be freed by the user!
          
          @memo Get data descriptor information.
          @param desc Data descriptor
          @param dim1 Dimension 1 (return)
          @param dim2 Dimension 2 (return)
          @param complex true if complex data (return)
          @param XYdata true if XY data format (return)
          @param dataptr Pointer to float array (return)
          @param cpy true if data is copy (return)
          @param compress if true, multiple referenced objects are dealt as one
	  @param forceXY force X-Y mode?
          @return true if successful
       ******************************************************************/
      static bool GetDataInfo (const BasicDataDescriptor* desc,
                        int& dim1, int& dim2, bool& complex, bool& XYdata, 
                        const float** dataptr, bool& cpy, 
                        bool compress = false, bool forceXY = false);
   
      /** Returns information about a data descriptor. The second
          dimension is the number of data points, whereas the first
          dimension describes the number of arrays. If copy is true,
          the returned data pointer is a copy of the data and must
          be freed by the user!
          
          @memo Get data descriptor information.
          @param desc Data descriptor
          @param dim1 Dimension 1 (return)
          @param dim2 Dimension 2 (return)
          @param complex true if complex data (return)
          @param XYdata true if XY data format (return)
          @param dataptr Pointer to double array (return)
          @param cpy true if data is copy (return)
          @param compress if true multiple referenced objects are dealt as one
	  @param forceXY force X-Y mode?
          @return true if successful
       ******************************************************************/   
      static bool GetDataInfo (const BasicDataDescriptor* desc,
                        int& dim1, int& dim2, bool& complex, bool& XYdata, 
                        const double** dataptr, bool& cpy, 
                        bool compress = false, bool forceXY = false);
   };


}

#endif // _LIGO_XSILSTD_H
