#include "PConfig.h"
#include "web/webcache.hh"
#include "web/webclient.hh"
#include <strings.h>
#include <signal.h>
#include <unistd.h>
#include <iostream>
#include <cstring>

namespace web {
   using namespace std;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// cachepage	                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   webcache::cachepage::cachepage()
   : fData (0), fLen (0), fAllowCompress (false)
   {
   }

//______________________________________________________________________________
   webcache::cachepage::cachepage (const Time& exp, const char* d, 
                     int len, const header_type& header)
   : fTime (exp), fData (0), fLen (0), fHeader (header),
   fAllowCompress (false)
   {
      load (d, len);
   }

//______________________________________________________________________________
   webcache::cachepage::cachepage (const Time& exp, const std::string& s,
                     const header_type& header)
   : fTime (exp), fData (0), fLen (0), fHeader (header), 
   fAllowCompress (false)
   {
      load (s);
   }

//______________________________________________________________________________
   webcache::cachepage::cachepage (const webcache::cachepage& page)
   : fData (0), fLen (0)
   {
      *this = page;
   }

//______________________________________________________________________________
   webcache::cachepage::~cachepage()
   {
      if (fData) delete[] fData;
   }

//______________________________________________________________________________
   webcache::cachepage& webcache::cachepage::operator= (const cachepage& page)
   {
      if (this != &page) {
         fTime = page.fTime;
         load (page.fData, page.fLen);
         fHeader = page.fHeader;
         fAllowCompress = page.fAllowCompress;
      }
      return *this;
   }


//______________________________________________________________________________
   void webcache::cachepage::load (const char* d, int len)
   {
      if (fData) {
         delete[] fData;
         fData = 0;
         fLen = 0;
      }
      if (d && (len >=0)) {
         fData = new char[len + 1];
         memcpy (fData, d, len);
         fData[len] = 0; // this makes it work for char*
         fLen = len;
      }
   }


//______________________________________________________________________________
   void webcache::cachepage::load (const std::string& s)
   {
      load (s.c_str(), s.size());
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// urlcompare	                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool webcache::urlcompare::operator() (const std::string& s1, 
                     const std::string& s2) const
   {
      string p1 = http_request::demangle (s1);
      string p2 = http_request::demangle (s2);
      return (strcasecmp (p1.c_str(), p2.c_str()) < 0);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// webcache	                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   webcache::webcache (int maxsize, bool autocleanup)
   
   : fMaxSize (-1), fCleanupID (0)
   {
      setMax (maxsize);
      if (autocleanup) enableAutoCleanup();
   }

//______________________________________________________________________________
   webcache::~webcache()
   {
       if (run()) stop_thread();
   }

//______________________________________________________________________________
   bool webcache::add (const char* url, const cachepage& page)
   {
      if (!url || !*url || (fMaxSize == 0)) {
         return false;
      }
      thread::semlock lockit (fMux);
      if (fMaxSize > 0) {
         int allowed = fMaxSize - page.size();
         if (allowed < 0) {
            return false;
         }
         fCache[url] = cachepage(); // remove old if there
         reduceCache (allowed);
      }
      fCache[url] = page;
      return true;
   }

//______________________________________________________________________________
   bool webcache::lookup (const char* url, cachepage& page)
   {
      if (!url || !*url) {
         return false;
      }
      thread::semlock lockit (fMux);
      cachelist::iterator i = fCache.find (url);
      if (i == fCache.end()) {
         return false;
      }
      page = i->second;
      return true;
   }

//______________________________________________________________________________
   void webcache::cleanup (const Time& now)
   {
      thread::semlock lockit (fMux);
   start:
      for (cachelist::iterator i = fCache.begin(); i != fCache.end(); ++i) {
         if (i->second.getTime() < now) {
            cout << "Erase from cache " << i->first << endl;
            fCache.erase (i);
            goto start;
         }
      }
   }

//______________________________________________________________________________
  void* webcache::thread_entry(void)
  {
     while (run()) {
        sleep (30);
        cleanup (Now());
     }
     return 0;
  }

//______________________________________________________________________________
   bool webcache::enableAutoCleanup ()
   {
       set_detached(true);
       return (start_thread() == 0);
   }

//______________________________________________________________________________
   void webcache::setMax (int max)	
   {
      fMaxSize = max;
      reduceCache (max);
   }

//______________________________________________________________________________
   int webcache::size() const
   {
      thread::semlock lockit (fMux);
      int len = 0;
      for (cachelist::const_iterator i = fCache.begin(); 
          i != fCache.end(); ++i) {
         len += i->second.size();
      }
      return len;
   }

//______________________________________________________________________________
   bool webcache::reduceCache (int max)
   
   {
      // no limit
      if (max < 0) {
         return true;
      }
      // clear
      if (max == 0) {
         thread::semlock lockit (fMux);
         fCache.clear();
         return true;
      }
      // check if we have enough room
      thread::semlock lockit (fMux);
      int len = size();
      if (len <= max) {
         return true;
      }
      // delete oldest entry until enough space
      do {
         cachelist::iterator oldest = fCache.begin();
         for (cachelist::iterator i = fCache.begin(); 
             i != fCache.end(); ++i) {
            if (i->second.getTime() < oldest->second.getTime()) {
               oldest = i;
            }
         }
         if (oldest != fCache.end()) {
            len -= oldest->second.size();
            fCache.erase (oldest);
         }
      } while (!fCache.empty() && (len > max));
   
      return true;
   }

}
