/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   Shared memory stream class
//
//   Compile flags defined:
//      INTRDBG    Print status to cerr on failure of getBuffer()
//
#include <time.h>
#include "iSMbuf.hh"
#include "lsmp_con.hh"
#include <stdio.h>
#include <iostream>
#include <errno.h>

using namespace std;

#ifdef ISMBUF_TRACE
#define TRACEPOINT(a,b,c) mTrace.tracepoint(a, b, c);
#else
#define TRACEPOINT(a,b,c) 
#endif

//--------------------------------------  Constructors.
iSMbuf::iSMbuf(void) 
  : mConsumer(0), mBuffer(0), mLength(0)
{
    TRACEPOINT("iSMbuf()", 0, 0)
}

iSMbuf::iSMbuf(const char *partition, openmode_type mode) 
  : mConsumer(0), mBuffer(0), mLength(0)
{
    TRACEPOINT("iSMbuf(const char*, ios::openmode)", 0, 0)
    open(partition, mode);
}

iSMbuf* 
iSMbuf::open(const char *partition, openmode_type mode) {
    TRACEPOINT("open(const char*, ios::open_mode)", 0, 0)
    if ((mode & (ios::out | ios::trunc))) return (iSMbuf*) 0;

    //----------------------------------  Create/open a new partition.
    mConsumer = new LSMP_CON(partition, 0);
    if (mConsumer->isConnected()) return this;
    delete mConsumer;
    mConsumer = 0;
    return (iSMbuf*) 0;
}

//--------------------------------------  Destructor
iSMbuf::~iSMbuf() {
    if (mConsumer) delete mConsumer;
#ifdef ISMBUF_TRACE
    mTrace.pTrace();
#endif
}

//======================================  Test for latest buffer
bool
iSMbuf::latest(void) const {
    if (!mConsumer) return false;
    return mConsumer->latest_buffer();
}

//--------------------------------------  Set the reserved buffer count
void
iSMbuf::setBCount(int nbuf) {
    if (mConsumer) mConsumer->setNBuffer(nbuf);
    TRACEPOINT("setBCount(int)", nbuf, 0)
}

//======================================  Wait for a buffer of data
bool
iSMbuf::waitBuf(bool nowait) {
    return timedWait(nowait? 0.0 : -1.0);
}

//======================================  Wait for a buffer with timeout
bool
iSMbuf::timedWait(double wtime) {
    TRACEPOINT("TimedWait(bool)", int(wtime), 0)
    if (!mConsumer)     return false;
    if (this->gptr() >= this->egptr()) relse();
    if (mBuffer) return true;
    if (wtime == 0) {
	mBuffer = mConsumer->get_buffer(NOWAIT);
    } else {
	mConsumer->setTimeout(wtime);
	mBuffer = mConsumer->get_buffer(0);
	mConsumer->setTimeout(-1.0);
    }
    if (!mBuffer) return false;
    mLength = mConsumer->getLength();
    setptrs(0);
    return true;
}

//======================================  Allocate a buffer
int 
iSMbuf::doallocate() {
    TRACEPOINT("doallocate()", 0, 0)

    //----------------------------------  Make sure a partition is attached.
    if (!mConsumer)          return EOF;
    if (!mConsumer->valid()) return EOF;

    //----------------------------------  Release a buffer if one is allocated
    relse();

    //----------------------------------  Get a new SM buffer.
    errno = 0;
    do {
        mBuffer = mConsumer->get_buffer();
#ifdef INTRDBG
	if (!mBuffer || !mConsumer->getLength()) {
	    perror("error in iSMbuf::doallocate()");
	    cerr << "Length is: " << mConsumer->getLength() << endl;
	}
#endif  //   def(INTRDBG)
    } while (!mBuffer && errno == EINTR);
    mLength = mConsumer->getLength();
#ifdef __GNU_STDC_OLD
    setb(const_cast<char*>(mBuffer), const_cast<char*>(mBuffer)+mLength, 1);
#endif
    return 1;
}

//--------------------------------------  Get the next data buffer
int 
iSMbuf::underflow() {
    TRACEPOINT("underflow()", in_avail(), 0);

    //----------------------------------  See if data are already available
    if (in_avail() > 0) return *(this->gptr());

    //----------------------------------  Make sure a partition is attached.
    if (!mConsumer)          return EOF;
    if (!mConsumer->valid()) return EOF;

    //----------------------------------  Release a buffer if one is allocated
    relse();

    //----------------------------------  Get a new SM buffer.
    errno = 0;
    do {
        mBuffer = mConsumer->get_buffer();
#ifdef INTRDBG
	if (!mBuffer || !mConsumer->getLength()) {
	    perror("error in iSMbuf::underflow()");
	    cerr << "Length is: " << mConsumer->getLength() << endl;
	}
#endif  // def(INTRBDG)
    } while (!mBuffer && errno == EINTR);
    if (mBuffer) {
        mLength = mConsumer->getLength();
	setptrs();
	return *mBuffer;
    } else {
        mLength = 0;
        setptrs();
        return EOF;
    }
}

//======================================  Release shared memory buffer
void
iSMbuf::relse(void) {
    if (mBuffer && mConsumer) mConsumer->free_buffer();
    mBuffer = static_cast<const char*>(0);
    mLength = 0;
    setptrs(0);
}

//======================================  Seek and ye might find
iSMbuf::pos_type 
iSMbuf::seekoff(off_type off, seekdir_type dir, openmode_type mode)
{
    TRACEPOINT("seekoff(streamoff, seekdir_type, int)", off, dir)
    if (!mBuffer) underflow();
    streampos newpos(0);
    if (dir == ios::beg) {
        newpos = off;
    } else if (dir == ios::cur) {
        newpos = long(this->gptr()) - long(mBuffer) + off;
    } else if (dir == ios::end) {
        newpos = mLength + off;
    }
    if (newpos > mLength || newpos < 0) return (streampos) EOF;
    // setptrs(newpos);
    char* p0 = const_cast<char*>(mBuffer);
    this->setg(p0, p0+newpos, p0+mLength);
    return newpos;
}
 
//======================================  Get the current buffer event id
int
iSMbuf::eventid(void) const {
    if (!mConsumer) return 0;
    return mConsumer->getEvtID();
}

void 
iSMbuf::setptrs(int ix) {
    char* p0 = const_cast<char*>(mBuffer);
    this->setg(p0, p0+ix, p0+mLength);
}

#ifdef ISMBUF_TRACE
void 
iSMbuf::trace::pTrace(void) const {
    cout << "Number of traced function calls: " << nTrace << endl;
    int npr = (nTrace < maxTrace) ? nTrace : maxTrace;
    for (int i=0 ; i < npr ; i++) {
        int inx = (nTrace - i - 1) % maxTrace;
	cout << mFuncName[inx] << " " << mArg1[inx] << " " << mArg2[inx] 
	     << endl;
    }
}
#endif
