/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LSMP_INT_HH
#define LSMP_INT_HH
//
//   Ligo Shared Memory Partition internal structures.
//
#include "PConfig.h"
#include <sys/types.h>
#include "ManyFlags.hh"
#include "lsmp.hh"

#define LSMP_VERSION 102                    // Package version number
#define LSMP_SHMBASE 32                     // shm base id
#define LSMP_MAXSHM   8                     // maximum # of shm partitions
#define LSMP_MAXCONS  32                    // maximum number of consumers
#define LSMP_CONSPWD  8                     // consumers per mask word
#define LSMP_LMASK   ((LSMP_MAXCONS-1)/LSMP_CONSPWD+1) // consumer mask length
#define CON_BIT(i)  (i%LSMP_CONSPWD)
#define CON_MASK(i) (1 << (i%LSMP_CONSPWD))
#define CON_WORD(i) (i/LSMP_CONSPWD)

//--------------------------------------  Consumer flag class
typedef ManyFlags<LSMP_LMASK,LSMP_CONSPWD> ConFlags;

//---------------------------------------  Buffer header structure
#define SEG_QUEUED 0x0001

struct LSMP_consbk;

/**  Ligo shared memory buffer class contains information about the buffers 
  *  managed by the LSMP software.
  *  @author John Zweizig
  *  @version 2.1; Last Modified March 3, 2008
  *  @ingroup IO_lsmp
  */
struct LSMP_buffer {
    /**  Buffer status flag word.
     */
    unsigned int status;

    /**  Buffer offset in bytes from the start of the shared memory partition.
     */
    long boff;

    /**  Bit mask containing one bit per consumer and is used to indicate
      *  which consumers have reserved the buffer.
      *  @memo Mask of consumers that have reserved the buffer.
      */
    ConFlags reserve_mask;

    /**  Bit mask containing one bit per consumer and is used to prevent 
      *  a buffer. The seen_mask bit for a given consumer is set in 
      *  LSMP_CONS::get_buffer() when the buffer is allocated to a consumer 
      *  or when it is skipped by the setNSkip() mechanism. The entire 
      *  seen_mask is cleared when a buffer is returned or released (for 
      *  distribution) by a producer.
      *  @memo Mask of consumers that have seen the buffer.
      */
    ConFlags seen_mask;

    /**  The use count indicates the number of consumers that have reserved
      *  or allocated the buffer. The use count is zeroed when the partition 
      *  is constructed or when the buffer is released or returned by a 
      *  producer. The count is decremented when a consumer frees the buffer 
      *  or a consumer to which the buffer is allocated is deleted, and it 
      *  is incremented when buffer is allocated by a consumer.
      *  @memo Number of consumers reserving this buffer.
      */
    int use_count;

    ///  Length of data in buffer.
    int ldata;

    ///  Trigger mask.
    int trig;

    ///  Time associated with the data.
    time_t   fill_time;

    ///  data id.
    LSMP::eventid_type data_ID;

    ///  Process identifier associated with this consumer.
    pid_t    owner_pid;

    ///  Link to the next buffer in the free queue
    int      link;

    ///  Number of times the buffer has been refilled.
    int      evt_count;

    /** Test whether the buffer is in use.
      *  \brief Test if buffer is in use.
      *  \return True if in use.
     */
    bool inUse(void)  const {return use_count || reserve_mask.any();}

    /**  Test whether the buffer has been seen
      *  \brief Test if buffer was seen.
      *  \return True if seen.
     */
    bool seen(void)   const {return seen_mask.any();}

    /**  Test whether the buffer is in a queue.
      *  \brief Test if buffer in queue.
      *  \return True if in queue.
      */
    bool queued(void) const {return (status & SEG_QUEUED) != 0;}
};

//---------------------------------------  Buffer queue structure

/**  Ligo shared memory buffer queue class contains an ordered list of buffers.
  *  @author John Zweizig
  *  @version 2.0; Last Modified March 3, 2008
  *  @ingroup IO_lsmp
  */
class LSMP_BufferQ {
  public:
    /// Index of the first buffer in the queue
    int head;

    /// Index of the last buffer in the queue
    int tail;

    /**  Buffer queue constructor.
      *  \brief default constructor
      */
    LSMP_BufferQ(void) {
        init();
    }

    /**  Initialize the buffer queue.
      *  \brief Initialize queue pointers.
      */
    void init(void) {
        head = -1;
	tail = -1;
    }

    /**  Test if the buffer queue is empty.
      *  \brief Test empty queue.
      *  \return True if no buffers are linked into this queue.
      */
    bool empty(void) const {
        return (head < 0);}

    /**  Count the number of entries in the buffer queue. The queue must be 
      *  locked for read access when this method is called.
      *  \brief Buffer queue length.
      *  \param buf Buffer vector pointer.
      *  \return Number of buffers in the queue.
      */
    int length(const LSMP_buffer *buf) const;

    /**  Link the Specified buffer to the end of the list. The queue must be 
      *  locked for read access when this method is called.
      *  \brief link buffer to tail of list.
      *  \param buf Pointer to buffer control vector.
      *  \param ib  Buffer number to be linked
      *  \return ID of removed buffer or -1 on failure.
      */
    void link(LSMP_buffer *buf, int ib);

    /**  Remove the first buffer from the buffer list. The queue must be 
      *  locked for read access when this method is called.
      *  \brief Remove head buffer from the list.
      *  \param buf Pointer to buffer control vector.
      *  \return ID of removed buffer or -1 on failure.
      */
    int remove(LSMP_buffer *buf);

    /**  Remove the specified buffer from the buffer list. The queue must be 
      *  locked for read access when this method is called.
      *  \brief Remove a buffer from the list.
      *  \param buf Pointer to buffer control vector.
      *  \param ib  Buffer number to be removed
      *  \return ID of removed buffer or -1 on failure.
      */
    int remove(LSMP_buffer *buf, int ib);
};

//--------------------------------------  Shared memory structure.
#define RELBUF  0x0001                     // release unrequired buffers
#define SCAVAGE 0x0002                     // search for unused buffers
#define RQSYNCH 0x0004                     // synchronize on requests
#define KEEP    0x0008                     // Keep an unaccessed partition
#define EXPOSE  0x0010                     // Expose data to all consumers

/**  The shared memory global class contains the control and status 
  *  held in the shared memory partition.
  *  @brief Shared memory global data.
  *  @author John Zweizig
  *  @version 1.2; Last modified March 4, 2008
  *  @ingroup IO_lsmp
  */
struct LSMP_global {
    /**  The status flags control the buffer allocation policy. The default
      *  policy is to allocate buffers from the free list only and to release
      *  a buffer only when it is not reserved, is not in use and has been 
      *  seen by at least one consumer. The policy modification bits are as 
      *  follows:
      *  <table>
      *  <tr><td>\c RELBUF</td><td>A buffer is released if no consumers have 
      *                         reserved it, even if it has not been seen.
      *                     </td></tr>
      *  <tr><td>\c SCAVAGE</td><td>The oldest unused and unreserved buffer 
      *                          may be allocated from the full queue. Unused
      *                          buffers are not returned to the free queue.
      *                     </td></tr>
      *  <tr><td>\c RQSYNCH</td><td>No buffer may be allocated by a producer 
      *                          unless there is a consumer waiting.</td></tr>
      *  <tr><td>\c KEEP</td><td>Don't delete the partition when the global 
      *                       access count goes to zero.</td></tr>
      *  <tr><td>\c EXPOSE</td><td>All buffers will be automatically reserved 
      *                         by all current consumers.</td></tr>
      *  </table>
      */
    int  status;                   // Status flags
    ///  LSMP version number
    int  version;                  // LSMP version number
    ///  Number of times partition was accessed
    int  gbl_count;                // Number of accesses
    ///  Global semaphore ID
    int  gbl_semid;                // global semaphore ID
    ///  Number of buffer in partition
    int  nbuf;                     // Number of buffers
    ///  Buffer data length
    int  lbuf;                     // Buffer length
    ///  Full buffer list
    LSMP_BufferQ full;             // Full buffer queue
    ///  Free buffer list
    LSMP_BufferQ free;             // Free buffer queue
    ///  Partition name
    char name[LSMP_LNAME];         // Partition name
    ///  Number of active consumers
    int  ncons;                    // Number of active consumers
    ///  Consumer semaphore IDs
    int  con_semid[LSMP_LMASK];    // Consumer semaphore IDs
    ///  Consumer use masks
    ConFlags conmask;              // Consumer use mask
    ///  Consumer use masks
    ConFlags conresrv;             // Consumer reservation

    /**
     *  RefCons returns a pointer to the specified consumer.
     *  \brief Consumer block pointer.
     *  \param icon Consumer number.
     *  \return Pointer to the specified consumer block.
     */
    LSMP_consbk* refCons(int icon);

    /**
     *  refBuffer returns a pointer to the specified buffer control block.
     *  \brief Buffer block pointer.
     *  \param ibuf Buffer number.
     *  \return Pointer to the specified buffer block.
     */
    LSMP_buffer* refBuffer(int ibuf);
};

/**  Global semaphore enumeration.
  *  @brief Global semaphore numbers.
  */
enum gblsem {
    gbl_gate,
    gbl_empty,
    gbl_full,
    gbl_synch,
    gbl_nsems
};

/**  The consumer structure contains the status and control information 
  *  for a single consumer.
  *  @brief Consumer status information.
  */
struct LSMP_consbk {
    ///  Maximum buffers consumer may request
    int mxbuf;
    ///  Trigger mask
    int trig_mask;
    ///  Minimum time between events
    int min_time;
    ///  Minimum separation between events
    int min_sep;
    ///  Time Counter
    int time_ctr;  
    ///  Skip Counter     
    int skip_ctr;

    /**  The number of allocated segments is the number of allocated or 
      *  reserved segments. This counter is incremented when a buffer is 
      *  reserved (\c LSMP_PROD::distribute() ) or allocated 
      *  (\c LSMP_CON::get_buffer() ) and decremented when the buffer is 
      *  freed (\c LSMP_CON::free_buffer() ).
      *  @brief Number of segments currently allocated to the consumer.
      */
    int seg_ctr;
    ///  Segments read since consumer allocation
    int seg_tot;
    /// Consumer status flags
    int flags;
    /// Consumer process identifier
    pid_t pid;

    /**  Test consumer read-all flags
      *  @brief  Test for read-all mode.
      *  @return true if consumer is read-all.
      */
    bool isReadAll(void) const {return (flags & READALL) != 0;}

    /**  Clear the waiting for event (EVWAIT) status flag.
      *  @brief Clear waiting flag.
      */
    void clrWait(void) {flags &= ~EVWAIT;}

    /**  Set the waiting for event (EVWAIT) status flag.
      *  @brief Set waiting flag.
      */
    void setWait(void) {flags |= EVWAIT;}
};

//--------------------------------------  Define the semctl union
//  Note: This is now necessary on most systems in use by LIGO: sun, linux 
//        and hpux. Other systems (e.g. AIX and Irix) may choke on this, 
//        but for now it's easier to leave it in unconditionally.
// -- Mac OS X sets it, so add a case
/**  Union semun used by semctl.
  */
#ifdef P__DARWIN
#else
union semun {
    ///  Value for SETVAL
    int val;
    ///  Buffer for IPC_STAT & IPC_SET
    struct semid_ds *buf;
    ///  Array for GETALL & SETALL
    unsigned short *array;
};
#endif

//======================================  Inline functions.
#include "lsmp_int.icc"
#endif
