/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_ATTRIB_HH
#define HTML_ATTRIB_HH

#include <time.h>
#include <map>
#include <string>

namespace html {
    class writer;

    /**  Abstract attribute API.
      *  @memo Attribute API.
      *  @author J. Zweizig
      *  @version 1.1; Modified May 4, 2001.
      *  @ingroup IO_html
      */
    class attrib {
    public:
        /**  Null attribute constructor.
	  *  @memo Default constructor.
	  */
        attrib(void);

        /**  Attribute copy constructor.
	  *  @memo Copy Constructor.
	  */
        attrib(const attrib& x);

        /**  Attribute destructor.
	  *  @memo Destructor.
	  */
        virtual ~attrib(void);

        /**  Make an exact copy of the current attribute.
	  *  @memo Clone an attribute.
	  *  @return Pointer to the cloned attribute.
	  */
        virtual attrib* clone(void) const = 0;

        /**  Set the writer according to the attribute specified.
	  *  @memo set the writer with an attribute.
	  *  @param w HTML writer to be set.
	  */
        virtual void setWrite(writer& w) const;

        /**  Test whether the attribute is set to its default value.
	  *  @memo Test for default attribute.
	  *  @return True if the attribute is the default value.
	  */
        virtual bool isDefault(void) const;

        /**  Test whether the attribute takes an argument.
	  *  @memo Test for boolean attribute.
	  *  @return True if the attribute takes an argument.
	  */
        virtual bool hasArg(void) const;

        /**  Write the attribute value to be used inside a tag.
	  *  @memo write the attrubute value to an html writer.
	  *  @param w HTML writer to which the attribute value is written.
	  */
        virtual void putAttr(writer& w) const;

    private:
    };

    /**  An attribute list is a map associating a set of attribute names 
      *  to with their appropriate values.
      *  @memo Attribute list class.
      *  @author J. Zweizig
      *  @version 1.0; Modified January 5, 2001
      *  @ingroup IO_html
      */ 
    class attList {
    public:

        /**  Construct a null attribute list.
	  *  @memo Default constructor.
	  */
        attList(void);

        /**  Copy an attribute list by cloning all known attributes.
	  *  @memo Copy constructor.
	  *  @param x Attribute list to be copied.
	  */
        attList(const attList& x);

        /**  Delete all attribute values in the attribute list and clear 
	  *  the list.
	  *  @memo destructor.
	  */
        virtual ~attList(void);

        /**  The specified attribute value is cloned and added to the 
	  *  attribute list with the given name. 
	  *  @memo Add an attribute to the list.
	  *  @param name Attribute name to be added.
	  *  @param a    Value of attribute to be added
	  *  @return reference to the attribute value entry.
	  */
        attrib& addAttr(const std::string& name, const attrib& a);

        /**  The specified attribute value is used to replace the 
	  *  attribute with the given name. 
	  *  @memo Replace an attribute in the list.
	  *  @param name Attribute name to be replaced.
	  *  @param a    Replacement attribute value.
	  *  @return reference to the attribute value entry.
	  */
        attrib& repAttr(const std::string& name, const attrib& a);

        /**  Remove the specified attribute from the list.
	  *  @memo remove an attribute.
	  *  @param name Name of attribute to be removed.
	  */
        void remAttr(const std::string& name);

        /**  Delete and destroy all attributes from the list.
	  *  @memo Clear attribute list.
	  */
        void clear(void);

        /**  Test if the list has any attributes.
	  *  @memo test for empty list.
	  *  @return true if the list is empty.
	  */
        bool empty(void) const;

        /**  Test whether the named attribute is included in the list.
	  *  @memo Test if attribute exists.
	  *  @return true if the named attribute is in the list.
	  *  @param key Name of attribute to search for.
	  */
        bool exists(const std::string& key) const;

        /**  Look for the specified attribute. Return a reference if it is 
	  *  found.
	  *  @memo Find a named attribute.
	  *  @return Reference to the found attribute instance.
	  *  @param key name of the attribute to be searched for.
	  */
        const attrib& find(const std::string& key) const;

        /**  Test whether all specified attributes have their default values.
	  *  @memo Test for default list.
	  */
        bool isDefault(void) const;

        /**  List all non-default attributes to the HTML document stream.
	  *  @memo List attributes.
	  *  @param w HTML document stream to receive attribute list.
	  */
        virtual void putAttr(writer& w) const;

        /**  Set HTML document writer to reflect any state attributes.
	  *  @memo Set writer attributes.
	  *  @param w Writer to modify based on attribute values.
	  */
        virtual void setWrite(writer& w) const;

        /**  Merge specified attribute list into the current list. Any
	  *  attribute in the specified list that is not covered by the 
	  *  current list will be added to the current list.
	  *  @memo Merge attribute lists.
	  *  @param x List to be merged into the current list.
	  */
        virtual void merge(const attList& x);

    private:
        typedef std::map<std::string, attrib*> AttrList;
        typedef AttrList::iterator attr_iter;

    public:
        /**  Constant iterator for use in traversing the attribute list.
	  *  @memo Constant attribute iterator.
	  */
        typedef AttrList::const_iterator const_attr_iter;

        /**  Fetch an iterator pointing to the start of the attribute list.
	  *  @memo List start iterator.
	  */
        const_attr_iter begin(void) const;

        /**  Fetch an iterator pointing to the end of the attribute list.
	  *  @memo List end iterator.
	  */
        const_attr_iter end(void) const;

    private:
        AttrList mList;
    };

}

inline bool
html::attList::empty(void) const {
    return mList.empty();
}

inline html::attList::const_attr_iter 
html::attList::begin(void) const {
    return mList.begin();
}

inline html::attList::const_attr_iter 
html::attList::end(void) const {
    return mList.end();
}

#endif  // HTML_ATTRIB_HH
