/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "file_stream.hh"
#include "scandir.hh"
#include <sys/inotify.h>
#include <iostream>
#include <sys/select.h>
#include <stdexcept>
#include <unistd.h>

using namespace std;

//======================================  Default constructor
file_stream::file_stream(void) 
  : mNotifyFd(-1)
{}

//======================================  Pattern constructor
file_stream::file_stream(const std::string& pattern, gps_type tStart) 
  : mNotifyFd(-1)
{
   parse_pattern(pattern);
   add_files(tStart);

   add_watch();
}

//======================================  Default constructor
file_stream::~file_stream(void) {
   while (!event_list.empty()) pop_event();
   if (is_open()) close(mNotifyFd);
}

//======================================  Add files and a book a pattern
void 
file_stream::add_files(gps_type tStart) {

   //-----------------------------------  Add single file to the list
   if (pattern_list.empty()) return;
   if (pattern_list.size() == 1) {
      insert_file(pattern_list[0]);
      return;
   }

   //-----------------------------------  fetch all the files.
   fetch_files(0, pattern_list[0], tStart);
}

//======================================  Add files and a book a pattern
void 
file_stream::add_watch(void) {
   if (pattern_list.empty() || !open_notify()) return;

   string base = pattern_list[0];
   push_event(base);
   for (size_t i=1; i<pattern_list.size()-1; i++) {
      scandir sd(base);
      string maxdir;
      while (sd.next_entry()) {
	 string file = sd.file_name();
	 if (sd.is_dir() && match_pattern(file, pattern_list[i])) {
	    if (maxdir.empty() || maxdir < file) maxdir = file;
	 }
      }
      base += "/";
      base += maxdir;
      push_event(base);
   }
}

//======================================  Add files and a book a pattern
std::string
file_stream::next_file(void) {
   while (!wait_file());
   string file = file_list.front().first;
   file_list.pop_front();
   return file;
}

//======================================  Inser all file that match the pattern
void 
file_stream::fetch_files(size_t pos, const std::string& base, gps_type tStart) {
   pos++;
   if (pos >= pattern_list.size()) return;
   bool is_leaf = (pos == pattern_list.size() - 1);

   //-----------------------------------  Scan matching directory contents.
   // cerr << "scan directory[" << pos << "]: " << base << endl;
   scandir sd(base);
   while (sd.next_entry()) {
      if (match_pattern(sd.file_name(), pattern_list[pos], tStart)) {
	 // cerr << "pos: " << pos << " leaf: " << is_leaf << " dir: " 
	 //      << sd.is_dir() << " file: " << sd.file_name() << endl;

	 //-----------------------------  Scan next level directory
	 if (!is_leaf && sd.is_dir()) {
	    fetch_files(pos, sd.file_path(), tStart);
	 }

	 //-----------------------------  Fill matching strings
	 else if (is_leaf && !sd.is_dir()) {
	    insert_file(sd.file_path());
	 }
      }
   }
}

//======================================  Insert a file into the ordered list.
void 
file_stream::insert_file(const std::string& name) {
   // cerr << "insert file: " << name << endl;
   //-----------------------------------  find the file leaf-name
   string leaf;
   if (name.empty()) return;
   string::size_type leaf_index = name.find_last_of("/");
   if (leaf_index == string::npos) leaf = name;
   else                            leaf = name.substr(leaf_index+1);
   if (leaf.empty()) return;

   //-----------------------------------  insert the file entry.
   bool ok = false;
   for (file_list_iter i=file_list.end(); !ok && i != file_list.begin(); ) {
      file_list_iter isav = i--;
      if (i->first.substr(i->second) < leaf) {
	 file_list.insert(isav, file_list_node(name, leaf_index));
	 ok = true;
      }
   }
   if (!ok) {
      file_list.push_front(file_list_node(name, leaf_index));
   }
}

//======================================  pull a number out of a string
static int
scan_digits(const string& s, size_t& i) {
   size_t N=0;
   while (i < s.size() && s[i] >= '0' && s[i] <= '9') {
      N = N*10 + int(s[i++]) - int('0');
   }
   return N;
}

//======================================  Match a file name to a pattern
bool
file_stream::match_pattern(const std::string& name, const std::string& pattern,
			   gps_type tMin) const {
   int compare = 0;
   size_t Ni = pattern.size();
   size_t Nj = name.size();
   size_t i = 0, j = 0; 
   while (!compare && i<Ni && j<Nj) {
      switch (pattern[i]) {
      case '*': 
	 i++;
	 if (i == Ni) {
	    j = Nj;
	    break;
	 }
	 while (j<Nj && name[j++] != pattern[i]);
	 i++;
	 break;
      case '%':
	 {
	    i++;
	    size_t N = scan_digits(pattern, i);
	    size_t t = scan_digits(name, j);
	    if (pattern[i] == 'g') {
	       if (tMin) {
		  if (N) for (size_t ix=0; ix < 10-N; ix++) t *= 10;
		  if (t < tMin) {
		     compare = 1;
		  }
	       }
	    } else if (pattern[i] == 'r') {
	       if (tMin) {
		  for (size_t ix=0; ix < N; ix++) t *= 10;
		  if (t < tMin) {
		     compare = 1;
		  }
	       }
	    }
	    i++;
	 }
	 break;
      default:
	 if (pattern[i++] != name[j++]) {
	    compare = 1;
	 }
      }
   }
   if (!compare && (i != Ni || j != Nj)) {
      compare = 1;
   }
   return compare == 0;
}

//======================================  Open the inotify device
bool 
file_stream::open_notify(void) {
   mNotifyFd = inotify_init();
   return (mNotifyFd >= 0);
}

//======================================  Add files and a book a pattern
void 
file_stream::parse_pattern(const std::string& pattern) {
   if (pattern.empty()) return;

   //-----------------------------------  resolve add working directory. 
   string temp;
   if (pattern[0] == '/') {
      temp = pattern;
   } else {
      char mycwd[1024];
      if (!getcwd(mycwd, sizeof(mycwd))) {
	 throw logic_error("file_stream: cwd buffer too short");
      }
      temp = mycwd;
      if (pattern.substr(0,2) == "./" ) {
	 temp += pattern.substr(1);
      } else {
	 temp += "/";
	 temp += pattern;
      }
   }

   //-----------------------------------  Skip constant portion 
   string::size_type wild_index  = temp.find_first_of("*%");
   string::size_type slash_index = temp.find_last_of('/', wild_index);
   pattern_list.push_back(temp.substr(0, slash_index));
   if (slash_index != string::npos) temp.erase(0,slash_index+1);
   while (!temp.empty()) {
      slash_index = temp.find_first_of('/');
      pattern_list.push_back(temp.substr(0, slash_index));
      if (slash_index != string::npos) slash_index++;
      temp.erase(0, slash_index);
   }
}

//======================================  Add files and a book a pattern
void 
file_stream::pop_event(void) {
   if (event_list.empty()) return;
   size_t N = event_list.size() - 1;
   // cerr << "pop_event: remove watch: " << event_list[N].first 
   //	   << "[" << event_list[N].second << "]" << endl;
   int id = inotify_rm_watch(mNotifyFd, event_list[N].second);
   if (id < 0) {
      throw runtime_error(string("file_stream: error removing watch ") 
			  + event_list[N].first);
   }
   event_list.pop_back();
}

//======================================  Add files and a book a pattern
void 
file_stream::push_event(const std::string& path) {
   if (!is_open()) {
      throw runtime_error("file_stream: inotify device not open");
   }
   uint32_t mask = IN_CREATE + IN_MOVED_TO;
   int      id   = inotify_add_watch(mNotifyFd, path.c_str(), mask);
   if (id < 0) {
      throw runtime_error(string("file_stream: unable to watch ") + path);
   }
   event_list.push_back(event_list_node(path, id));
   //cerr << "push_event: added watch: " << path << "[" << id << "]" << endl;
}

//======================================  Add files and a book a pattern
void 
file_stream::replace_event(const std::string& path) {
   pop_event();
   push_event(path);
}

//======================================  print notify
inline void
print_notify(ostream& out, struct inotify_event& event) {
   out << "wd:     " << event.wd << endl;
   uint32_t mask = event.mask;
   out << "mask:   " << hex << mask << dec;
   if (mask) {
      out << " (";
      if ((mask & IN_ACCESS) != 0)        out << "access,";
      if ((mask & IN_MODIFY) != 0)        out << "modify,";
      if ((mask & IN_ATTRIB) != 0)        out << "attrib,";
      if ((mask & IN_CLOSE_WRITE) != 0)   out << "write,";
      if ((mask & IN_CLOSE_NOWRITE) != 0) out << "nowrite,";
      if ((mask & IN_OPEN) != 0)          out << "open,";
      if ((mask & IN_MOVED_FROM) != 0)    out << "moved-from,";
      if ((mask & IN_MOVED_TO) != 0)      out << "moved-to,";
      if ((mask & IN_CREATE) != 0)        out << "create,";
      if ((mask & IN_DELETE) != 0)        out << "delete,";
      if ((mask & IN_DELETE_SELF) != 0)   out << "delete-self,";
      if ((mask & IN_MOVE_SELF) != 0)     out << "move-self,";
      out << ")" << endl;
   }
   out << "cookie: " << event.cookie << endl;
   out << "len:    " << event.len << endl;
}

//======================================  Add files and a book a pattern
bool 
file_stream::read_event(void) {
   bool return_code = false;
   struct inotify_event event;
   while (test_event()) {
      size_t length = sizeof(event);
      int rc = read(mNotifyFd, &event, length);
      if (rc != int(length)) {
	 if (rc < 0) perror("file_stream: error reading event");
	 else        cerr << "file_stream: read return code: " << rc << endl; 
	 throw runtime_error("file_stream: failed to read an event");
      }

      // cerr << "received event:" << endl;
      // print_notify(cerr, event);

      if (!event.len) continue;
      char* name = new char[event.len];
      name[0] = 0;
      rc = read(mNotifyFd, name, event.len);
      if (rc != int(event.len)) {
	 delete[] name;
	 if (rc < 0) perror("file_stream: error reading path");
	 else        cerr << "file_stream: read return code: " << rc << endl;
	 throw runtime_error("file_stream: failed to read event path");
      }
      // cerr << "name:   " << name << endl;

      //---------------------------------  Find the 
      size_t N = event_list.size();
      size_t inx = N;
      for (size_t i=0; i<N; i++) {
	 if (event.wd == event_list[N-1].second) {
	    inx = i;
	    break;
	 }
      }
      if (inx == N) {
	 delete[] name;
	 throw logic_error("file_stream: Unexpected event id");
      }

      if (match_pattern(name, pattern_list[inx+1])) {
	 if (inx == N-1) {
	    insert_file(name);
	 } else {
	    for (size_t i=inx; i<N-1; i++) pop_event();
	    push_event(name);
	 }
      }
      delete[] name;
   }
   return return_code;
}

//======================================  Add files and a book a pattern
bool
   file_stream::test_event(void) const {
   return (wait_event(0.0) > 0);
}

//======================================  Add files and a book a pattern
bool 
file_stream::test_file(void) {
   while (test_event()) read_event();
   return !file_list.empty();
}

//======================================  Wait for a notify event
int
file_stream::wait_event(double maxtime) const {
   if (!is_open()) throw runtime_error("notify device not open");

   //-------------------------------  Get the timeout specifier
   struct timeval t, *timeout;
   if (maxtime < 0) {
      timeout = NULL;
   } else {
      long nSec = (long) maxtime;
      t.tv_sec  = nSec;
      t.tv_usec = (long) ((maxtime - nSec) * 1000000.0);
      timeout = &t;
   }

   //-------------------------------  Set up the socket mask            */
   fd_set fdmask;
   FD_ZERO (&fdmask);
   FD_SET (mNotifyFd, &fdmask);

   /*-------------------------------  Wait for the condition            */
   int nset = select(mNotifyFd+1, &fdmask, NULL, NULL, timeout);
   return nset;
}

//======================================  Add files and a book a pattern
bool 
file_stream::wait_file(void) {
   while (file_list.empty()) {
      if (wait_event(1.0) > 0) read_event();
   }
   return true;
}
