/* -*- mode: c++; c-basic-offset: 3; -*- */
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: frametype						*/
/*                                                         		*/
/* Module Description: Frame data types					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 26Sep00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: framefast.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FRAMETYPE_H
#define _LIGO_FRAMETYPE_H


#include <iosfwd>
#include "framefast/fftype.hh"


namespace framefast {


/** @name Frame data types
    This header exports data types used by the fast frame reader and
    writer.
   
    @memo Frame data types
    @author Written October 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/

//@{


   /// maximum size of names
   const int maxName = 64;
   /// maximum number of elements in a dictionary structure
   const int maxDictElements = 100;

   /// PTR_STRUCT
   struct ptr_struct {
      /// class id
      int_2u_t 	fDataClass;
      /// instance counter
      int_4u_t	fDataInstance;
   
      /// Initialize with zeros
      ptr_struct();
      /// Read from frame
      long read (int version, const char* p, bool swapit);
      /// Write to frame
      long write (int version, char* p, bool swapit) const;
      /// set class id from name and reset instance counter
      bool set (int version, const char* name);
      /// increment pointer
      void operator++ (int) {
         fDataInstance++; }
   };

   /// Type ID for int_1s_t
   const int typeID_int_1s_t = 0;
   /// Type ID for int_2s_t
   const int typeID_int_2s_t = 1;
   /// Type ID for real_8_t
   const int typeID_real_8_t = 2;
   /// Type ID for real_4_t
   const int typeID_real_4_t = 3;
   /// Type ID for int_4s_t
   const int typeID_int_4s_t = 4;
   /// Type ID for int_8s_t
   const int typeID_int_8s_t = 5;
   /// Type ID for complex_8_t
   const int typeID_complex_8_t = 6;
   /// Type ID for complex_16_t
   const int typeID_complex_16_t = 7;
   /// Type ID for char*
   const int typeID_string = 8;
   /// Type ID for int_2u_t
   const int typeID_int_2u_t = 9;
   /// Type ID for int_4u_t
   const int typeID_int_4u_t = 10;
   /// Type ID for int_8u_t
   const int typeID_int_8u_t = 11;
   /// Type ID for int_1u_t
   const int typeID_int_1u_t = 12;

   /// Frame header name
   const char* const kFrameH = "FrameH";
   /// Detcetor structure name
   const char* const kFrDetector = "FrDetector";
   /// History structure name
   const char* const kFrHistory = "FrHistory";
   /// Raw data structure name
   const char* const kFrRawData = "FrRawData";
   /// ADC data structure name
   const char* const kFrAdcData = "FrAdcData";
   /// Vector structure name
   const char* const kFrVect = "FrVect";
   /// End of frame structure name
   const char* const kFrEndOfFrame = "FrEndOfFrame";
   /// Table of contents structure name
   const char* const kFrTOC = "FrTOC";
   /// End of file structure name
   const char* const kFrEndOfFile = "FrEndOfFile";


   /// Frame header
   struct fileheader_t {
      /// Magic number
      int_1s_t		fIGWD[5];
      /// version
      int_1u_t		fVersion;
      /// minor version
      int_1u_t		fMinorVersion;
      /// size of int2
      int_1u_t		fSize_int2;
      /// size of int4
      int_1u_t		fSize_int4;
      /// size of int8
      int_1u_t		fSize_int8;
      /// size of single
      int_1u_t		fSize_real4;
      /// size of double
      int_1u_t		fSize_real8;
      /// byte order (short)
      int_2u_t		fByteOrder2;
      /// byte order (int)
      int_4u_t		fByteOrder4; // alignment
      /// byte order (long long)
      int_8u_t		fByteOrder8; // alignment
      /// Pi in single
      real_4_t		fPi4;
      /// Pi in double
      real_8_t		fPi8; // alignment
      /// A an Z
      int_1s_t		fAlpha[2]; // library, checksum type for V >= 8
   
      /// Initialize with zeros
      fileheader_t();
      /// Initializes with valid version 4
      void init (int version);
      /// Read from frame
      long read (const char* p);
      /// Write to frame
      long write (char* p, bool swap = false) const;
   };


   /// Generic frame structure
   struct generic_t {
      /// length of structure
      int_8u_t		fLen;          // 4u in V<5
      /// Class identifier
      int_2u_t		fCheck;        // 1u, introduced in V8
      /// Class identifier
      int_2u_t		fClass;        // 1u in V>7
      /// Instance counter
      int_4u_t		fInstance;     // 2u in V<5
   
      /// Initialize with zeros
      generic_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false, 
		 bool skipSH = false);

      /// Write structure header to frame. This does NOT add a checksum.
      long write (int version, char* p, bool swap = false) const;

      ///  Fix structure length after writing. Add a checksum for V8 and later.
      ///  Return the total structure length (with checksum)
      long fixlength (int version, char* p, int_8u_t len, 
                     bool swapit = false) const;
   };


   /// Frame dictionary element
   struct dict_element_t : public generic_t {
      /// Name of structure described in this doctionary
      char		fName[maxName];
      /// class number of structure being described
      char		fType[maxName];
      /// comment
      char		fComment[maxName];
      /// Checksum
      int_4u_t		fChkSum;     // Only in V>8
   
      /// Initialize with zeros
      dict_element_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Frame dictionary header
   struct dict_header_t : public generic_t {
      /// Name of structure described in this doctionary
      char		fName[maxName];
      /// class number of structure being described
      int_2u_t		fClassNum;
      /// comment
      char		fComment[maxName];
      /// Checksum
      int_4u_t		fChkSum;      // Only in V>8
      /// structure elements
      dict_element_t*	fElements;
      /// Number of structure elements
      int_2u_t		fElementNum;
   
      /// Initialize with zeros
      dict_header_t();
      /// Copy constructor
      dict_header_t (const dict_header_t& dicth);
      /// Destructor
      ~dict_header_t();
      /// Assignment operator
      dict_header_t& operator= (const dict_header_t& dicth);
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Frame dictionary
   struct dict_t {
      /// List of dictionary headers
      dict_header_t	fDict[maxDictElements];
      /// Number of dictionary headers
      int_2u_t		fDictNum;
   
      /// Initialize with zeros
      dict_t() : fDictNum (0) {
      }
      /// Initialize with standard structures
      void standard (int version);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Frame header
   struct frameheader_t : public generic_t {
      /// Project name
      char		fName[maxName];
      /// Run number
      int_4s_t		fRun;
      /// Frame number
      int_4u_t		fFrame;
      /// Data qulaity bit
      int_4u_t		fDataQual;
      /// GPS sec
      int_4u_t		fGTimeS;
      /// GPS nano sec
      int_4u_t		fGTimeN;
      /// Leap seconds
      int_2u_t		fULeapS;
      /// local time (V4 only)
      int_4s_t		fLocalTime;
      /// frame length in sec
      real_8_t		fFrameLen;
      /** Pointers to data structures: type, user, detector simulation,
          detector info, history information, raw data,  
          processed data, strain data (V4 only), simulated data, triggered data,
          simulated events, summary data, aux. data, aux. table */
      ptr_struct	fDir[14];
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      frameheader_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Detector structure
   struct detector_t : public generic_t  {
      /// Detector name
      char		fName[maxName];
      /// prefix
      char		fPrefix[4];
      /// Longitude degree
      int_2s_t		fLongitudeD;
      /// Longitude minute
      int_2s_t		fLongitudeM;
      /// Longitude second
      real_4_t		fLongitudeS;
      /// Latitude degree
      int_2s_t		fLatitudeD;
      /// Latitude minute
      int_2s_t		fLatitudeM;
      /// Latitude second
      real_4_t		fLatitudeS;
      /// Elevation (V4)
      real_4_t		fElevation;
      /// X arm azimuth
      real_4_t		fArmXAzimuth;
      /// Y arm azimuth
      real_4_t		fArmYAzimuth;
      /// X arm altitude (V5 and up)
      real_4_t		fArmXAltitude;
      /// Y arm altitude (V5 and up)
      real_4_t		fArmYAltitude;
      /// X arm mid point (V5 and up)
      real_4_t		fArmXMidpoint;
      /// Y arm mid point (V5 and up)
      real_4_t		fArmYMidpoint;
      /// local time (V5 and up)
      int_4s_t		fLocalTime;
      /// data quality (V5 and up)
      int_4u_t		fDataQuality;
      /// quality bit list (V5 and up)
      char		fQaBitList[maxName];
   
      /// Pointers to other structures: more & moreTable & next (V5 an up);
      ptr_struct	fMore[3];
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      detector_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
      /// Get longitude
      real_8_t get_longitude() const;
      /// Set longitude
      void set_longitude (real_8_t lon);
      /// Get latitude
      real_8_t get_latitude() const;
      /// Set latitude
      void set_latitude (real_8_t lat);
   };


   /// History structure
   struct hist_t : public generic_t  {
      /// History record name
      char		fName[maxName];
      /// Time of post processing
      int_4u_t		fTime;
      /// Comment
      char		fComment[maxName];
      /// Pointers to next history structure;
      ptr_struct	fNext;
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      hist_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Raw data structure
   struct rawdata_t : public generic_t  {
      /// Raw data name
      char		fName[maxName];
      /// Pointers to data structures: serial, ADC, table, Msg, more
      ptr_struct	fData[5];
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      rawdata_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };

   /// Type of data structure (must be in this order!)
   enum datatype_t {
   /// ADC sampled data
   kAdcData = 0,
   /// processed data
   kProcData = 1,
   /// simulated data
   kSimData = 2,
   /// serial data
   kSerData = 3
   };

   /// ADC Data
   struct adcdata_t : public generic_t {
      /// Data type
      datatype_t	fDatatype;
      /// Channel name
      char		fName[maxName];
      /// Comment
      char		fComment[maxName];
      /// channel group
      int_4u_t		fChannelGroup;
      /// channel number
      int_4u_t		fChannelNumber;
      /// Number of bits in ADC
      int_4u_t		fNBits;
      /// DC offset (ADC counts)
      real_4_t		fBias;
      /// slope (unit/count)
      real_4_t		fSlope;
      /// unit name
      char		fUnit[maxName];
      /// sample rate
      real_8_t		fSampleRate;
      /// time offset (sec)
      int_4s_t		fTimeOffsetS;
      /// time offset (nsec)
      int_4u_t		fTimeOffsetN;
      /// frequency shift
      real_8_t		fFShift;
      /// phase (V5 annd up)
      real_4_t		fPhase;
      /// Data validity
      int_2u_t		fDataValid;
      /// Pointers to data structures: data, user, next ADC, table (V4 only)
      ptr_struct	fData[4];
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      adcdata_t();
      /// Read from frame
      long read (int version, datatype_t dtype, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// Vector Data
   struct frvect_t : public generic_t {
      /// Data copy type
      enum datacopy {
      /// No copy
      fv_nocopy = 0,
      /// Copy
      fv_copy = 1,
      /// Avoid copy if aligned & no swap & no compression
      fv_original = 2
      };
   
      /// Channel name
      char		fName[maxName];
      /// Compression scheme
      int_2u_t		fCompress;
      /// Vector class
      int_2u_t		fType;
      /// Number of data points
      int_8u_t		fNData;        // 4u for V4
      /// Number of bytes in the data vector
      int_8u_t		fNBytes;       // 4u for V4
      /// Number of bytes in the original (compressed) vector
      int_8u_t		fNCompBytes;   // only useful for dump
      /// N dimensions
      int_4u_t		fNDim;
      /// X dimensions
      int_8u_t		fNx[4];        // 4u for V4
      /// X slope
      real_8_t		fDx[4];
      /// X offset
      real_8_t		fStartX[4];
      /// X units	
      char		fUnitX[4][maxName];
      /// Y units
      char		fUnitY[maxName];
      /// Next vector
      ptr_struct	fNext;
      /// Structure checkSum
      int_4u_t		fChkSum;     // 2u in V<5
   
      /// Initialize with zeros
      frvect_t();
      /// Copy contructor
      frvect_t (const frvect_t& vect);
      /// Destruct vector
      ~frvect_t();
      /// Assignment operator
      frvect_t& operator= (const frvect_t& vect);
   
      /// allocate memory
      bool allocate (int_2u_t type, int size);
      /// deallocate memory
      void deallocate ();
      /// clone structure without copying data
      void clone (const frvect_t& templ);
      /// clone structure with data reference
      void image (const frvect_t& templ);
      /// Read from frame; skip FrSh structures if required
      long read (int version, const char* p, bool swap = false, datacopy cpy = fv_copy);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
      /// copy real_4 data
      int get (real_4_t* dat, int max) const;
      /// copy real_8 data
      int get (real_8_t* dat, int max) const;
      /// copy complex_16 data
      int get (complex_8_t* dat, int max) const;
      /// copy complex_16 data
      int get (complex_16_t* dat, int max) const;
      /// copy int_2s data
      int get (int_2s_t* dat, int max) const;
      /// copy int_4s data
      int get (int_4s_t* dat, int max) const;
      /// Fill data into container
      bool fill (int_4u_t pos, int_4u_t num, const frvect_t& src, 
                int_4u_t ofs = 0, real_8_t mul = 1);
   
      /// Fill data into container
      bool fill (int_4u_t pos, int_4u_t num, const char* values,
                bool swapit = false);
      /// get data vector
      const char* data (int offset = 0) const {
         return fData ? ((const char*) fData + offset) : 0; }
   
   private:
      /// Pointer to data
      void*		fData;
      /// own data?
      mutable bool	fOwn;
   };


   /// Data structure
   struct data_t {
      /// ADC header
      adcdata_t		fADC;
      /// Data vector
      frvect_t		fVect;
   
      // Constructor
      data_t();
      /// Destructor
      ~data_t();
      /// Allocate an empty array
      bool allocate (int_2u_t type, int size);
      /// Dellocate array
      void deallocate ();
      /// Clone the header(s) from a template data structure
      void clone (const data_t& templ);
      /** Fill data into container.
          @param pos fill position
          @param num Number of points
          @param src source vector
          @param ofs Offset into soure vector
          @param mul Data rate multiplication factor
          @return true if successful
       */
      bool fill (int_4u_t pos, int_4u_t num, const data_t& src, 
                int_4u_t ofs = 0, real_8_t mul = 1);
   };


   /// End of frame structure
   struct endof_frame_t : public generic_t  {
      /// Run number
      int_4s_t		fRun;
      /// Frame number
      int_4u_t		fFrame;
      /// check type
      int_4u_t		fChkType;
      /// check type
      int_4u_t		fGTimeS;
      /// check type
      int_4u_t		fGTimeN;
      /// checksum
      int_4u_t		fChkSum;
   
      /// Initialize with zeros
      endof_frame_t();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
   };


   /// End of file structure 
   struct endof_file_t : public generic_t {
      /// Number of frames
      int_4u_t		fNFrames;
      /// Number of bytes
      int_8u_t		fNBytes;       // 4u for V4
      /// checksum flag
      int_4u_t		fChkType;
      /// check sum
      int_4u_t		fChkSum;
      /// TOC offset
      int_8u_t		fSeekTOC;      // 4u for V4
      /// FrHeader checksum
      int_4u_t		fHeaderChkSum;
      /// FrHeader checksum
      int_4u_t		fileChkSum;
   
      /// Initialize with zeros
      endof_file_t ();
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Read from end of frame
      long read (int version, const char* framestart, int_8u_t framelen,
               bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
      /// Size of this structure in frame file
      static int size (int version);
   };


   /// TOC: Frame entry
   struct toc_frame_t {
      /// Data quality (V5 and up)
      int_4u_t		fDataQual;
      /// GPS sec
      int_4u_t		fGTimeS;
      /// GPS nano sec
      int_4u_t		fGTimeN;
      /// Frame length in sec
      real_8_t		fDt;
      /// Run number
      int_4s_t		fRun;
      /// Frame number
      int_4u_t		fFrame;
      /// Position of frame header
      int_8u_t		fPositionH;
      /// Position of first ADC struct
      int_8u_t		fNFirstADC;
      /// Position of first ser struct
      int_8u_t		fNFirstSer;
      /// Position of first table struct
      int_8u_t		fNFirstTable;
      /// Position of first msg struct
      int_8u_t		fNFirstMsg;
   
      /// Initialze with zeros
      toc_frame_t ();
   };

   /// TOC: Frame structure header entry
   struct toc_SH_t {
      /// ID
      int_2u_t		fSHid;
      /// Name	
      char		fSHName[maxName];
      /// Initialze with zeros
      toc_SH_t ();
   };

   /// TOC: Detector entry
   struct toc_detector_t {
      /// Detector name
      char 		fDetector[maxName];
      /// Address
      int_8u_t		fPos;
   
      /// Initialze with zeros
      toc_detector_t ();
   };

   /// TOC: Static data block entry
   struct toc_stat_t {
      /// Name
      char 		fName[maxName];
      /// Detector name
      char 		fDetector[maxName];
      /// Number of instances
      int_4u_t		fStatInstance;
      // skip rest
   
      /// Initialze with zeros
      toc_stat_t ();
   };

   /// TOC: Data block entry (IMPORTANT: use types w/o constructors
   struct toc_data_t {
      /// name
      char 	fName[maxName];
      /// channel id
      int_4u_t	fChannelID;
      /// channel group
      int_4u_t	fGroupID;
      /// array of data positions (array length = # of frames)
      int_8u_t*	fPosition;
   
      /// Initialze with zeros
      toc_data_t ();
      /// comparison (name only)
      bool operator== (const toc_data_t& d2) const;
      /// ordering (name only)
      bool operator< (const toc_data_t& d2) const;
   };

   /// TOC: Event entry
   struct toc_Event_t {
      /// Name
      char 	fName[maxName]; // must be the first element!
      /// GPS sec
      int_4u_t	fGTimeS;
      /// GPS nano sec
      int_4u_t	fGTimeN;
      /// GPS nano sec
      real_4_t	fAmplitude;
      /// Position of event
      int_8u_t	fPosition;
   
      /// Initialze with zeros
      toc_Event_t ();
   };

   /// Table of contents (TOC)
   struct toc_t : public generic_t {
      /// Leap sec
      int_2s_t		fULeapS;
      /// loacl time (V4 only)
      int_4s_t		fLocalTime;
      /// Number of frames
      int_4u_t		fNFrame;
      /// List of frames
      toc_frame_t*	fFrames;
      /// Number of frame structure headers
      int_4u_t		fNSH;
      /// List of frame structure headers
      toc_SH_t*		fSH;
      /// Number of detector information records
      int_4u_t		fNDetector;
      /// Detector information records
      toc_detector_t*	fDetInfo;
      /// Number of static block types
      int_4u_t		fNStatType;
      /// Total number of static blocks
      int_4u_t		fNTotalStat;
      /// List of static blocks
      toc_stat_t*	fStat;
      /// Number of data blocks: ADC, Proc, Sim, Ser, Summary
      int_4u_t		fNData[5];
      /// List of data blocks: ADC, Proc, Sim, Ser, Summary
      toc_data_t*	fData[5];
      /// Number of event types
      int_4u_t		fNEvt[2];

      /// Number of Total events.
      int_4u_t		fEvtTotal[2];
      /// List of events
      toc_Event_t*	fEvt[2];

      /// Initialize with zeros
      toc_t();
      /// Copy constructor
      toc_t (const toc_t& toc);
      /// Destructor
      ~toc_t();
      /// Copy operator
      toc_t& operator= (const toc_t& toc);
      /// Allocate array for channel entries
      bool allocate (int newsize, int datablck = 0);
      /// initialize TOC; set fNFrame before calling
      bool init (int version);
      /// Read from frame
      long read (int version, const char* p, bool swap = false);
      /// Scan through frame and regenerate TOC (call init first)
      bool scan (const char* p, int len, bool swap = false);
      /// Write to frame
      long write (int version, char* p, bool swap = false) const;
      /// Size the TOC would take up when written to frame
      int size (int version) const;
   
      /// find a channel entry
      toc_data_t* find (const char* chnname) const;
      /// find a channel entry
      toc_data_t* find (const char* chnname, int datablck) const;
      /// add a channel entry
      toc_data_t* add (const char* chnname, int datablck = 0);
   
      /// find detector entry
      toc_detector_t* findDetector (const char* name) const;
      /// add detector entry
      toc_detector_t* addDetector (const char* name);
   
   private:
      /// Buffer (do not use)
      int_8u_t*		fDataPosBuf[5];
      /// maximum number of channels
      int_4s_t		fChannelMax[5];
   };



/** Dump a frame.
    @memo Dump frame.
    @param os Output stream
    @param p pointer to frame
    @param len Length of frame
    @return Output stream
 ************************************************************************/
   std::ostream& dumpframe (std::ostream& os, const char* p, int len);

/** Output operator for writing the frame file header.
    @memo Output operator.
    @param os Output stream
    @param h Frame file header
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const fileheader_t& h);
   std::ostream& dump (std::ostream& os, const fileheader_t& h);

/** Output operator for writing a dictionary element.
    @memo Output operator.
    @param os Output stream
    @param el Dictionary element
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const dict_element_t& el);
   std::ostream& dump (std::ostream& os, const dict_element_t& el,
                     int version = 4);

/** Output operator for writing a dictionary header.
    @memo Output operator.
    @param os Output stream
    @param h Dictionary header
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const dict_header_t& h);
   std::ostream& dump (std::ostream& os, const dict_header_t& h,
                     int version = 4, bool elements = true);

/** Output operator for writing a dictionary.
    @memo Output operator.
    @param os Output stream
    @param dict Dictionary
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const dict_t& dict);
   std::ostream& dump (std::ostream& os, const dict_t& dict,
                     int version = 4);

/** Output operator for writing the frame header.
    @memo Output operator.
    @param os Output stream
    @param h Frame header
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const frameheader_t& h);
   std::ostream& dump (std::ostream& os, const frameheader_t& h,
                     int version = 4);

/** Output operator for writing the detector information.
    @memo Output operator.
    @param os Output stream
    @param det Detector information
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const detector_t& d);
   std::ostream& dump (std::ostream& os, const detector_t& d,
                     int version = 4);

/** Output operator for writing the history information.
    @memo Output operator.
    @param os Output stream
    @param hist History information
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const hist_t& hist);
   std::ostream& dump (std::ostream& os, const hist_t& hist,
                     int version = 4);

/** Output operator for writing the raw data information.
    @memo Output operator.
    @param os Output stream
    @param raw Raw data information
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const rawdata_t& d);
   std::ostream& dump (std::ostream& os, const rawdata_t& d,
                     int version = 4);

/** Output operator for writing the adc data structure.
    @memo Output operator.
    @param os Output stream
    @param adc adc structure
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const adcdata_t& adc);
   std::ostream& dump (std::ostream& os, const adcdata_t& adc,
                     int version = 4);

/** Output operator for writing the frame vector.
    @memo Output operator.
    @param os Output stream
    @param vec Frame vector
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const frvect_t& vec);
   std::ostream& dump (std::ostream& os, const frvect_t& vec,
                     int version = 4);

/** Output operator for writing the frame eof frame marker.
    @memo Output operator.
    @param os Output stream
    @param eof End of frame marker
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const endof_frame_t& eof);
   std::ostream& dump (std::ostream& os, const endof_frame_t& eof,
                     int version = 4);

/** Output operator for writing the frame eof file marker.
    @memo Output operator.
    @param os Output stream
    @param eof End of file marker
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const endof_file_t& eof);
   std::ostream& dump (std::ostream& os, const endof_file_t& eof,
                     int version = 4);

/** Output operator for writing the table of content.
    @memo Output operator.
    @param os Output stream
    @param fr Frame reader
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const toc_t& toc);
   std::ostream& dump (std::ostream& os, const toc_t& toc,
                     int version = 4);


   /// Forward declaration of frame reader
   class framereader;
   /// Forward declaration of frame writer
   class framewriter;

//@}


}

#endif /* _LIGO_FRAMEFAST_H */
