//
//    DMT Data processing base class implementation.
//
//======================================  Header files.
#include "TrigEnv.hh"

#ifndef __CINT__
#include <iostream>
#include <fstream>
#include "Interval.hh"
#include "Time.hh"
#include <stdexcept>
#include <string>

#include <stdio.h>

#else
#define DACC_ONLDEV "/online/"

#endif // __CINT__

using namespace std;

//======================================  TrigEnv constructor.
TrigEnv::TrigEnv(int argc, const char *argv[]) 
  : mActive(false), mEpoch(1.0), mOffset(0.0), mStride(1.0), mDebug(0),
    mTrigRead(0), mTrigProc(0), mProcStride(0)
{
    string tableFile;

    //----------------------------------  Parse the arguments
    bool syntax=false;
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-infile") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -infile." << endl;
		syntax= true;
	        break;
	    }
	    mFrDir.add(argv[i]);
	} else if (argi == "-inlist") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -inlist." << endl;
		syntax= true;
	        break;
	    }
	    ifstream inf(argv[i]);
	    if (!inf.is_open()) {
	        cerr << "Unable to open file " << argv[i] << endl;
		continue;
	    }
	    string fname;
	    while (inf.good()) {
	        inf >> fname;
		while (fname.size() > 0 && fname[0] == ' ') fname.erase(0,1);
		int l = fname.size();
		while (l>0 && (fname[l-1]==' ' || fname[l-1]=='\n')) {
		    fname.erase(--l);
		}
		if (l>0) mFrDir.add(fname.c_str());
	    }
	} else if (argi == "-debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	    getDacc().setDebug(mDebug);
	} else if (argi == "-stride") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -stride." << endl;
		syntax= true;
	        break;
	    }
	    setStride(strtod(argv[i], 0));
	} else if (argi == "-tepoch") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -tepoch." << endl;
		syntax= true;
	        break;
	    }
	    setEpochLength(strtod(argv[i], 0));
	} else if (argi == "-toffset") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -toffset." << endl;
		syntax= true;
	        break;
	    }
	    setEpochOffset(strtod(argv[i], 0));
	} else if (argi == "-ttable") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -ttable." << endl;
		syntax= true;
	        break;
	    }
	    tableFile = argv[i];
	}
    }

    //----------------------------------  Get the table file
    if (syntax) {
        cerr << "Syntax error in monitor command line syntax" << endl;
	return;
    }

    //----------------------------------  Get the table file
    if (tableFile.empty()) {
        cout << "Syntax error in TrigEnv: No table specified." << endl;
	finish();
	return;
    }

    if ( mMetaIO.open(tableFile.c_str()) || !mMetaIO.is_open() ) {
        cout << "Unable to open trigger table file: " << tableFile << endl;
	finish();
	return;
    } else {
        cout << "Opened trigger table file: " << tableFile << endl;
    }

    //----------------------------------  Add DMTINPUT data path
    if (mFrDir.empty()) {
        const char* in = getenv("DMTINPUT");
	if (in) mFrDir.add(in);
    }
    if (Debug()) cout << "Frame directory contains " << mFrDir.size() 
		      << " files." << endl;

#ifndef __CINT__
    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    mAttn.setMode(SigFlag::kBlock);
    mAttn.add(SIGUSR1);
    mAttn.add(SIGIO);

#else
    gROOT->SetInterrupt(kTRUE);
#endif   // !def(__CINT__)

    //----------------------------------  Open a frame stream (FrameCPP)
    mActive = true;
}

//======================================  Test if TrigEnv argument
bool
TrigEnv::isTrigEnvArg(const char* argc) const {
    string arg(argc);
    if      (arg == "-infile")  return true;
    else if (arg == "-inlist")  return true;
    else if (arg == "-debug")   return true;
    else if (arg == "-stride")  return true;
    else if (arg == "-tepoch")  return true;
    else if (arg == "-toffset") return true;
    else if (arg == "-ttable")  return true;
    return false;
}

//======================================  TrigEnv object destructor.
TrigEnv::~TrigEnv(void) {
    //----------------------------------  DMT has terminater
    cout << "Trigger environment has terminated with Term/Attn/finish =" 
	 << bool(mTerm) << "/" << bool(mAttn) << "/" << !mActive << endl;
    cout << "Triggers read/Triggers processed/Strides processed =" 
	 << mTrigRead << "/" << mTrigProc << "/" << mProcStride << endl;

#ifndef __CINT__
    //----------------------------------  Release signals
    mTerm.zero();
    mAttn.zero();

#endif //  __CINT__

    //----------------------------------  Close the Input file/partition.
    getDacc().close();
}

//======================================  Main processing loop function.
void
TrigEnv::MainLoop(void) {

    //----------------------------------  Loop until terminated.
    bool newFill = true;
    while(!testTerm() && mActive) {

        //------------------------------  Check for an attention interrupt.
        if (testAttn()) {
#ifndef __CINT__
	    mAttn.clear();
#endif
	    Attention();
	    if (!mActive) continue;
	}

        //------------------------------  Get the Trigger 
	if (readTrigger()) break;

	if (Debug()) cout << "Read trigger: " << mTrigger.getID()  
			  << " subID: " << mTrigger.getSubID() << " Time: " 
			  << mTrigger.getTime() << endl;

	mTrigRead++;

	try {
	    bool proc = ProcessTrigger(mTrigger);
	    if (!proc || mEpoch <= Interval(1.0E-09)) continue;
	} catch(exception& e) {
	    cerr << "Exception in ProcessTrigger: " << e.what() << endl;
	    break;
	} catch(...) {
	    cerr << "Unidentified exception in ProcessTrigger." << endl;
	    break;
	}

	//------------------------------  Set up the input epoch
	Time tStart(mTrigger.getTime()+mOffset);
	if (!tStart) {
	    cout << "Trigger time improperly specified." << endl;
	    continue;
	}
	Time tStop(tStart + mEpoch);
	if (Debug() > 3) cout << "Setting up frame directory from: " << tStart
			      << " to: " << tStop << endl;

	//------------------------------  Get the data, keep track of times.
	Time tEnd(tStart);
	for (FrameDir::file_iterator i=mFrDir.begin() ;
	     i != mFrDir.end() ; ++i) {
	    if (i->getStartTime() >= tStop)  continue;
	    Time EndTime(i->getEndTime());
	    if (EndTime <= tStart) continue;
	    if (EndTime > tEnd)    tEnd = EndTime;
	    string file = i->getFile();
	    getDacc().addFile(file.c_str());
	    if (Debug() > 3) cout << "Add frame file: " << file << endl;
	}
	if (tEnd < tStop) tStop  = tEnd;
	getDacc().seek(tStart);

        //------------------------------  Loop over time strides
	while(!testTerm() && mActive) {
	    Time t=getDacc().getCurrentTime();
	    if (t >= tStop || Almost(t,tStop)) break;
	    Interval nextStride = mStride;
	    if (t+nextStride > tStop) nextStride = tStop - t;

	    //--------------------------  Read a frame
	    int rc;
	    try {
	        rc = getDacc().fillData(nextStride, newFill);
	    } catch(exception& e) {
	        cerr << "Exception in fillData: " << e.what() << endl;
		finish();
		break;
	    }

	    //--------------------------  Crunch it
	    if (rc == 0) {
	        try {
		    ProcessData();
		    newFill = true;
		} catch(exception& e) {
		    cerr << "Exception in ProcessData: " << e.what() << endl;
		    finish();
		} catch(...) {
		    cerr << "Unidentified exception in ProcessData." << endl;
		    finish();
		}
	    } else if (rc == -8) {
	        newFill = false;
	    } else if (rc <= -4) {
	        cerr << "MainLoop: Error while reading frame." << endl;
		break;
	    } else if (Debug()) {
	        cerr << "MainLoop: Frame missing from input stream." << endl;
		newFill = true;
	    }
	    mProcStride++;
	}
	getDacc().close();
	mTrigProc++;
    }
}

//======================================  Read/Process trigger.
int
TrigEnv::readTrigger(void) {

    //----------------------------------  Get the next row; return if done
    int rc = mMetaIO.getRow();
    if (rc != 1) {
        if (Debug()) cout << "readTrigger failed with " << rc 
			  << " from  MetaioGetRow." << endl;
        return rc-1;
    }

    //----------------------------------  Get all specified fields.
    string name    = mMetaIO.getString("name");
    string subtype = mMetaIO.getString("subtype");

    if (Debug() > 3) cout << "Reading trigger " << name << ":" << subtype << endl;

    mTrigger = trig::TrigBase(name.c_str(), subtype.c_str(),
			      Time(mMetaIO.getInt("start_time"),
				   mMetaIO.getInt("start_time_ns")),
			      mMetaIO.getFloat("duration"),
			      mMetaIO.getFloat("size"));

    string ifo = mMetaIO.getString("ifo");
    mTrigger.setIfos (ifo.c_str());
    mTrigger.setPriority(trig::TrigPrio(mMetaIO.getInt("priority")));
    mTrigger.setDisposition(mMetaIO.getInt("disposition"));
    mTrigger.setSignificance(mMetaIO.getFloat("significance"));
    mTrigger.setFrequency(mMetaIO.getFloat("frequency"));
    mTrigger.setData (mMetaIO.getString("binarydata").c_str(),
		      mMetaIO.getInt("binarydata_length"));
    return 0;
}

//======================================  Read/Process trigger.
bool
TrigEnv::ProcessTrigger(const trig::TrigBase& t) {
    return true;
}

//======================================  Set the epoch length
void 
TrigEnv::setEpochLength(const Interval& t) {
    mEpoch = t;
}

//======================================  Set the epoch offset
void 
TrigEnv::setEpochOffset(const Interval& t) {
    mOffset = t;
}

//======================================  Set the stride length
void 
TrigEnv::setStride(const Interval& t) {
    mStride = t;
}

//======================================  Background interrupt handling.
#ifndef __CINT__  
void
TrigEnv::Attention(void) {
}

bool 
TrigEnv::testAttn(void) const {
    return mAttn;
}

bool 
TrigEnv::testTerm(void) const {
    return mTerm;
}

//======================================  Root interrupt handling.
#else   
void
TrigEnv::Attention(void) {
    mActive = false;
}

bool 
TrigEnv::testAttn(void) {
    gSystem->ProcessEvents();
    return gROOT->IsInterrupted();
}

bool 
TrigEnv::testTerm(void) {
    return false;
}
#endif
