/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DMTBASE_HH
#define DMTBASE_HH

//
//    Base class for the Data Monitor processing routines!
//

//-----------------------------------  Mention classes to be used.
#include "FramePointerTypes.hh"

class SigFlag;
class Dacc;

/**  The frame processing base class defines the basic routines necessary 
  *  for a frame processing application. The strategy for writing a monitor
  *  program is to create a class (myclass) based on DMTBase including a 
  *  constructor, destructor and ProcessFrame() method. EXECDMT(myclass) 
  *  is used to generate the main function. These steps are described in 
  *  DMTemplate.h and DMTemplate.cc. The base class constructor recognizes
  *  the following command line arguments: "-infile <file>" instructs the
  *  base class to read frames from the specified file, "-partition <part>"
  *  causes the frames to be read from the named shared memory partition.
  *  If no input option is specified, the frames are read from the partition
  *  specified by LIGOSMPART if this is undefined from "LIGO_Online".
  *  The debug print level can be specified with "-debug n".
  *  @memo Frame processor base class for data monitors.
  *  @author John G. Zweizig
  *  @version 1.3; Last modified August 3, 1999
  */
class DMTBase {
public:
  typedef frameh_pointer frame_ptr_type;

  public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  @memo Class constructor.
      *  @param argc Number of arguments defined.
      *  @param argv Address of a vector of pointers to the arguments.
      */
    DMTBase(int argc, const char *argv[]);

    /**  Close the frame stream and go away.
      *  @memo Class destructor.
      */
    virtual ~DMTBase(void);

    /**  This function is provided by the DMT base class. MainLoop reads 
      *  frames and invokes #ProcessFrame# for each frame successfully 
      *  read in. #MainLoop# exits when a #SIGINT# or #SIGTERM# signal is 
      *  caught, or when #finish# is called by a subsidary function (e.g. 
      *  by #ProcessFrame#).
      *  @memo Main loop over frames.
      */
    void MainLoop(void);

    /**  #ProcessFrame# performs the monitor specific processing on a single
      *  data frame. It must be defined as part of the derived monitor 
      *  class implementation.
      *  @memo Process a data frame (Supplied by user).
      *  @param frame pointer to a frame in the local format.
      */
    virtual void ProcessFrame(frame_ptr_type frame)=0;

    /**  #Attention# performs the monitor specific attention interrupt 
      *  processing. Its definition as part of the derived monitor class 
      *  is optional.
      *  @memo Attention interrupt handler (Supplied by user).
      */
    virtual void Attention(void) {}

    /**  #finish# stops processing by setting a flag that breaks the main 
      *  loop. Note that #finish# returns to the calling method (generally 
      *  #ProcessFrame#) which must exit normally for the process 
      *  to terminate.
      *  @memo Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  #Debug# returns the value specified by the "-debug nnn" command 
      *  line argument. If "-debug" isn't specified in the command line 
      *  it defaults to 0.
      *  @memo Get the debug level.
      *  @return Debug level.
      */
    int Debug(void) const {return mDebug;}

    /**  #Boundary# is set if the current frame is not contiguous in time 
      *  with the previously read frame. The first frame read by an 
      *  application is considered to be a boundary frame.
      *  @memo Test for a time-gap before this frame.
      *  @return true if the last frame read was a boundary frame.
      */
    bool Boundary(void) const {return mContig;}

    /**  Get a reference to the data accessor.
      *  @memo Get the data accessor.
      *  @return Data accessor reference.
      */
    Dacc& getDacc(void) {return *mIn;}

    /**  The frame count is incremented each time a new frame is read by
      *  the base class functions.
      *  @memo Get the number of frames processed.
      *  @return Number of frames read.
      */
    int  getNFrames(void) const {return mNFrames;}

    /**  This function returns a counter that is incremented every time a 
      *  boundary frame is found (see #Boundary#).
      *  @memo Get the number of discontinuities in the frame stream.
      *  @return The number of discontinuities found in the frame stream.
      */
    int  getNContig(void) const {return mNContig;}

    /**  this function returns true if the aprguemt is a command line
      *  keyword used by the base-class.
      */
    bool isEnvArg(const char* arg) const;

    /**  Set the #Dacc# buffering depth. The buffering depth is the number 
      *  of buffer that may be reserved for this process. If the the 
      *  buffering depth is specified as -1, all buffers will be read in.
      *  Note that setting a large number of buffer (or -1) could starve
      *  other processes of data if the current process is unable to keep 
      *  up with the data stream or if it terminates without releasing the
      *  input partition. The buffer depth has no effect when reading data 
      *  from files.
      *  @memo Set buffering depth.
      *  @param nbuf Number of buffers to be reserved or -1 to read all.
      */
    void  setBuffer(int nbuf);

  private:
    /**  Flag which when cleared will cause the base class to exit the frame
      *  loop and terminate the process. #mActive# is set after successful
      *  initialization and cleared by the #finish# method or an end-of-file
      *  condition.
      *  @memo Internal run flag
      */
    bool mActive;

    ///  Termination signal handling object
    SigFlag* mTerm;

    ///  Attention signal handling object
    SigFlag* mAttn;

    ///  Frame Input object
    Dacc*  mIn;
  
    ///  Debug Level
    int  mDebug;

    /**  mContig is set if the current frame is not contiguous. By 
      *  definition, the first frame is non-contiguous.
      *  @memo Contiguity flag.
      */
    bool mContig;

    /**  This is kept track of by the DMTBase code.
      *  @memo Number of frames read/processed.
      */
    int  mNFrames;

    /**  This counter is incremented every time a non-sequential frame is 
      *  found. The first frame read is counted as a discontinuous frame.
      *  This counter is maintained by the Base class code.
      *  @memo Number of discontinuities found in the frame stream.
      */
    int  mNContig;

    /**  mEOF is set when a frame read error occurs.
      *  @memo Frame read error.
      */
    bool mEOF;
};

//
//   The EXECDMT macro generates the main function to execute a class
//   based on DMTBase. The argument is whatever name you have chosen
//   for your class.
//
#define EXECDMT(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

#endif     //  _DMTBASE_H_
