/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Ramp.hh"
#include "Time.hh"
#include "constant.hh"
#include <math.h>

#define fMod(x,y)	((x) - (y) * floor ((x) / (y)))

//-------------------------------------  normalized phase. 
inline double normPhase (double phi)
{
   if ((phi >= 0) && (phi < twopi)) {
      return phi;
   }
   else {
      return fMod (phi, twopi);
   }
}

//-------------------------------------  Ramp Constructor. 
Ramp::Ramp(double Freq, double ampl, double phi0, 
                       const Interval& dT, const Time& t0)
  : mT0(t0), mTc(t0), mPhic(phi0), mAmpl(ampl)
{
    mOmega = twopi * Freq;
    mTl     = t0 + dT;
}

//-------------------------------------  Frequency domain template bin.
fComplex 
Ramp::Fspace(double Freq, double dF) const {
    fComplex r(0);
    double f = mOmega/twopi;
    if (f <= 0) {
       return r;
    }
    int n = int(2 * Freq / f + 1.5);
    if (n < 10) n = 10;
    for (int i = 0; i < 2*n+1; ++i) {
       double x = -pow((Freq - double(i)*f)/dF, 2);
       r += mAmpl * exp (fComplex (x, mPhic+i*pi)) / i;
    }
    return 2./pi*r;
}

//-------------------------------------  Time domain template bin.
double 
Ramp::Tspace(const Time& t0) const {
    return Ampl(t0) * normPhase (phi(t0)) / twopi;
}

//-------------------------------------  Get the t0.
Time 
Ramp::getT0(void) const {
    return mT0;
}

//-------------------------------------  Get the End time.
Time
Ramp::getTEnd(void) const {
    return mTl;
}

//-------------------------------------  Get the Critical time.
Time
Ramp::getTc(void) const {
    return mTc;
}

//-------------------------------------  Time versus Frequency.
Time
Ramp::TvsF(double f) const {
    return mT0;
}

//-------------------------------------  Waveform Frequency.
double
Ramp::freq(const Time& t) const {
    return mOmega/twopi;
}

//-------------------------------------  Waveform Phase angle.
double
Ramp::phi(const Time& t) const {
    return mOmega*double(t-mTc)-mPhic;
}

//-------------------------------------  Amplitude.
double 
Ramp::Ampl(const Time& t) const {
    return mAmpl;
}

void
Ramp::setAmp(double amp) {
    mAmpl = amp;
}
