/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: fdstreambuf						*/
/*                                                         		*/
/* Module Description: Stream buffer which uses file descriptors for IO	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 17Apr01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: pipe_exec.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FDSTREAMBUF_H
#define _LIGO_FDSTREAMBUF_H

#ifndef __CINT__
#include "PConfig.h"
#include <iosfwd>
#ifndef __GNU_STDC_OLD
#include <ios>
#include <streambuf>

namespace gdsbase {
   // file descriptor read
   int fdread (int fd, void* s, int num, int elsize);
   // file descriptor write
   int fdwrite (int fd, const void* s, int num, int elsize);
   // file descriptor seek
   int fdseek (int fd, int offset, std::ios_base::seekdir way, 
              int elsize);
}

namespace std {

/** @name Streams that work with file descriptors. 
    This header defines streams that work with normal UNIX file
    descriptors. It utilizes a stream buffer which reads and writes from
    file descriptors. The current implementation does not use buffered 
    output.
   
    @memo File descriptor streams
    @author Written April 2001 by Daniel Sigg
    @version 1.0
 ************************************************************************/

//@{


/** Basic stream buffer with file descriptors.
    
    @memo Basic file descriptors stream buffer
 ************************************************************************/
template <class charT, class traits = char_traits <charT> >
   class basic_fdbuf : public basic_streambuf <charT, traits> {
   public:
      typedef typename basic_streambuf <charT, traits>::int_type int_type;
      typedef typename basic_streambuf <charT, traits>::off_type off_type;
      typedef typename basic_streambuf <charT, traits>::pos_type pos_type;
      typedef typename basic_streambuf <charT, traits>::char_type char_type;
      typedef traits traits_type;
      
      /** Constructor.
          Create a streambuf from open file descriptors. The first
          descriptor will be used for input whereas the second
          is used for output. If one of the is not used, set it to -1.
        */
      basic_fdbuf (int fdin, int fdout);
   
   protected:
      /// Inpput file descriptor
      int		fFdIn;
      /// Inpput file descriptor
      int		fFdOut;
      /// size of input data buffer
      static const streamsize	fBufferSize = 1024;
      /// Input data buffer
      char_type		fBuffer[fBufferSize];
   
      /// Read one character
      virtual int_type underflow ();
      /// Write one character
      virtual int_type overflow (int_type c = traits::eof());
      /// Write multiple characters
      virtual streamsize xsputn (const char_type* s, streamsize num);
      /// Seek with offset
      virtual pos_type seekoff(off_type off, ios_base::seekdir way,
                        ios_base::openmode which =
                        ios_base::in | ios_base::out);
      /// Seek with position
      virtual pos_type seekpos(pos_type sp,
                        ios_base::openmode which =
                        ios_base::in | ios_base::out);
   
      /// file descriptor read
      static streamsize fdread (int fd, char_type* s, streamsize num) {
         return gdsbase::fdread (fd, (void*)s, num, sizeof (char_type)); }
      /// file descriptor write
      static streamsize fdwrite (int fd, const char_type* s, streamsize num) {
         return gdsbase::fdwrite (fd, (const void*)s, num, sizeof (char_type)); }
      /// file descriptor seek
      static pos_type fdseek (int fd, off_type off, 
                        ios_base::seekdir way) {
         return gdsbase::fdseek (fd, off, way, sizeof (char_type)); }
   };


/** Stream buffer with file descriptors (char).
    
    @memo Stream buffer with file descriptors (char)
 ************************************************************************/
   typedef basic_fdbuf<char> fd_streambuf;


/** Stream buffer with file descriptors (wchar_t).
    
    @memo Stream buffer with file descriptors (wchar_t)
 ************************************************************************/
#ifndef __CINT__
   typedef basic_fdbuf<wchar_t> fd_wstreambuf;
#endif


/** Basic input stream with file descriptors.
    
    @memo Basic input stream buffer with file descriptors
 ************************************************************************/
template <class charT, class traits = char_traits <charT> >
   class basic_fdistream : public basic_istream <charT, traits> {
   public:
      /// Create an output stream from an open file descriptors
      explicit basic_fdistream (int fd) :
      basic_istream <charT, traits>(0), fBuf (fd, -1) {
         this->init (&fBuf); this->clear(); }
   
   protected:
      /// Stream buffer
      basic_fdbuf<charT, traits> fBuf;
   };


/** Basic output stream with file descriptors.
    
    @memo Basic output stream buffer with file descriptors
 ************************************************************************/
template <class charT, class traits = char_traits <charT> >
   class basic_fdostream : public basic_ostream <charT, traits> {
   public:
      /// Create an output stream from an open file descriptors
      explicit basic_fdostream (int fd) : 
      basic_ostream <charT, traits>(0), fBuf (-1, fd) {
         this->init (&fBuf); this->clear(); }
   
   protected:
      /// Stream buffer
      basic_fdbuf<charT, traits> fBuf;
   };


/** Basic IO stream with file descriptors.
    
    @memo Basic IO stream buffer with file descriptors
 ************************************************************************/
template <class charT, class traits = char_traits <charT> >
   class basic_fdiostream : public basic_iostream <charT, traits> {
   public:
      /// Create an output stream from an open file descriptors
      explicit basic_fdiostream (int fd) :
      basic_iostream <charT, traits>(0), fBuf (fd, fd) {
         this->init (&fBuf); this->clear(); }
      /// Create an output stream from an open file descriptors
      basic_fdiostream (int fdin, int fdout) : fBuf (fdin, fdout) {
         this->init (&fBuf); this->clear(); }
   
   protected:
      /// Stream buffer
      basic_fdbuf<charT, traits> fBuf;
   };



/** Input stream with file descriptors (char).
    
    @memo Input stream buffer with file descriptors (char)
 ************************************************************************/
   typedef basic_fdistream<char> fd_istream;

/** Input stream with file descriptors (wchar_t).
    
    @memo Input stream buffer with file descriptors (wchar_t)
 ************************************************************************/
#ifndef __CINT__
   typedef basic_fdistream<wchar_t> fd_wistream;
#endif

/** Output stream with file descriptors (char).
    
    @memo Output stream buffer with file descriptors (char)
 ************************************************************************/
   typedef basic_fdostream<char> fd_ostream;

/** Output stream with file descriptors (wchar_t).
    
    @memo Output stream buffer with file descriptors (wchar_t)
 ************************************************************************/
#ifndef __CINT__
   typedef basic_fdostream<wchar_t> fd_wostream;
#endif

/** IO stream with file descriptors (char).
    
    @memo IO stream buffer with file descriptors (char)
 ************************************************************************/
   typedef basic_fdiostream<char> fd_iostream;

/** IO stream with file descriptors (wchar_t).
    
    @memo IO stream buffer with file descriptors (wchar_t)
 ************************************************************************/
#ifndef __CINT__
   typedef basic_fdiostream<wchar_t> fd_wiostream;
#endif

//@}

//#include <iostream>
//______________________________________________________________________________
template <class charT, class traits>
   basic_fdbuf<charT,traits>::basic_fdbuf (int fdin, int fdout) 
   : fFdIn (fdin), fFdOut (fdout) 
   {
      this->setg (fBuffer+4, fBuffer+4, fBuffer+4);
   }

//______________________________________________________________________________
template <class charT, class traits >
   typename basic_fdbuf<charT,traits>::int_type 
   basic_fdbuf<charT,traits>::underflow () 
   {
      if (fFdIn == -1) {
         return traits_type::eof(); 
      }
      //cerr << "Underflow" << endl;
   
      // read before end of buffer?
      if (this->gptr() < this->egptr()) {
         return *this->gptr();
      }
      // process putback area
      streamsize numPutback;
      numPutback = this->gptr() - this->eback();
      if (numPutback > 4) numPutback = 4;
      // copy putback area upfront
      traits_type::copy (fBuffer+(4-numPutback), this->gptr()-numPutback, numPutback);
      // get new characters
      streamsize num = fdread (fFdIn, fBuffer+4, fBufferSize-4);
      //cerr << "read char " << num << endl;
      if (num <= 0) {
         return traits_type::eof();
      }
      // reset buffer pointers
      this->setg (fBuffer+(4-numPutback), fBuffer+4, fBuffer+4+num);
      return *this->gptr(); 
   }

//______________________________________________________________________________
template <class charT, class traits >
   typename basic_fdbuf<charT,traits>::int_type 
   basic_fdbuf<charT,traits>::overflow (int_type c) 
   {
      if (fFdOut == -1) {
         return traits_type::eof(); 
      }
      //cerr << "Overflow = " << (int) c << endl;
      if (c != traits_type::eof()) {
         char z = c;
         if (fdwrite (fFdOut, &z, 1) != 1) {
            return traits_type::eof();
         }
      }
      //cerr << "WRITE = " << (int)c << endl;
      return c;
   }

//______________________________________________________________________________
template <class charT, class traits >
   streamsize 
   basic_fdbuf<charT,traits>::xsputn (const char_type* s, streamsize num) 
   {
      if (fFdOut == -1) {
         return 0; 
      }
      // cerr << "WRITE " << num << endl;
      // ::write (2, s, num);
      // cerr << endl;
      return fdwrite (fFdOut, s, num); 
   }

//______________________________________________________________________________
template <class charT, class traits>
   typename basic_fdbuf<charT,traits>::pos_type 
   basic_fdbuf<charT,traits>::seekoff (off_type off, 
                     ios_base::seekdir way,
                     ios_base::openmode which)
   {
      pos_type pos = pos_type(off_type(-1));
      if (which & ios_base::in) {
         pos = fdseek (fFdIn, off, way);
         if (pos == pos_type(off_type(-1))) {
            return pos_type(off_type(-1));
         }
      }
      if (which & ios_base::out) {
         pos = fdseek (fFdOut, off, way);
      }
      return pos;
   }

//______________________________________________________________________________
template <class charT, class traits>
   typename basic_fdbuf<charT,traits>::pos_type 
   basic_fdbuf<charT,traits>::seekpos (pos_type sp,
                     ios_base::openmode which)
   {
      return seekoff (sp, ios_base::beg, which);
   }

}

#else // __GNU_STDC_OLD

#include <fstream>

   typedef filebuf fd_streambuf;
   typedef ifstream fd_istream;
   typedef ofstream fd_ostream;
   typedef fstream fd_iostream;

#endif // __GNU_STDC_OLD

#endif // __CINT__

#endif // _LIGO_FDSTREAMBUF_H

