/******************************************************************************
 *
 * Project:  EDIGEO Translator
 * Purpose:  Implements OGREDIGEOLayer class.
 * Author:   Even Rouault, <even dot rouault at spatialys.com>
 *
 ******************************************************************************
 * Copyright (c) 2011-2013, Even Rouault <even dot rouault at spatialys.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 ****************************************************************************/

#include "ogr_edigeo.h"
#include "cpl_conv.h"
#include "cpl_string.h"
#include "ogr_p.h"
#include "ogr_srs_api.h"

CPL_CVSID("$Id: ogredigeolayer.cpp 355b41831cd2685c85d1aabe5b95665a2c6e99b7 2019-06-19 17:07:04 +0200 Even Rouault $")

/************************************************************************/
/*                          OGREDIGEOLayer()                            */
/************************************************************************/

OGREDIGEOLayer::OGREDIGEOLayer( OGREDIGEODataSource* poDSIn,
                                const char* pszName, OGRwkbGeometryType eType,
                                OGRSpatialReference* poSRSIn ) :
    poDS(poDSIn),
    poFeatureDefn(new OGRFeatureDefn( pszName )),
    poSRS(poSRSIn),
    nNextFID(0)
{
    if( poSRS )
        poSRS->Reference();

    poFeatureDefn->Reference();
    poFeatureDefn->SetGeomType( eType );
    if( poFeatureDefn->GetGeomFieldCount() != 0 )
        poFeatureDefn->GetGeomFieldDefn(0)->SetSpatialRef(poSRS);
    SetDescription( poFeatureDefn->GetName() );
}

/************************************************************************/
/*                          ~OGREDIGEOLayer()                           */
/************************************************************************/

OGREDIGEOLayer::~OGREDIGEOLayer()

{
    for(int i=0;i<(int)aosFeatures.size();i++)
        delete aosFeatures[i];

    poFeatureDefn->Release();

    if (poSRS)
        poSRS->Release();
}

/************************************************************************/
/*                            ResetReading()                            */
/************************************************************************/

void OGREDIGEOLayer::ResetReading()

{
    nNextFID = 0;
}

/************************************************************************/
/*                           GetNextFeature()                           */
/************************************************************************/

OGRFeature *OGREDIGEOLayer::GetNextFeature()
{

    while( true )
    {
        OGRFeature *poFeature = GetNextRawFeature();
        if (poFeature == nullptr)
            return nullptr;

        if((m_poFilterGeom == nullptr
            || FilterGeometry( poFeature->GetGeometryRef() ) )
        && (m_poAttrQuery == nullptr
            || m_poAttrQuery->Evaluate( poFeature )) )
        {
            return poFeature;
        }

        delete poFeature;
    }
}

/************************************************************************/
/*                         GetNextRawFeature()                          */
/************************************************************************/

OGRFeature *OGREDIGEOLayer::GetNextRawFeature()
{
    if (nNextFID < (int)aosFeatures.size())
    {
        OGRFeature* poFeature = aosFeatures[nNextFID]->Clone();
        nNextFID++;
        return poFeature;
    }
    else
        return nullptr;
}

/************************************************************************/
/*                            GetFeature()                              */
/************************************************************************/

OGRFeature * OGREDIGEOLayer::GetFeature(GIntBig nFID)
{
    if (nFID >= 0 && nFID < (int)aosFeatures.size())
        return aosFeatures[(int)nFID]->Clone();
    else
        return nullptr;
}

/************************************************************************/
/*                           TestCapability()                           */
/************************************************************************/

int OGREDIGEOLayer::TestCapability( const char * pszCap )

{
    if (EQUAL(pszCap, OLCFastFeatureCount))
        return m_poFilterGeom == nullptr && m_poAttrQuery == nullptr;

    else if (EQUAL(pszCap, OLCRandomRead))
        return TRUE;

    else if (EQUAL(pszCap, OLCStringsAsUTF8))
        return poDS->HasUTF8ContentOnly();

    return FALSE;
}

/************************************************************************/
/*                              GetExtent()                             */
/************************************************************************/

OGRErr OGREDIGEOLayer::GetExtent(OGREnvelope *psExtent, int bForce)
{
    /*if (poDS->bExtentValid)
    {
        psExtent->MinX = poDS->dfMinX;
        psExtent->MinY = poDS->dfMinY;
        psExtent->MaxX = poDS->dfMaxX;
        psExtent->MaxY = poDS->dfMaxY;
        return OGRERR_NONE;
    }*/

    return OGRLayer::GetExtent(psExtent, bForce);
}

/************************************************************************/
/*                          GetFeatureCount()                           */
/************************************************************************/

GIntBig OGREDIGEOLayer::GetFeatureCount( int bForce )
{
    if (m_poFilterGeom != nullptr || m_poAttrQuery != nullptr)
        return OGRLayer::GetFeatureCount(bForce);

    return (int)aosFeatures.size();
}

/************************************************************************/
/*                             AddFeature()                             */
/************************************************************************/

void OGREDIGEOLayer::AddFeature(OGRFeature* poFeature)
{
    poFeature->SetFID(aosFeatures.size());
    aosFeatures.push_back(poFeature);
}

/************************************************************************/
/*                         GetAttributeIndex()                          */
/************************************************************************/

int OGREDIGEOLayer::GetAttributeIndex(const CPLString& osRID)
{
    std::map<CPLString,int>::iterator itAttrIndex =
                                            mapAttributeToIndex.find(osRID);
    if (itAttrIndex != mapAttributeToIndex.end())
        return itAttrIndex->second;
    else
        return -1;
}

/************************************************************************/
/*                           AddFieldDefn()                             */
/************************************************************************/

void OGREDIGEOLayer::AddFieldDefn(const CPLString& osName,
                                  OGRFieldType eType,
                                  const CPLString& osRID)
{
    if (!osRID.empty())
        mapAttributeToIndex[osRID] = poFeatureDefn->GetFieldCount();

    OGRFieldDefn oFieldDefn(osName, eType);
    poFeatureDefn->AddFieldDefn(&oFieldDefn);
}
