------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUN-TIME COMPONENTS                         --
--                                                                          --
--                       ADA.STRINGS.TEXT_OUTPUT.UTILS                      --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--            Copyright (C) 2020-2021, Free Software Foundation, Inc.       --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

package Ada.Strings.Text_Output.Utils with Pure is

   --  This package provides utility functions on Sink'Class. These are
   --  intended for use by Put_Image attributes, both the default versions
   --  generated by the compiler, and user-defined ones.

   procedure Full (S : in out Sink'Class) with Inline;
   --  Must be called when the current chunk is full. Dispatches to
   --  Full_Method.

   procedure Flush (S : in out Sink'Class) with Inline;
   --  Dispatches to Flush_Method

   --  Full_Method and Flush_Method should be called only via Full and Flush

   procedure Put_Character (S : in out Sink'Class; Item : Character);
   procedure Put_Wide_Character (S : in out Sink'Class; Item : Wide_Character);
   procedure Put_Wide_Wide_Character
     (S : in out Sink'Class; Item : Wide_Wide_Character);
   procedure Put_String (S : in out Sink'Class; Item : String);
   procedure Put_Wide_String (S : in out Sink'Class; Item : Wide_String);
   procedure Put_Wide_Wide_String
     (S : in out Sink'Class; Item : Wide_Wide_String);
   --  Encode characters or strings as UTF-8, and send them to S.

   subtype Character_7 is
     Character range Character'Val (0) .. Character'Val (2**7 - 1);
   --  7-bit character. These are the same in both Latin-1 and UTF-8.

   procedure Put_7bit (S : in out Sink'Class; Item : Character_7)
     with Inline, Pre => Item /= NL;
   procedure Put_7bit_NL (S : in out Sink'Class; Item : Character_7)
     with Inline;
   --  Put a 7-bit character, and adjust the Column. For Put_7bit_NL, Item can
   --  be NL.

   procedure Put_UTF_8 (S : in out Sink'Class; Item : UTF_8) with Inline;
   procedure Put_UTF_8_Lines (S : in out Sink'Class; Item : UTF_8_Lines);
   --  Send data that is already UTF-8 encoded (including 7-bit ASCII) to
   --  S. These are more efficient than Put_String.

   procedure New_Line (S : in out Sink'Class) with
     Inline, Post => Column (S) = 1;
   --  Puts the new-line character.

   function Column (S : Sink'Class) return Positive with Inline;
   --  Current output column. The Column is initially 1, and is incremented for
   --  each 7-bit character output, except for the new-line character, which
   --  sets Column back to 1. The next character to be output will go in this
   --  column.

   procedure Tab_To_Column (S : in out Sink'Class; Column : Positive);
   --  Put spaces until we're at or past Column.

   procedure Set_Indentation (S : in out Sink'Class; Amount : Natural)
     with Inline;
   function Indentation (S : Sink'Class) return Natural with Inline;
   --  Indentation is initially 0. Set_Indentation sets it, and Indentation
   --  returns it. This number of space characters are put at the start of
   --  each nonempty line.

   subtype Optional_Indentation is Integer range -1 .. Natural'Last;
   Default : constant Optional_Indentation := -1;

   procedure Indent
     (S : in out Sink'Class; Amount : Optional_Indentation := Default)
      with Inline;
   procedure Outdent
     (S : in out Sink'Class; Amount : Optional_Indentation := Default)
      with Inline;
   --  Increase/decrease Indentation by Amount. If Amount = Default, the amount
   --  specified by the Indent_Amount parameter of the sink creation function
   --  is used. The sink creation functions are New_Buffer, Create_File, and
   --  Create_New_File.

end Ada.Strings.Text_Output.Utils;
