#!/bin/sh
# Copyright (c) 2007 David Soria Parra <dsp at php dot net>
#
# Licensed under the terms of the MIT License
# See /usr/share/doc/gcutils/copyright
# or http://www.opensource.org/licenses/mit-license.php

. $LIBDIR$/gc-sh-setup

usage ()
{
    echo "Usage: "`basename $0`" [OPTIONS]"
    echo "Options are:"
    echo "    -h        show help"
    echo "    -v        verbosity"
    echo "    -m        use merge strategy instead of rebasing"
    echo "    -cn       do not update the cvs repository"
    echo "    -V        show version information"
    echo
    exit 127
}

check_git

cvsupdate=t
verbose=
quiet="-q"
do_rebase="-i"
while test $# != 0
do
    case "$1" in
        -m)
            do_rebase=
            ;;
        -v)
            verbose="-v"
            quiet=""
            ;;
        -cn)
            cvsupdate=""
            ;;
        -h)
            usage
            ;;
        -V)
            version
            exit
            ;;
        -*)
            echo >&2 "Parameter $1 is not known."
            usage
            ;;
    esac
    shift
done


locate_git_repo

if test ! ".cvs" -o ! -d ".cvs/CVS/"
then
    echo >&2 "CVS working directory not found."
    die "Make sure you imported your CVS repository using gc-import."
fi

branch=`git branch | grep "^* " | cut -c 3-`

CVSROOT=`cat .cvs/CVS/Root`
MODULE=`cat .cvs/CVS/Repository`

echo "Repository: $CVSROOT"
echo "Module:     $MODULE"
echo "GIT Branch: $branch"
echo

# first stash away our local changes
echo "Stashing local changes"
test -n "$verbose" && git stash save "stashed by gcutils"
test -z "$verbose" && git stash save "stashed by gcutils" 2>/dev/null 1>/dev/null
echo

echo "Updating"

remote=`git config gc-utils.remote`

if test -n "$remote"
then
    refspec="refs/remotes/$remote/master"
    remote="-r $remote"
else
    refspec="refs/heads/origin"
    remote=""
fi

# if -m is not specified we do a rebase instead of a merge
# therefore git cvsimport will be called -i ($do_rebase)
git cvsimport $remote $do_rebase $verbose -a -m -k -d $CVSROOT "$MODULE"

if test -n "$do_rebase"
then
    # we actually did not yet merged back into master but we try to rebase
    git rebase `git show-ref -s $refspec` master
    if test $? -ne 0
    then
        git rebase --abort
        echo "Rebasing failed."
        echo "Please use rebase to rebase the origin branch or use merge to"
        echo "do a merge instead. Notice that merge might result in duplicated commits"
        echo
        echo "gc utils aborted the rebase"

        git checkout $branch 1>/dev/null 2>/dev/null

        # reapply our stashed dirty changes
        echo
        echo "Reapplying local changes"

        test -n "$verbose" && git stash pop
        test -z "$verbose" && git stash pop 2>/dev/null 1>/dev/null

        exit 127
    fi
fi

git checkout $branch 1>/dev/null 2>/dev/null

# reapply our stashed dirty changes
echo
echo "Reapplying local changes"
test -n "$verbose" && git stash pop
test -z "$verbose" && git stash pop 2>/dev/null 1>/dev/null

if test -n "$cvsupdate"
then
    echo "Running cvs update"
    cd .cvs
    cvs $quiet update -d -P
fi
