/*
 * pcreg_ex.c - Perl compatible Regular Expression functions
 * 
 * include LICENSE
 */
#include <stdio.h>
#include <string.h>

#include <msglog.h>
#include <strmem.h>
#include <strcatdup.h>
#include <pcreg_ex.h>


/*
 *** \brief Allocates memory for a new Pcregex object.
 *   pattern : A  string containing the regular expression to be compiled.
 *   options : flags for compile regex -see pcre.h
 *   nmatch  : number of expected match, if pattern extraction.
 *             0 whole match
 *             1 first sub pat, ...
 */

Pcregex *pcregex_new( char *pattern, int options, int nmatch )
{
   Pcregex *re;

   re =  app_new0(Pcregex, 1);
   pcregex_construct( re, pattern, options, nmatch );
   app_class_overload_destroy( (AppClass *) re, pcregex_destroy );
   return re;
}

/** \brief Constructor for the Pcregex object. */

void pcregex_construct( Pcregex *re, char *pattern, int options, int nmatch )
{
   app_class_construct( (AppClass *) re );

   re->pattern = app_strdup(pattern);
   re->options = options;
   re->exflags = nmatch;
   nmatch &= PCREG_EX_MASK;
   re->nmatch = nmatch;
   if ( nmatch ) {
      /* 0 whole matched pat, + nmatch sub patterns */ 
      re->ovector = app_new0( int, 3 * nmatch );  /* should be a multiple of 3 */
      re->matchStr = array_strPtr_new( nmatch );
   }
}

/** \brief Destructor for the Pcregex object. */

void pcregex_destroy(void *re)
{
   Pcregex *this = (Pcregex *) re;

   if (re == NULL) {
      return;
   }
   app_free(this->ovector);
   app_free(this->pattern);
   if ( this->creg ) {
      pcre_free( this->creg );
   }
   array_strPtr_destroy( this->matchStr);

   app_class_destroy( re );
}


pcre *pcregex_compile(Pcregex *re)
{
   const char *errptr;             /* for error message */
   int erroffset;            /* for error offset */
   pcre *creg;

   creg = pcre_compile( re->pattern, re->options, &errptr, &erroffset, re->tableptr );
   if (creg == NULL) {
      msg_fatal( "Compilation failed at offset %d (in regexp '%s') '%s':\n",
		 erroffset, re->pattern, errptr );
   }
   re->creg = creg;
   return creg;
}

/*
 *  return < 0, if no match found
 *  return 0 if match found and no extraction requested.
 *  return 0 if match found and not enough room in ovector.
 *  return > 0 if match found and enough room in ovector.
 */
int pcregex_exec(Pcregex *re, char *subject, int length )
{
   int ret;
   
   if ( re->creg == NULL ){
      pcregex_compile( re );
   }
   if ( length == 0 ){
      length = (int) strlen(subject);
   }
   ret = pcre_exec( re->creg, NULL, subject, length, 0, 0, re->ovector, 3 * re->nmatch);
   if ( ret < 0 ) {
      msg_dbg("pcre_exec returned %d code", ret); 
   } else if ( re->exflags & PCREG_NO_EXTRACT ) {
      ; /* return */
   } else if ( re->nmatch ) {
       /* match found */
      int i;
      int nmatch =  re->nmatch < ret ? re->nmatch : ret;
      array_tbl_delete_all( (Array *) re->matchStr );
      for ( i = 0 ; i < nmatch ; i++ ) {
         array_strPtr_nadd( re->matchStr, subject + re->ovector[2 * i],
                            re->ovector[2 * i + 1] - re->ovector[2 * i] );
      }
   }
   return ret;
}

/*
 * substitute in subject
 */
char *pcregex_subst(Pcregex *re, char *subject, int len, char *subst, int i )
{
   if ( pcregex_exec(re, subject, len ) >= 0 ){
      if ( ! subst ) {
         subst = "";
      }
      char *s = subject + re->ovector[2 * i];
      int len = re->ovector[2 * i + 1] - re->ovector[2 * i];

      *s = 0;
      char *new_subject = app_strcatdup( subject, subst, s + len, NULL );
      app_free(subject);
      return new_subject;
   }
   return subject;
}

/*
 * Execute regex for string from dlist_lookup.
 * Look if regex match the string
 * return 0 if regex match
 */

int pcregex_str_cmp( AppClass *d1, AppClass *d2 )
{
   Pcregex *list_re = (Pcregex *) d1;
   char *str =  (char *) d2;
   
   if ( pcregex_exec(list_re, str, 0 ) < 0 ) {
      return 1;
   }
   return 0;
}

