/*
 * filerec.c - object for files and vars
 * 
 * include LICENSE
 */
#include <stdio.h>
#include <string.h>

#include <strcatdup.h>
#include <duprintf.h>
#include <variable.h>
#include <descrec.h>
#include <descelem.h>
#include <filerec.h>
#include <mkcf.h>
#include <array.h>
#include <pcreg_ex.h>


/*
 *** \brief Allocates memory for a new FileRecord object.
 */

FileRecord *filerec_new( char *rcfile, char *proto, char *svarn, char *relvar )
{
   FileRecord *frec;

   frec =  app_new0(FileRecord, 1);
   filerec_construct( frec, rcfile, proto, svarn, relvar );
   app_class_overload_destroy( (AppClass *) frec, filerec_destroy );
   return frec;
}

/** \brief Constructor for the FileRecord object. */

void filerec_construct( FileRecord *frec, char *rcfile, char *proto,
			char *svarn, char *relvar )
{
   app_class_construct( (AppClass *) frec );

   frec->rcfile = app_strdup(rcfile);
   frec->proto = app_strdup(proto);
   filerec_dup_svarn( frec, svarn );
   frec->relvar = app_strdup(relvar);
   if ( relvar ){
      char *name = app_strcatdup( relvar, "Desc", NULL);
      filerec_dup_descname(frec, name);  /* default desc name */
      app_free(name);
   } else {
      filerec_dup_descname(frec, "confDesc");  /* userprefs.h */
   }
   filerec_dup_funcname(frec, proto);  /* default func name */
   filerec_lower(frec->funcname);  
}

/** \brief Destructor for the FileRecord object. */

void filerec_destroy(void *frec)
{
   FileRecord *this = (FileRecord *) frec;

   if (frec == NULL) {
      return;
   }
   app_free(this->prefix);
   app_free(this->rcfile);
   app_free(this->proto);
   app_free(this->svarn);
   app_free(this->relvar);
   app_free(this->funcname);
   app_free(this->descname);
   dlist_delete_all( this->vars );
   dlist_delete_all( this->includes );

   app_class_destroy( frec );
}

void filerec_lower(char *name)
{
   char *p;
   
   for (p = name; *p ; p++ ) {
      *p |= 0x20;
   }
}

void filerec_dup_descname(FileRecord *frec, char *descname)
{
   app_dup_str( &frec->descname, descname);
}

void filerec_dup_funcname(FileRecord *frec, char *name)
{
   app_dup_str( &frec->funcname, name);
}

char *filerec_do_short(FileRecord *frec)
{
   char *temp = app_new0( char, 4);
   app_strncpy(temp, frec->proto, 2);
   filerec_lower(temp);
   return temp;
}

void filerec_dup_svarn( FileRecord *frec, char *svarn )
{
   if ( ! svarn ) {
      frec->svarn = filerec_do_short(frec);
   } else {
      frec->svarn = app_strdup(svarn);
   }
}

void filerec_dup_prefix( FileRecord *frec, char *prefix )
{
   if ( ! prefix ) {
      frec->prefix = filerec_do_short(frec);
   } else {
      frec->prefix = app_strdup(prefix);
   }
}

      
char *filerec_get_prefix(FileRecord *frec)
{
   return frec->prefix;
}

void filerec_add_var( FileRecord *frec, char *varname, char *vartype,
		      char *varcmt, int tini, int lineno )
{
   int res = 1;
   char *tmptype;

   /* cleanup vartype */
   char *pattern = "([^\\s*]+)\\s*([^\\s]*)";
   Pcregex *varRe = pcregex_new( pattern, 0, 3 );
   if ( pcregex_exec(varRe, vartype, 0 ) >= 2 ){
      tmptype = app_strdup_printf("%s %s",
                                  array_strPtr_get(varRe->matchStr, 1 ),
                                  array_strPtr_get(varRe->matchStr, 2 ) );
   } else {
      tmptype = app_strdup(vartype);
   }
   Variable *var = var_new( varname, tmptype, varcmt, tini, lineno );
   app_free(tmptype);
   var_dup_varfunc( var, frec->prefix );
   
   frec->vars = dlist_add( frec->vars, (AppClass *) var,
                           var_varname_cmp, &res );

   if ( res ) {
      /* extTok already in list - var is destroyed by dlist_add */
      msg_warningl( 2, "Variable '%s' at line %d  already in list;\n"
                        "this record is discarded",
                    varname, lineno) ;
   }
}

/*
 * this function return 0 to avoid deleting the node
 */
int filerec_iter_print_vars( AppClass *data, void *user_data )
{
   FileRecord *frec = (FileRecord *) data;
   
   fprintf(stderr, "File '%s', Prot '%s', svarn '%s', Relvar '%s'\n",
	 frec->rcfile, frec->proto, frec->svarn,  frec->relvar) ;

   dlist_iterator( frec->vars, var_iter_print_var, user_data );
   return 0;
}

void filerec_print_variables_with_comment_from_desc_table(FileRecord *frec )
{
   UserData *ud = (UserData *) frec->user_data;
   
   DescRecord *desc = (DescRecord *) dlist_lookup( ud->descrecs,
			      (AppClass *) frec->proto, descrec_proto_cmp );
   if ( desc ) {
      fprintf( ud->ofd, "\n#if 0\n");
      fprintf( ud->ofd, "typedef struct _%s %s;\n", frec->proto, frec->proto );
      fprintf( ud->ofd, "struct _%s {\n", frec->proto );

      DList *node = frec->vars->next;
      while (node != frec->vars) {
	 Variable *var = (Variable *) node->data;
	 DescElem *elem = (DescElem *) dlist_lookup(desc->elems,
				(AppClass *) var->varname, descelem_name_cmp );
	 char *cmt ;
	 if ( elem && elem->desccmt ){ 
	    cmt = app_strdup_printf("/* %s */", elem->desccmt );
	 } else if ( var->varcmt ) {
	    cmt = app_strdup_printf("/* %s */", var->varcmt );
	 } else {
	    cmt = app_strdup("");
	 }
	 fprintf( ud->ofd, "   %s%s;  %s\n", var->vartype, var->varname, cmt );
	 app_free(cmt);

	 node = node->next;
      }
      
      fprintf( ud->ofd, "};\n");
      fprintf( ud->ofd, "#endif\n\n");
   } else {
      msg_error( "DescRecord not found for '%s'", frec->proto);
   }
}

int filerec_iter_print_desc_from_vars( AppClass *data, void *user_data )
{
   Variable *var = (Variable *) data;
   FileRecord *frec = (FileRecord *) user_data;
   UserData *ud = (UserData *) frec->user_data;

   if ( var->index ){
      return 0;     /* already visited */
   }
   char *extTok = app_strdup_printf("%s_%s", frec->prefix, var->varname);
      
   fprintf( ud->ofd, "{ \"%s\", %s, %s,\n     \"%s\"       },\n",
	    extTok, var->varfunc, var->desctype, var->varcmt);

   app_free(extTok);
   return 0;
}

int filerec_iter_print_desc_from_desc( AppClass *data, void *user_data )
{
   DescElem *elem = (DescElem *) data;
   FileRecord *frec = (FileRecord *) user_data;
   UserData *ud = (UserData *) frec->user_data;
   char *func;
   char *type;
   char *cmt;
   Variable *var;

   func = elem->varfunc;
   type = elem->desctype;
   cmt = elem->desccmt;
   if ( app_strcmp(elem->desctype, "TCMT" ) == 0 ){
      func = "0";
   } else {
      var = (Variable *) dlist_lookup(frec->vars,
			      (AppClass *) elem->varname, var_name_cmp );
      if ( ! var ) {
	 return 0;    /* this variable is suppresed */
      }
      var->index = 1;
      if ( app_strcmp(elem->desctype, var->desctype ) ){
	 if ( app_strcmp(elem->desctype, "TBOOL" ) ){
	    msg_warning("var type %s ne desc type %s for Variable %s\n"
			"     changing type to %s for Variable %s",
			var->desctype, elem->desctype, var->varname,
			var->desctype, var->varname);
	    type = var->desctype;
	 }
      }
      if ( ud->fcflag ){
	 /* force comment from var if any */
	 if ( var->varcmt && *var->varcmt ) {
            cmt = var->varcmt;
	 }
      } else {
	 if ( ! elem->desccmt || ! *elem->desccmt ) {
	    cmt = var->varcmt;
	 }
      }
   }
      
   fprintf( ud->ofd, "{ \"%s\", %s, %s,\n     \"%s\"       },\n",
	    elem->extTok, func, type, cmt);

   return 0;
}


void filerec_print_conf_desc_table(FileRecord *frec )
{
   UserData *ud = (UserData *) frec->user_data;
   
   DescRecord *desc = (DescRecord *) dlist_lookup( ud->descrecs,
			      (AppClass *) frec->proto, descrec_proto_cmp );

   fprintf( ud->ofd, "\n%s %s[] = {  /* %s - by mkcf */\n",
	       ud->defdesc, frec->descname, frec->proto );
   if ( desc ) {
      /* use stuff from old conf desc table */
      dlist_iterator( desc->elems, filerec_iter_print_desc_from_desc, frec );
   }
   /* unvisited or all variables */
   dlist_iterator( frec->vars, filerec_iter_print_desc_from_vars, frec );
   fprintf( ud->ofd, "    {  0, 0, 0, 0, },\n};\n\n"); /* end table */
}

/*
 * print access function for all the variables
 */

void filerec_print_access_func(FileRecord *frec )
{
   UserData *ud = (UserData *) frec->user_data;
   char *curptr;
   
   if ( app_strcmp( frec->proto, "UserPrefs") == 0 ){
      curptr = app_strdup("up");
   } else {
      curptr = app_strdup_printf("((%s *) up->curptr)", frec->proto);
   }
   
   DList *node = frec->vars->next;
   while (node != frec->vars) {
      int flag = 0;
      FileRecord *varfrec;
      Variable *var = (Variable *) node->data;

      if ( var->tini ) {
	 varfrec = (FileRecord *) dlist_lookup(ud->filerecs,
		       (AppClass *) var->varname, filerec_relvar_cmp );
	 if ( ! varfrec ) {
	    msg_fatal("FileRecord not found for relvar '%s'", var->varname);
	 }
	 flag = 2;
	 if ( array_strPtr_lookup(ud->initfuncs, var->varfunc ) ){
	    flag = 1; /* func alreday printed, do not print */
	 }
      }
      if ( flag == 0) {
	 fprintf( ud->ofd, "\nvoid *%s( UserPrefs *up )\n{\n", var->varfunc);
	 fprintf( ud->ofd, "   return &%s->%s;\n}\n", curptr, var->varname );
      }
      if ( flag == 2) {
	 fprintf( ud->ofd, "\n#if 1\n");
         fprintf( ud->ofd, "void *%s( %s *%s, int write )\n{\n", 
		  var->varfunc, frec->proto, frec->svarn );
         fprintf( ud->ofd, "   %s *%s;\n\n", 
		  varfrec->proto, varfrec->svarn);
         fprintf( ud->ofd, "   %s = %s_new(0);\n", varfrec->svarn, varfrec->funcname );
         fprintf( ud->ofd, "   %s->%s = dlist_add( %s->%s, (AppClass *) %s,  NULL, NULL );\n",
		  frec->svarn, var->varname, frec->svarn, var->varname, varfrec->svarn);
         fprintf( ud->ofd, "   %s->curptr = (AppClass *) %s;\n",
		  frec->svarn, varfrec->svarn );
         fprintf( ud->ofd, "   return %s;\n", varfrec->descname );
         fprintf( ud->ofd, "}\n");
         fprintf( ud->ofd, "#endif\n");
      }
    
      node = node->next;
   }
   app_free(curptr);
}

int filerec_iter_print_content( AppClass *data, void *user_data )
{
   FileRecord *frec = (FileRecord *) data;
   UserData *ud = (UserData *) user_data;
   frec->user_data = user_data;
   
   if ( ud->cflag ) {
      filerec_print_variables_with_comment_from_desc_table(frec );
   } 
   /* print functions to access variables */
   filerec_print_access_func( frec );
   /*  print the ConfigDescTable */
   filerec_print_conf_desc_table(frec );
   return 0;
}

int filerec_iter_print_prototypes( AppClass *data, void *user_data )
{
   FileRecord *frec = (FileRecord *) data;
   UserData *ud = (UserData *) user_data;

   /* skip UserPrefs */
   if ( app_strcmp(frec->proto, "UserPrefs") == 0 ) {
       return 0;
   } 
   fprintf( ud->ofd, "void *up_init_%s( UserPrefs *up, int write );\n",
	    frec->relvar);

   /* print declaration for desc table */
   fprintf( ud->ofd, "extern %s %s[];\n", ud->defdesc,frec->descname );
   
   return 0;
}

/*
 * used in dlist_lookup
 *   confDesc    -> UserPrefs
 *   serviceDesc -> Service 
 */
int filerec_proto_cmp(AppClass *d1, AppClass *d2 )
{
   FileRecord *frec = (FileRecord *) d1 ;
   char *name = (char *) d2 ;
   if ( app_strcmp(frec->proto, "UserPrefs") == 0 &&
	app_strcmp(name, "confDesc") == 0 ){ 
      return 0;
   }
   return app_strcmp(name, frec->proto);
}

/*
 * used in dlist_lookup
 */
int filerec_relvar_cmp(AppClass *d1, AppClass *d2 )
{
   FileRecord *frec = (FileRecord *) d1 ;
   char *name = (char *) d2 ;
   return app_strcmp( frec->relvar, name );
}
