/*
 * descelem.c - hold a conf desc table entry
 * 
 * include LICENSE
 */
#include <stdio.h>
#include <string.h>

#include <descelem.h>
#include <stutil.h>
#include <strmem.h>


/*
 *** \brief Allocates memory for a new DescElem object.
 */

DescElem *descelem_new( char *extTok, char *varfunc, char *desctype,
			char *desccmt, int lineno)
{
   DescElem *elem;

   elem =  app_new0(DescElem, 1);
   descelem_construct( elem, extTok, varfunc, desctype, desccmt, lineno );
   app_class_overload_destroy( (AppClass *) elem, descelem_destroy );
   return elem;
}

/** \brief Constructor for the DescElem object. */

void descelem_construct( DescElem *elem, char *extTok, char *varfunc,
			 char *desctype, char *desccmt, int lineno )
{
   app_class_construct( (AppClass *) elem );
   
   elem->lineno = lineno;
   elem->extTok = app_strdup(extTok);
   elem->varfunc = app_strdup(varfunc);
   descelem_set_desctype(elem, desctype);
   desccmt = stu_rm_blank(desccmt);
   elem->desccmt = app_strdup(desccmt);
}

/** \brief Destructor for the DescElem object. */

void descelem_destroy(void *elem)
{
   DescElem *this = (DescElem *) elem;

   if (elem == NULL) {
      return;
   }
   app_free(this->extTok);
   app_free(this->varfunc);
   app_free(this->desccmt);
   app_free(this->varname);
   
   app_class_destroy( elem );
}

void descelem_dup_varname(DescElem *elem , char *varname)
{
   elem->varname = app_strdup(varname);
}

/*
 * this function return 0 to avoid deleting the node
 */
int descelem_iter_print_elem( AppClass *data, void *user_data )
{
   DescElem *elem = (DescElem *) data;
   
   fprintf(stderr, "    \"%s\", %s, %s,\n    /* %s */\n",
         elem->extTok, elem->varfunc, elem->desctype, elem->desccmt ) ;
   return 0;
}
/*
 * used in dlist_lookup
 */
int descelem_name_cmp(AppClass *d1, AppClass *d2 )
{
   DescElem *elem = (DescElem *) d1 ;
   char *name = (char *) d2 ;
   return app_strcmp( elem->varname, name );
}

void descelem_set_desctype(DescElem *elem , char *desctype)
{
   elem->desctype = descelem_convert_type(desctype); /* not malloced */
   if ( ! elem->desctype ) {
      msg_warning("Unknown type '%s' in desc table at line %d",
	       desctype, elem->lineno ); 
      elem->desctype = "TCMT";
   }
}

typedef struct _NameStrVal NameStrVal;
struct _NameStrVal {
   char *name;
   char *strval;
};

NameStrVal textype[] = {
   {  "TDECI",   "TDECI"       },
   {  "THEXA",   "THEXA"       },
   {  "TPTR",    "TPTR"        },
   {  "TBOOL",   "TBOOL"       },
   {  "TLIST",   "TLIST"       },
   {  "TARY",    "TARY"        },
   {  "TARYSTR", "TARYSTR"     },
   {  "TINI",    "TINI"        },
   {  "TCMT",    "TCMT"        },
   {  "TDECIMAL",   "TDECI"    },
   {  "TSTRING",    "TPTR"     },
   {  "TCHAINE",    "TPTR"     },
   {  "TPOINTER",   "TPTR"     },
   {  "TLISTE",     "TLIST"    },
   {  "TCOMMENT",    "TCMT"    },
};

char *descelem_convert_type( char *text)
{
   int i ;
   
   for (i = 0 ; i < sizeof(textype) / sizeof(textype[0])  ; i++ ) {
      if ( app_strcmp(text, textype[i].name) == 0 ){
         return textype[i].strval;
      }
   }
   return NULL;
}

/*
 * used in dlist_add
 */
int descelem_extTok_cmp(AppClass *d1, AppClass *d2 )
{
   DescElem *elem    = (DescElem *) d1 ;
   DescElem *newelem = (DescElem *) d2 ;
   return app_strcmp( elem->extTok, newelem->extTok );
}

