/*
 * gaw - Gtk analog waveform viewer
 *   GTK interface to Gaw
 * 
 * include LICENSE
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <libgen.h>
#include <stdarg.h>
#include <locale.h>

#include <gtk/gtk.h>

#include <strcatdup.h>
#include <fileutil.h>
#include <gaw.h>
 
#ifdef TRACE_MEM
#include <tracemem.h>
#endif
        
int prog_debug;
MsgLogData *msglogp; /* global pointeur to MsgLogData in main */

#undef GTK_STOCK_DELETE
#define GTK_STOCK_DELETE GTK_STOCK_CUT   /* fedora 13 */

#ifdef HAVE_GTK_OLD
/*
 * this is to make compile new versions with old gtk libs
 * and should be removed one day !!!
 */

void
hq_gdk_window_get_root_coords (GdkWindow *window,
                            gint       xx,
                            gint       yy,
                            gint      *root_x,
                            gint      *root_y)
{
   g_return_if_fail (GDK_IS_WINDOW (window));

   int x;
   int y;
   int width;
   int height;
   int depth;
   
    gdk_window_get_geometry(window, &x, &y, &width, &height, &depth );
   if (x)
      *root_x = x;
   if (y)
      *root_y = y;
   return;
}

void
hq_gtk_widget_get_allocation (GtkWidget     *widget,
                           GtkAllocation *allocation)
{
  g_return_if_fail (GTK_IS_WIDGET (widget));
  g_return_if_fail (allocation != NULL);

  *allocation = widget->allocation;
}

void 
hq_gtk_adjustment_set_page_size (GtkAdjustment *adjustment, gdouble page_size)
{
    g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));

  if (page_size != adjustment->page_size)
    g_object_set (adjustment, "page-size", page_size, NULL);
}
gdouble
hq_gtk_adjustment_get_upper (GtkAdjustment *adjustment)
{
  g_return_val_if_fail (GTK_IS_ADJUSTMENT (adjustment), 0.0);

  return adjustment->upper;
}

void
hq_gtk_widget_set_tooltip_text (GtkWidget   *widget,
                             const gchar *text)
{
  g_return_if_fail (GTK_IS_WIDGET (widget));

//  g_object_set (G_OBJECT (widget), "tooltip-text", text, NULL);
}

gboolean
hq_gtk_widget_get_visible (GtkWidget *widget)
{
  g_return_val_if_fail (GTK_IS_WIDGET (widget), FALSE);

  return (GTK_OBJECT_FLAGS (widget) & GTK_VISIBLE) != 0;
}
#endif

GtkWidget *hq_gtk_statusbar_get_label (GtkWidget *statusbar)
{
   GtkWidget *hbox = gtk_statusbar_get_message_area (GTK_STATUSBAR (statusbar));

   return GTK_WIDGET (gtk_container_get_children (GTK_CONTAINER (hbox))->data);
}


/*
 *  Analog wave globals
 */
char *progname ;
UserData  *userData  = NULL;

char *aw_panel_not_selected_msg ; /* see init in main */


/*
 * Global functions
 */
void aw_do_exit (GtkWidget *widget,  UserData *ud );


void aw_factory_settings_action (GtkAction *action, UserData *ud )
{
   gint response;
   char *message = _("You are going to reset your rc parameters to factory values\n"
		     "You will need to restart Gaw.\n"
		     "If this is what you want, click OK\n");

   response = aw_dialog_show ( AW_MSG_T_SHOW_CANCEL | MSG_T_WARNING, message );
   if (response == GTK_RESPONSE_OK) {
      up_init_defaults(ud->up);
      up_rc_rewrite(ud->up);

      /*
       * before doing restart, we need to cleanly destroy all objects !!!
       */
//      ud->restart = 1;
      aw_do_exit (NULL, ud);
   }
}

void aw_do_save_config ( UserData *ud )
{
   ap_set_user_panel_size(ud);
   up_rc_rewrite(ud->up);
}

/*
 * save configuration file
 */
void aw_save_config_action (GtkAction *action, UserData *ud )
{
   ud->up->npanels = g_list_length( ud->panelList);

   if ( ud->listFiles ){
      app_dup_str( &ud->up->lastDataFile, (char *) g_slist_nth_data ( ud->listFiles, 0 ) );
   }
   aw_do_save_config (ud );
}

/*
 * clear the default input file name and save config
 */
void aw_default_file_name_action (GtkAction *action, UserData *ud )
{
   app_dup_str( &ud->up->lastDataFile, NULL );
   prog_debug = 0;

   aw_do_save_config (ud );
}

/*
 * get data for a menu
 */
gpointer
aw_get_menu_data (GtkUIManager *ui, gchar *menuName,  gchar *actionName)
{
   msg_dbg("aw_get_menu_data menu: %s %s", menuName, actionName );
   gpointer data;
   GtkWidget *menu = gtk_ui_manager_get_widget (ui, menuName);

   if ( (data = g_object_get_data (G_OBJECT(menu), actionName))){
      msg_dbg("%s %s received 0x%lx", menuName, actionName, (unsigned long) data );
      g_object_set_data (G_OBJECT(menu), actionName, NULL);
   }
   return data;
}

static void
toggle_allow_resize_action (GtkAction *action, UserData *ud )
{
   ud->up->allowResize = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
//   msg_dbg ("Action allowResize %d", ud->up->allowResize );
}

static void
toggle_bar_style_action (GtkAction *action, UserData *ud )
{
   GtkToolbar *tb = GTK_TOOLBAR(ud->toolBar);
   GtkToolbarStyle newstyle = GTK_TOOLBAR_BOTH ;
   
   if ( gtk_toolbar_get_style(tb) == GTK_TOOLBAR_ICONS ){
      newstyle = GTK_TOOLBAR_BOTH;
   } else {
      newstyle = GTK_TOOLBAR_ICONS;
   }
   gtk_toolbar_set_style(tb, newstyle );
   ud->up->toolBarStyle = newstyle ;
}

static void
activate_action (GtkAction *action, UserData *ud )
{
   const gchar *name = gtk_action_get_name (action);

//   msg_dbg ("Action \"%s\" activated", name );
   if ( strcmp( name, "Quit") == 0 ){
      ud->restart = 0;
      aw_do_exit( NULL, ud);
   }
}

void
aw_redraw_all_action (GtkAction *action, UserData *ud )
{
   ap_all_redraw(ud);
}

static void
aw_showXlabel_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->showXLabels = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
//   msg_dbg ("Action showXlabel %d", ud->up->showXLabels );
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created */
   }
   ap_xlabel_box_show(ud);
}

static void
aw_showYlabel_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->showYLabels = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
//   msg_dbg ("Action showYlabel %d", ud->up->showYLabels );
   /*  this happen when setting default value from rc */
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created  */
   }

   g_list_foreach(ud->panelList, (GFunc) pa_ylabel_box_show, NULL );
}
static void
aw_showMoreYlabel_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->showMoreYLabels = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
//   msg_dbg ("Action showMoreYlabel %d", ud->up->showMoreYLabels );

   /*  this happen when setting default value from rc */
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created  */
   }
   g_list_foreach(ud->panelList, (GFunc) pa_panel_label_meas_box_update, NULL );
   ap_all_redraw(ud);
}

static void
aw_scientific_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->scientific = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created */
   }
   msg_dbg ("ud->up->scientific = %d", ud->up->scientific);
   g_list_foreach(ud->panelList, (GFunc) pa_panel_label_size, NULL );
   ap_all_redraw(ud);
   ap_mbtn_update_all(ud);
   ap_xmeasure_button_draw(ud);
}

static void
aw_logx_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->setLogX = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created */
   }
   msg_dbg ("userPrefs->setLogX = %d", ud->up->setLogX);
   GawLabels *lbx = ud->xLabels;
   al_label_set_logAxis( lbx, ud->up->setLogX);
   az_cmd_zoom_absolute(ud, lbx->start_val, lbx->end_val );

   ap_all_redraw(ud);
}

static void
aw_ydiff_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->showYDiff = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created */
   }
   ap_mbtn_update_all(ud);
}


static void aw_logy_panel_set (WavePanel *wp, int active )
{
   if ( active && ! al_label_get_logAble(wp->yLabels) ) {
      msg_info ( _("Logarithmic Y axe not available for this panel (null or negative values in series)"));
      return;
   }
   al_label_set_logAxis(wp->yLabels, active );
   pa_panel_full_redraw(wp);
}

/*
 * preference menu logY action
 */
static void
aw_logy_selected_action (GtkToggleAction *action, UserData *ud )
{
   WavePanel *wp = ud->selected_panel;
   static int toggled;
      
   toggled = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
   msg_dbg ("Action aw_logy_selected_action %d", toggled );

   if ( ! wp) {
      if ( ! toggled) { /* no other way to modify the toggle state : stupid ! */ 
	 return;
      }
      gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), FALSE );
      msg_info (aw_panel_not_selected_msg);
      return;
   }
   aw_update_toggle_state(wp->ui, "/PanelPopup/pPLogY", toggled);
   aw_logy_panel_set (wp, toggled );
}

/*
 * panel action
 */
static void
aw_pop_logy_action (GtkAction *action, WavePanel *wp )
{
//   msg_dbg("aw_pop_logy_action wp = 0x%x", wp );
   if ( wp ) {
      int active = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
      aw_logy_panel_set (wp, active );
      if ( wp->ud->selected_panel == wp ) {
	 aw_update_toggle_state(wp->ud->ui,
			"/MenuBar/PreferencesMenu/LogYPanel", active);
      }
   }
}


static void
aw_grid_panels_set (UserData *ud )
{
   GList *list = ud->panelList;
   while (list) {
      GList *next = list->next;

      WavePanel *wp = (WavePanel *) list->data;  
      wp->showGrid =  ud->up->showGrid;
      char *path = "/PanelPopup/pShowGrid";
      aw_update_toggle_state(wp->ui, path, wp->showGrid );
      pa_panel_full_redraw(wp);
      list = next;
   }
}

static void
aw_show_grid_action (GtkToggleAction *action, UserData *ud )
{
   ud->up->showGrid = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
   if ( ud->xlabel_box == NULL ){ 
      return; /* widget not yet created */
   }
   aw_grid_panels_set ( ud );
}

void aw_show_grid_cmd (UserData *ud, int on )
{
   ud->up->showGrid = on;
   aw_grid_panels_set ( ud );
}

static void aw_pop_show_grid_action (GtkAction *action, WavePanel *wp )
{
   if ( wp ) {
      wp->showGrid = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
//      msg_dbg("showGrid = %d", wp->showGrid );
      pa_panel_full_redraw(wp);
   }
}


static void aw_pop_add_panel_above_action (GtkAction *action, WavePanel *wp )
{
   if ( wp ) {
      ap_panel_add_line( wp->ud, wp, 0);
   }
}


static void aw_pop_add_panel_below_action (GtkAction *action, WavePanel *wp )
{
   if ( wp ) {
      ap_panel_add_line( wp->ud, wp, 1);
   }
}


static void
aw_add_panel_action (GtkAction *action,  UserData *ud)
{
   ap_panel_add_line(ud, NULL, 0);
}

static void
aw_remove_panel_action (GtkAction *action,  UserData *ud)
{
   ap_panel_remove_line(ud, NULL);
}

static void aw_pop_remove_this_panel_action (GtkAction *action, WavePanel *wp )
{
   if ( wp ) {
      ap_panel_remove_line(wp->ud, wp);
   }
}


static void
aw_pop_delete_all_panel_wave_action (GtkAction *action,WavePanel *wp )
{
   if ( wp ) {
      pa_panel_vw_delete(wp);
   }
}

static void
aw_delete_this_wave_action (GtkAction *action,  UserData *ud)
{
//   msg_dbg("called");
   VisibleWave *vw = (VisibleWave *) aw_get_menu_data (ud->ui, "/ButtonPopup",
						    "ButtonPopup-action" );
   if ( vw ) {
      wave_destroy(vw);
      ap_all_redraw(ud);
   }
}

static void aw_delete_all_wave_action (GtkAction *action,  UserData *ud)
{
   ap_remove_all_wave_selected(ud);
}

static void aw_pop_delete_all_wave_action (GtkAction *action, WavePanel *wp)
{
   ap_remove_all_wave_selected(wp->ud);
}

static void aw_reload_all_files_action (GtkAction *action,  UserData *ud )
{
   g_list_foreach(ud->wdata_list, (GFunc) ap_reload_wave_file, NULL);
}

static void aw_pop_reload_all_files_action (GtkAction *action, WavePanel *wp )
{
   g_list_foreach(wp->ud->wdata_list, (GFunc) ap_reload_wave_file, NULL);
}

static void
aw_algo_activate_action (GtkAction *action, UserData *ud)
{
   const gchar *name = gtk_action_get_name (action);
   int i = 0;
   WaveDrawMethod *tbl = wavedraw_method_tab;

   while (tbl->func) {
      if ( app_strcmp(tbl->desc, name) == 0 ) {
         ud->up->drawAlgo = i;
         msg_dbg( "drawAlgo = %d => %s", i, name );
         g_list_foreach(ud->panelList, (GFunc) pa_panel_set_drawing_func, NULL);
         ap_all_redraw(ud);
         break;
      }
      i++;
      tbl++;
   }
}


static void
aw_var_list_create_action (GtkAction *action, DataFile *wdata)
{
   /* wlist_win is null if win has been destroyed */
   if ( wdata) {
//      msg_dbg( "calling datafile_create_list_win");
      datafile_create_list_win (wdata);
   }
}

static void
aw_this_wave_add_action (GtkAction *action,  UserData *ud)
{
//   msg_dbg("called");
   WaveVar *var = (WaveVar *) aw_get_menu_data (ud->ui, "/ListButtonPopup",
                                                    "ListButtonPopup-action" );
   if ( var ) {
      ap_panel_add_var( NULL, var, NULL);
   }
}

static void
aw_all_similar_add_action (GtkAction *action,  UserData *ud)
{
//   msg_dbg("called");
   WaveVar *var = (WaveVar *) aw_get_menu_data (ud->ui, "/ListButtonPopup",
                                                    "ListButtonPopup-action" );
   if ( var ) {
      DataFile *wdata = (DataFile *) wavetable_get_datafile((WaveTable *) var->wvtable);
      datafile_similar_vars_add (wdata, var);
   }
}

static void
aw_all_similar_delete_action (GtkAction *action,  UserData *ud)
{
//   msg_dbg("called");
   WaveVar *var = (WaveVar *) aw_get_menu_data (ud->ui, "/ListButtonPopup",
                                                    "ListButtonPopup-action" );
   if ( var ) {
      ap_remove_all_wave_if_var_name( ud, var);
   }
}
	   

static void
aw_script_exec_action (GtkAction *action,  UserData *ud)
{
   msg_info( "aw_script_exec_action has to be implemented...");
}

static void
aw_show_about_action (GtkAction *action,  UserData *ud)
{
   gaw_show_about_dialog (GTK_WIDGET (ud->window)); 
}

static void
aw_show_userguide_action (GtkAction *action,  UserData *ud)
{
   ah_show_userguide_dialog (ud); 
}

static void
aw_show_website_action (GtkAction *action,  UserData *ud)
{
   ah_show_website_dialog (ud); 
}

void aw_sound_dialog_action (GtkAction *action,  UserData *ud)
{
   if ( ! ud->sndData ) {
      ud->sndData = as_sound_new(ud);
      if ( ! ud->sndData ) {
         msg_info(_("Sound interface is not compiled in !"));
	 return ;
      }
   }
   as_sound_win_create(ud->sndData);
}

/*
 * Warning: toolbar entries are duplicated for labelling purpose.
 */

static GtkActionEntry entries[] = {
  { "FileMenu", NULL, N_("_File") },               /* name, stock id, label */
  { "ViewMenu", NULL, N_("_View") },               /* name, stock id, label */
  { "ZoomMenu", NULL, N_("_Zoom") },               /* name, stock id, label */
  { "VLMenuAction", GTK_STOCK_ADD, N_("Variable List") },   /* name, stock id, label */
  { "CursorMenu", NULL, N_("_Cursors") },          /* name, stock id, label */
  { "PreferencesMenu", NULL, N_("_Preferences") }, /* name, stock id, label */
  { "AlgoMenuAction", GTK_STOCK_ADD, N_("Algorithm List") },   /* name, stock id, label */
  { "ToolsMenu", NULL, N_("_Tools") },             /* name, stock id, label */
  { "HelpMenu", NULL, N_("_Help") },               /* name, stock id, label */
  { "Open", GTK_STOCK_OPEN,                    /* name, stock id */
   N_("_Open..."),"<control>O",                /* label, accelerator */     
   N_("Open a file"),                          /* tooltip */
    G_CALLBACK (af_open_file_action) }, 
  { "Capture", GTK_STOCK_PRINT,                /* name, stock id */
   N_("_Capture..."), "<control>C",            /* label, accelerator */     
   N_("Input data from the sound card"),       /* tooltip */
    G_CALLBACK (aw_sound_dialog_action) }, 
  { "ExportDis", GTK_STOCK_FILE,               /* name, stock id */
   N_("Export waves..."), NULL,                /* label, accelerator */     
   N_("Export displayed waves to a data file"),   /* tooltip */
    G_CALLBACK (af_export_displayed_action) }, 
  { "ExportImg", GTK_STOCK_FILE,               /* name, stock id */
   N_("Export to image..."), NULL,             /* label, accelerator */     
   N_("Export all panels to an image file"),   /* tooltip */
    G_CALLBACK (im_export_panels_img_action) }, 
  { "Plot", GTK_STOCK_PRINT,                   /* name, stock id */
   N_("_Plot..."), "<control>P",               /* label, accelerator */     
   N_("Output dat in plot format"),            /* tooltip */
    G_CALLBACK (activate_action) }, 
  { "SaveConf", GTK_STOCK_SAVE,                /* name, stock id */
   N_("_Save conf"),"<control>S",              /* label, accelerator */     
   N_("Save configuration file"),              /* tooltip */
    G_CALLBACK (aw_save_config_action) },
  { "ClearDef",  GTK_STOCK_UNDO,               /* name, stock id */
   N_("Clear _Default"),"<control>D",          /* label, accelerator */     
   N_("Clear default input file name and Save configuration file"),  /* tooltip */
    G_CALLBACK (aw_default_file_name_action) },
  { "DefConf",  GTK_STOCK_UNDO,                  /* name, stock id */
   N_("Factory Settings"),"<control>D",          /* label, accelerator */     
   N_("Reset configuration file to its default"),  /* tooltip */
    G_CALLBACK (aw_factory_settings_action) },
  { "Execute", GTK_STOCK_EXECUTE,               /* name, stock id */
   N_("Execute script..."), NULL,               /* label, accelerator */     
   N_("Execute a guile script"),                /* tooltip */
    G_CALLBACK (aw_script_exec_action) },
  { "TextAction", GTK_STOCK_EDIT,               /* name, stock id */
   N_("Open Text tool..."), NULL,               /* label, accelerator */     
   N_("Open text tool settings"),               /* tooltip */
    G_CALLBACK (gawtext_new_action) },
  { "TTextAction", GTK_STOCK_EDIT,              /* name, stock id */
   N_("Text"), "<control>T",                    /* label, accelerator */     
   N_("Open text tool settings"),               /* tooltip */
    G_CALLBACK (gawtext_new_action) },
  { "Quit", GTK_STOCK_QUIT,                    /* name, stock id */
   N_("_Quit"), "<control>Q",                  /* label, accelerator */     
   N_("Quit"),                                 /* tooltip */
    G_CALLBACK (activate_action) },
  { "AddPanel", GTK_STOCK_ADD,                 /* name, stock id */
   N_("_Add Panel at end"), NULL,                     /* label, accelerator */     
   N_("Add a new panel"),                      /* tooltip */
    G_CALLBACK (aw_add_panel_action) },
  { "TAddPanel", GTK_STOCK_ADD,                 /* name, stock id */
   N_("_Add Panel"), NULL,                     /* label, accelerator */     
   N_("Add a new panel"),                      /* tooltip */
    G_CALLBACK (aw_add_panel_action) },
  { "DeletePanel", GTK_STOCK_CLEAR,            /* name, stock id */
   N_("Remove Panel"), "<control>D",          /* label, accelerator */     
   N_("Remove selected or last panel"),        /* tooltip */
    G_CALLBACK (aw_remove_panel_action) },
  { "TDeletePanel", GTK_STOCK_CLEAR,            /* name, stock id */
   N_("Del Panel"), "<control>D",          /* label, accelerator */     
   N_("Remove selected or last panel"),        /* tooltip */
    G_CALLBACK (aw_remove_panel_action) },
  { "ZoomCursor0", GTK_STOCK_ZOOM_FIT,         /* name, stock id */
   N_("_Zoom centered C0"), NULL,              /* label, accelerator */     
   N_("Zoom in centered on cursor 0"),         /* tooltip */
    G_CALLBACK (az_zoom_cursor0_centered_action) },
  { "TZoomCursor0", GTK_STOCK_ZOOM_FIT,         /* name, stock id */
   N_("_Z C0"), NULL,                       /* label, accelerator */     
   N_("Zoom in centered on cursor 0"),         /* tooltip */
    G_CALLBACK (az_zoom_cursor0_centered_action) },
  { "ZoomCursors", GTK_STOCK_ZOOM_FIT,         /* name, stock id */
   N_("_Zoom Cursors"), NULL,                  /* label, accelerator */     
   N_("Display space between 2 cursors"),      /* tooltip */
    G_CALLBACK (az_zoom_cursors_action) },
  { "TZoomCursors", GTK_STOCK_ZOOM_FIT,        /* name, stock id */
   N_("_Z Cursors"), NULL,                     /* label, accelerator */     
   N_("Display space between 2 cursors"),      /* tooltip */
    G_CALLBACK (az_zoom_cursors_action) },
  { "ZoomXFull", GTK_STOCK_ZOOM_OUT,           /* name, stock id */
   N_("X unZoom all"), "<control>Z",               /* label, accelerator */     
   N_("Unzoom X all"),                         /* tooltip */
    G_CALLBACK (az_zoom_x_full_action) },
  { "TZoomXFull", GTK_STOCK_ZOOM_OUT,           /* name, stock id */
   N_("X unZ"), "<control>Z",               /* label, accelerator */     
   N_("Unzoom X all"),                         /* tooltip */
    G_CALLBACK (az_zoom_x_full_action) },
  { "ZoomYFull", GTK_STOCK_ZOOM_OUT,           /* name, stock id */
   N_("Zoom Y Full"), NULL,                    /* label, accelerator */     
   N_("Unzoom Y  all"),                        /* tooltip */
    G_CALLBACK (az_zoom_y_full_action) },
  { "ZoomX", GTK_STOCK_ZOOM_FIT,               /* name, stock id */
   N_("Zoom X..."), NULL,                      /* label, accelerator */     
   N_("Select X range with button 1"),         /* tooltip */
    G_CALLBACK (az_zoom_x_action) },
  { "ZoomY", GTK_STOCK_ZOOM_FIT,               /* name, stock id */
   N_("Zoom Y..."), NULL,                      /* label, accelerator */     
   N_("Select Y range with button 1"),         /* tooltip */
    G_CALLBACK (az_zoom_y_action) },
  { "ZoomXYarea", GTK_STOCK_ZOOM_FIT,          /* name, stock id */
   N_("Zoom X+Y..."), NULL,                    /* label, accelerator */     
   N_("Select X+Y range with button 1"),       /* tooltip */
    G_CALLBACK (az_zoom_xy_area_action) },
  { "ZoomDialog", GTK_STOCK_ZOOM_FIT,          /* name, stock id */
   N_("Zoom Dialog..."), NULL,                 /* label, accelerator */     
   N_("Zoom Dialog menu"),                     /* tooltip */
    G_CALLBACK (az_zoom_dialog_action) },
  { "RedrawAll", GTK_STOCK_REFRESH,            /* name, stock id */
   N_("_Redraw All"), "<control>A",            /* label, accelerator */     
   N_("Redraw All panels"),                    /* tooltip */
    G_CALLBACK (aw_redraw_all_action) },
  { "ClearCursor0", GTK_STOCK_CLEAR,           /* name, stock id */
   N_("Clear Cursor 0"), NULL,                 /* label, accelerator */     
   N_("Clear cursor 0 if displayed"),          /* tooltip */
    G_CALLBACK (cu_cursor_clear_cursor0_action) },
  { "ClearCursor1", GTK_STOCK_CLEAR,           /* name, stock id */
   N_("Clear Cursor 1"), NULL,                 /* label, accelerator */     
   N_("Clear cursor 1 if displayed"),          /* tooltip */
    G_CALLBACK (cu_cursor_clear_cursor1_action) },
  { "ClearCursors", GTK_STOCK_CLEAR,           /* name, stock id */
   N_("Clear all Cursors"), NULL,              /* label, accelerator */     
   N_("Clear all cursors if displayed"),       /* tooltip */
    G_CALLBACK (cu_cursor_clear_cursors_action) },
  { "SetCursor0", GTK_STOCK_APPLY,             /* name, stock id */
   N_("Set Cursor 0"), NULL,                   /* label, accelerator */     
   N_("Display cursor 0"),                     /* tooltip */
    G_CALLBACK (cu_cursor_set_cursor0_action) },
  { "SetCursor1", GTK_STOCK_APPLY,             /* name, stock id */
   N_("Set Cursor 1"), NULL,                   /* label, accelerator */     
   N_("Display cursor 1"),                     /* tooltip */
    G_CALLBACK (cu_cursor_set_cursor1_action) },
  { "SetCursors", GTK_STOCK_APPLY,             /* name, stock id */
   N_("Set all Cursors"), NULL,                /* label, accelerator */     
   N_("Display all cursors"),                  /* tooltip */
    G_CALLBACK (cu_cursor_set_cursors_action) },
  { "About", NULL,                             /* name, stock id */
   N_("_About"), NULL,                         /* label, accelerator */     
   N_("About"),                                /* tooltip */  
    G_CALLBACK (aw_show_about_action) },
  { "UserGuide", GTK_STOCK_HELP,               /* name, stock id */
   N_("UserGuide"), NULL,                      /* label, accelerator */     
   N_("Gaw User Guide"),                       /* tooltip */  
    G_CALLBACK (aw_show_userguide_action) },
  { "GawWebSite", GTK_STOCK_HOME,              /* name, stock id */
   N_("Gaw Web Site"), NULL,                      /* label, accelerator */     
   N_("Gaw Web Site"),                         /* tooltip */  
    G_CALLBACK (aw_show_website_action) },
  { "ZoomIn", GTK_STOCK_ZOOM_IN,               /* name, stock id */
   N_("_Zoom X In"), "Z",                      /* label, accelerator */     
   N_("Zoom X In"),                            /* tooltip */
    G_CALLBACK (az_zoom_in_action) },
  { "TZoomIn", GTK_STOCK_ZOOM_IN,              /* name, stock id */
   N_("_Z In"), "Z",                        /* label, accelerator */     
   N_("Zoom X In"),                            /* tooltip */
    G_CALLBACK (az_zoom_in_action) },
  { "ZoomOut", GTK_STOCK_ZOOM_OUT,             /* name, stock id */
   N_("_Zoom X Out"), "<shift>Z",              /* label, accelerator */     
   N_("Zoom X Out"),                           /* tooltip */
    G_CALLBACK (az_zoom_out_action) },
  { "TZoomOut", GTK_STOCK_ZOOM_OUT,            /* name, stock id */
   N_("_Z Out"), "<shift>Z",                /* label, accelerator */     
   N_("Zoom X Out"),                           /* tooltip */
    G_CALLBACK (az_zoom_out_action) },
  { "DeleteWave", GTK_STOCK_DELETE,            /* name, stock id */
   N_("Delete Wave"), NULL,                    /* label, accelerator */     
   N_("Delete all selected Waves"),            /* tooltip */
    G_CALLBACK (aw_delete_all_wave_action) },
  { "TDeleteWave", GTK_STOCK_DELETE,            /* name, stock id */
   N_("Del Wave"), NULL,                    /* label, accelerator */     
   N_("Delete all selected Waves"),            /* tooltip */
    G_CALLBACK (aw_delete_all_wave_action) },
  { "DelThisWave", GTK_STOCK_DELETE,           /* name, stock id */
   N_("Delete this Wave"), NULL,               /* label, accelerator */     
   N_("Delete this Wave"),                     /* tooltip */
    G_CALLBACK (aw_delete_this_wave_action) },
  { "ReloadAll", GTK_STOCK_FILE,               /* name, stock id */
   N_("Reload All waves"), NULL,               /* label, accelerator */     
   N_("Reread all waveform data files"),       /* tooltip */
    G_CALLBACK (aw_reload_all_files_action) },
  { "TReloadAll", GTK_STOCK_FILE,               /* name, stock id */
   N_("Reload All"), NULL,                     /* label, accelerator */     
   N_("Reread all waveform data files"),       /* tooltip */
    G_CALLBACK (aw_reload_all_files_action) },
  { "ChangeColor", GTK_STOCK_SELECT_COLOR,     /* name, stock id */
   N_("Change Color..."), NULL,                /* label, accelerator */     
   N_("Change wave color"),                    /* tooltip */
    G_CALLBACK (ac_color_wave_action) },
  { "PanelColor", GTK_STOCK_SELECT_COLOR,    /* name, stock id */
   N_("Change panel Colors..."), NULL,         /* label, accelerator */     
   N_("Change colors used in panel drawing area"),   /* tooltip */
    G_CALLBACK (ac_color_panel_colors_action) },
  { "AddThisSelected", GTK_STOCK_ADD,          /* name, stock id */
   N_("Add this to selected"), NULL,           /* label, accelerator */     
   N_("Add this wave to selected panel"),      /* tooltip */
    G_CALLBACK (aw_this_wave_add_action) },
  { "AddAllSelected", GTK_STOCK_ADD,           /* name, stock id */
   N_("Add all similar to selected"), NULL,    /* label, accelerator */     
   N_("Add all similar waves to selected panel"),  /* tooltip */
    G_CALLBACK (aw_all_similar_add_action) },
  { "DelAllSimilar", GTK_STOCK_DELETE,         /* name, stock id */
   N_("Delete all similar"), NULL,             /* label, accelerator */     
   N_("Delete all similar waves from panels"),    /* tooltip */
    G_CALLBACK (aw_all_similar_delete_action) },
};
static guint n_entries = G_N_ELEMENTS (entries);


static GtkToggleActionEntry toggle_entries[] = {
  { "showXlabels", NULL,                       /* name, stock id */
   N_("show X labels"), NULL,                  /* label, accelerator */     
   N_("show labels in X"),                     /* tooltip */
    G_CALLBACK (aw_showXlabel_action), 
    TRUE },                                    /* is_active */
  { "showYlabels", NULL,                       /* name, stock id */
   N_("show Y labels"), NULL,                  /* label, accelerator */     
   N_("show labels in Y"),                     /* tooltip */
    G_CALLBACK (aw_showYlabel_action), 
    TRUE },                                    /* is_active */
  { "moreYlabels", NULL,                       /* name, stock id */
   N_("show more  Y labels"), NULL,            /* label, accelerator */     
   N_("different way to show labels in Y"),    /* tooltip */
    G_CALLBACK (aw_showMoreYlabel_action), 
    TRUE },                                    /* is_active */
  { "Scientific", NULL,                        /* name, stock id */
   N_("Scientific conversion"), NULL,          /* label, accelerator */     
   N_("Use Scientific conversion mode"),       /* tooltip */
    G_CALLBACK (aw_scientific_action), 
    FALSE },                                   /* is_active */
  { "LogX", NULL,                              /* name, stock id */
   N_("log _X scale"), "<control>X",           /* label, accelerator */     
   N_("Log scale in X"),                       /* tooltip */
    G_CALLBACK (aw_logx_action), 
    FALSE },                                   /* is_active */
  { "LogYPanel", NULL,                         /* name, stock id */
   N_("log _Y selected panel"), NULL,          /* label, accelerator */     
   N_("Set Log scale in Y in selected panel"), /* tooltip */
    G_CALLBACK (aw_logy_selected_action), 
    TRUE },                                   /* is_active */
  { "ShowGrid", NULL,                         /* name, stock id */
   N_("Show Grid"), NULL,                     /* label, accelerator */     
   N_("Show Grid in panels"),                 /* tooltip */
    G_CALLBACK (aw_show_grid_action), 
    FALSE },                                   /* is_active */
  { "Ydiff", NULL,                             /* name, stock id */
   N_("show Y diff measure"), "<control>Y",    /* label, accelerator */     
   N_("show Y diff measure buttons"),          /* tooltip */
    G_CALLBACK (aw_ydiff_action), 
    FALSE },                                   /* is_active */
  { "BarStyle", GTK_STOCK_PREFERENCES,         /* name, stock id */
   N_("ToolBar Style Icon+text"), NULL,        /* label, accelerator */     
   N_("Toggle toolbar style"),                 /* tooltip */  
    G_CALLBACK (toggle_bar_style_action),
    TRUE },
  { "AllowResize", NULL,                        /* name, stock id */
   N_("Allow Resize"), NULL,                    /* label, accelerator */     
   N_("Allow Resize the main window"),          /* tooltip */  
    G_CALLBACK (toggle_allow_resize_action),
    TRUE },
};
static guint n_toggle_entries = G_N_ELEMENTS (toggle_entries);


static const gchar *ui_info = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu action='FileMenu'>"
"      <menuitem action='Open'/>"
"      <menuitem action='Capture'/>"
"      <menuitem action='ExportDis'/>"
"      <menuitem action='ExportImg'/>"
//"      <menuitem action='Plot'/>"
"      <separator/>"
"      <menuitem action='SaveConf'/>"
"      <menuitem action='ClearDef'/>"
"      <menuitem action='DefConf'/>"
//"      <menuitem action='Execute'/>"
"      <separator/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='ViewMenu'>"
"      <menuitem action='AddPanel'/>"
"      <menuitem action='RedrawAll'/>"
"      <menu  action='VLMenuAction'></menu>"
"    </menu>"
"    <menu action='ZoomMenu'>"
"      <menuitem action='ZoomCursor0'/>"
"      <menuitem action='ZoomCursors'/>"
"      <menuitem action='ZoomDialog'/>"
"      <menuitem action='ZoomX'/>"
"      <menuitem action='ZoomY'/>"
"      <menuitem action='ZoomXYarea'/>"
"      <menuitem action='ZoomXFull'/>"
"      <menuitem action='ZoomYFull'/>"
"      <menuitem action='ZoomIn'/>"
"      <menuitem action='ZoomOut'/>"
"    </menu>"
"    <menu action='CursorMenu'>"
"      <menuitem action='SetCursor0'/>"
"      <menuitem action='SetCursor1'/>"
"      <menuitem action='SetCursors'/>"
"      <menuitem action='ClearCursor0'/>"
"      <menuitem action='ClearCursor1'/>"
"      <menuitem action='ClearCursors'/>"
"    </menu>"
"    <menu action='PreferencesMenu'>"
"      <menuitem action='BarStyle'/>"
"      <menuitem action='showXlabels'/>"
"      <menuitem action='showYlabels'/>"
"      <menuitem action='moreYlabels'/>"
"      <menuitem action='LogX'/>"
"      <menuitem action='LogYPanel'/>"
"      <menuitem action='Ydiff'/>"
"      <menuitem action='ShowGrid'/>"
"      <menuitem action='Scientific'/>"
"      <separator action='Sep1'/>"
"      <menuitem action='PanelColor'/>"
"      <separator action='Sep2'/>"
"      <menuitem action='AllowResize'/>"
"      <separator action='Sep3'/>"
"      <menu  action='AlgoMenuAction'></menu>"
"    </menu>"
"    <menu action='ToolsMenu'>"
"      <menuitem action='TextAction'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='About'/>"
"      <menuitem action='UserGuide'/>"
"      <menuitem action='GawWebSite'/>"
"    </menu>"
"  </menubar>"
"  <toolbar  name='ToolBar'>"
"    <toolitem action='TAddPanel'/>"
"    <toolitem action='TDeletePanel'/>"
"    <separator action='Sep1'/>"
"    <toolitem action='TDeleteWave'/>"
"    <toolitem action='TReloadAll'/>"
"    <toolitem action='TTextAction'/>"
"    <separator action='Sep2'/>"
"    <toolitem action='TZoomIn'/>"
"    <toolitem action='TZoomOut'/>"
"    <toolitem action='TZoomCursor0'/>"
"    <toolitem action='TZoomCursors'/>"
"    <toolitem action='TZoomXFull'/>"
"  </toolbar>"
"  <popup  name='ButtonPopup'>"
"    <menuitem action='DelThisWave'/>"
"    <menuitem action='ChangeColor'/>"
"    <menuitem action='ReloadAll'/>"
"  </popup>"
"  <popup  name='ListButtonPopup'>"
"    <menuitem action='AddThisSelected'/>"
"    <menuitem action='AddAllSelected'/>"
"    <menuitem action='DelAllSimilar'/>"
"  </popup>"
"</ui>";


typedef struct _ActionStatus ActionStatus;

struct _ActionStatus {
   GtkAction *action;
   GtkWidget *statusbar;
};

static void
aw_action_status_destroy (gpointer data)
{
   ActionStatus *action_status = data;
   
   g_object_unref (action_status->action);
   g_object_unref (action_status->statusbar);

   g_free (action_status);
}

static void
aw_set_tip (GtkWidget *widget)
{
   ActionStatus *data;
   gchar *tooltip;
  
   data = g_object_get_data (G_OBJECT (widget), "action-status");
  
   if (data) { 
      g_object_get (data->action, "tooltip", &tooltip, NULL);
      if ( tooltip ){
         tmem_alloced(tooltip, 0);   /* trace mem */
         gtk_statusbar_push (GTK_STATUSBAR (data->statusbar), 0, tooltip );
         g_free (tooltip);
      }
   }
}

static void
aw_unset_tip (GtkWidget *widget)
{
  ActionStatus *data;

  data = g_object_get_data (G_OBJECT (widget), "action-status");

  if (data) {
    gtk_statusbar_pop (GTK_STATUSBAR (data->statusbar), 0);
  }
}
		    
void
aw_connect_proxy (GtkUIManager *ui, GtkAction *action, GtkWidget *proxy,
	       GtkWidget *statusbar)
{
//   msg_info("called");
   if (GTK_IS_MENU_ITEM (proxy)) {
      ActionStatus *data;

      data = g_object_get_data (G_OBJECT (proxy), "action-status");
      if (data) {
	 g_object_unref (data->action);
	 g_object_unref (data->statusbar);
	 
	 data->action = g_object_ref (action);
	 data->statusbar = g_object_ref (statusbar);
      } else {
	 data = g_new0 (ActionStatus, 1);
	 
	 data->action = g_object_ref (action);
	 data->statusbar = g_object_ref (statusbar);
	 
	 g_object_set_data_full (G_OBJECT (proxy), "action-status", 
				 data, aw_action_status_destroy);
	  
	 g_signal_connect (proxy, "select",
			   G_CALLBACK (aw_set_tip), NULL);
	 g_signal_connect (proxy, "deselect",
			   G_CALLBACK (aw_unset_tip), NULL);
      }
   }
}

void 
aw_vl_menu_item_add( DataFile *wdata)
{
   UserData *ud = (UserData *) wdata->ud;
   gchar name[32] ;
   GtkAction *action;
   const gchar *path = "/MenuBar/ViewMenu/VLMenuAction";
   const gchar *tooltip = _("reCreate variable List window");

   wdata->merge_id = gtk_ui_manager_new_merge_id (ud->ui);
   
   if (ud->vlactions == NULL ) {
      ud->vlactions = gtk_action_group_new ("DynamicActions");
      gtk_ui_manager_insert_action_group (ud->ui, ud->vlactions, 0);
   }
   
   sprintf(name, "varlist%d", wdata->ftag);
   action = g_object_new (GTK_TYPE_ACTION,
			  "name", name,
			  "label", wdata->wt->tblname,
			  "tooltip" , tooltip,
			  "stock-id", GTK_STOCK_FILE,
			  NULL);
   gtk_action_group_add_action (ud->vlactions, action);
   g_signal_connect (action, "activate",
		     G_CALLBACK (aw_var_list_create_action), wdata);
   g_object_unref (action);
   
   gtk_ui_manager_add_ui (ud->ui, wdata->merge_id, path,  name, name,
                         GTK_UI_MANAGER_MENUITEM, FALSE );
   gtk_ui_manager_ensure_update (ud->ui);
   msg_dbg("%s", name);
}

/*
 * remove item from variable list menu
 */
void 
aw_vl_menu_item_remove( DataFile *wdata)
{
   UserData *ud = (UserData *) wdata->ud;
   gchar name[64] ;
   GtkAction *action;
   const gchar *path = "/MenuBar/ViewMenu/VLMenuAction";

   if (ud->vlactions == NULL ) {
      return;
   }
   
   sprintf(name, "%s/varlist%d", path, wdata->ftag);

   action = gtk_ui_manager_get_action (ud->ui, name);
   g_object_unref (action);

   gtk_ui_manager_remove_ui (ud->ui, wdata->merge_id);
   gtk_ui_manager_ensure_update (ud->ui);
}

void 
aw_update_toggle_state(GtkUIManager *ui, const gchar *path, gboolean is_active)
{
   GtkAction *action = gtk_ui_manager_get_action (ui, path);
   
   if (action) {
      if ( gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)) != is_active) {
	 gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), is_active);
      }
   } else {
      msg_fatal (_("Can't not find action '%s'"), path);
   }
}

void 
aw_update_action_sensitive(GtkUIManager *ui, const gchar *path, gboolean is_sensitive )
{
   GtkAction *action = gtk_ui_manager_get_action (ui, path);
   
   if (action) {
      if (  gtk_action_get_sensitive (GTK_ACTION (action)) != is_sensitive ) {
	 gtk_action_set_sensitive (GTK_ACTION (action), is_sensitive );
      }
   } else {
      msg_fatal (_("Can't not find action '%s'"), path);
   }
}

void aw_algo_menu_item_add( UserData *ud, char *name)
{
   GtkAction *action;
   const gchar *path = "/MenuBar/PreferencesMenu/AlgoMenuAction";
   const gchar *tooltip = _("Drawing algorith selection");

   int merge_id = gtk_ui_manager_new_merge_id (ud->ui);
   
   action = g_object_new (GTK_TYPE_ACTION,
                          "name", name,
                          "label", name,
                          "tooltip" , tooltip,
                          "stock-id", GTK_STOCK_FILE,
                          NULL);
   gtk_action_group_add_action (ud->actions, action);
   g_signal_connect (action, "activate",
                     G_CALLBACK (aw_algo_activate_action), ud);
//   g_object_unref (action);
   
   gtk_ui_manager_add_ui (ud->ui, merge_id, path,  name, name,
                         GTK_UI_MANAGER_MENUITEM, FALSE );
   gtk_ui_manager_ensure_update (ud->ui);
   msg_dbg("%s", name);
}

void aw_algo_menu_create( UserData *ud )
{
   WaveDrawMethod *tbl = wavedraw_method_tab;

   while (tbl->func) {
      aw_algo_menu_item_add( ud, tbl->desc);
      tbl++;
   }
}

GtkUIManager *
aw_create_ui_manager ( UserData *ud )
{
   GtkActionGroup *actions;
   GError *error = NULL;
   GtkUIManager *ui;
   GtkWidget *label;
   GtkWidget *bar;
   GtkWidget *toolitem;
   GtkWidget *statusbar;
   
   actions = gtk_action_group_new ("Actions");
   ud->actions = actions;
   gtk_action_group_set_translation_domain (actions, GETTEXT_PACKAGE );
   gtk_action_group_add_actions (actions, entries, n_entries, ud );
   gtk_action_group_add_toggle_actions (actions, 
					toggle_entries, n_toggle_entries, 
					ud);
   
   /* status bar */
   statusbar = gtk_statusbar_new();
   gtk_statusbar_set_has_resize_grip ( GTK_STATUSBAR (statusbar), FALSE);
   ud->statusbar = statusbar;
   gtk_widget_show(statusbar);
//   label = GTK_WIDGET(GTK_STATUSBAR (statusbar)->label);
   label = hq_gtk_statusbar_get_label (statusbar);
   ud->statusLabel = label;
   gtk_widget_set_name(label, "statusbar" );
//   g_object_ref (statusbar); /* increment ref to avoid destruction */

   /* ui manager */
   ui = gtk_ui_manager_new ();
   ud->ui = ui;
   g_signal_connect (ud->ui, "connect-proxy",
		     G_CALLBACK (aw_connect_proxy), statusbar);
   g_object_set_data_full (G_OBJECT (ud->window), "ui-manager", ui, 
			      g_object_unref);       
   gtk_ui_manager_insert_action_group (ui, actions, 0);
   gtk_window_add_accel_group (GTK_WINDOW (ud->window), 
			       gtk_ui_manager_get_accel_group (ui));
      
   if ( ! gtk_ui_manager_add_ui_from_string (ui, ui_info, -1, &error)){
      msg_fatal (_("building menus failed: %s"), error->message);
      g_error_free (error);
   }


   bar = gtk_ui_manager_get_widget (ui, "/MenuBar");
   gtk_widget_show (bar);
   ud->menuBar = bar;

   /* Create the toolbar */
   bar = gtk_ui_manager_get_widget (ui, "/ToolBar");
   ud->toolBar = bar;
   gtk_widget_show (bar);

   /* add target dnd to toolbar delete button */
   toolitem = gtk_ui_manager_get_widget (ud->ui, "/ToolBar/TDeleteWave");
   ad_set_drag_dest(toolitem, ud, NULL, DND_DELETE_BUT);
   toolitem = gtk_ui_manager_get_widget (ud->ui, "/ToolBar/TAddPanel");
   ad_set_drag_dest(toolitem, ud, NULL, DND_ADD_BUT);

   /* modify toggle button state from userPrefs */
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/showXlabels", ud->up->showXLabels);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/showYlabels", ud->up->showYLabels);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/moreYlabels", ud->up->showMoreYLabels);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/LogX", ud->up->setLogX);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/LogYPanel", 0);
   aw_update_action_sensitive(ui, "/MenuBar/PreferencesMenu/LogYPanel", 0);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/Ydiff", ud->up->showYDiff);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/ShowGrid", ud->up->showGrid);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/Scientific", ud->up->scientific);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/AllowResize", ud->up->allowResize);
   aw_update_toggle_state(ui, "/MenuBar/PreferencesMenu/BarStyle", ud->up->toolBarStyle);
   gtk_toolbar_set_style(GTK_TOOLBAR(bar), (GtkToolbarStyle) ud->up->toolBarStyle );

   aw_algo_menu_create( ud );
   return ui;
}

/****************************************************************/
/*
 * menu stuff for panel popup menu
 */

static GtkActionEntry pop_entries[] = {
  { "pAddPanelAbove", GTK_STOCK_ADD,           /* name, stock id */
   N_("_Add Panel Above"), NULL,               /* label, accelerator */     
   N_("Add a new panel above this one"),       /* tooltip */
    G_CALLBACK (aw_pop_add_panel_above_action) },
  { "pAddPanelBelow", GTK_STOCK_ADD,           /* name, stock id */
   N_("_Add Panel Below"), NULL,               /* label, accelerator */     
   N_("Add a new panel below this one"),       /* tooltip */
    G_CALLBACK (aw_pop_add_panel_below_action) },
  { "pDelThisPanel", GTK_STOCK_CLEAR,          /* name, stock id */
   N_("_Remove this Panel"), "<control>D",     /* label, accelerator */     
   N_("Remove this panel"),                    /* tooltip */
    G_CALLBACK (aw_pop_remove_this_panel_action) },
  { "pZoomCursors", GTK_STOCK_ZOOM_FIT,        /* name, stock id */
   N_("_Zoom Cursors"), NULL,                  /* label, accelerator */     
   N_("Display space between 2 cursors"),      /* tooltip */
    G_CALLBACK (az_pop_zoom_cursors_action) },
  { "pZoomXFull", GTK_STOCK_ZOOM_OUT,          /* name, stock id */
   N_("X unZoom"), "<control>Z",               /* label, accelerator */     
   N_("Unzoom X all"),                         /* tooltip */
    G_CALLBACK (az_pop_zoom_x_full_action) },
  { "pZoomYFull", GTK_STOCK_ZOOM_OUT,          /* name, stock id */
   N_("Zoom Y Full"), NULL,                    /* label, accelerator */     
   N_("Unzoom Y  all"),                        /* tooltip */
    G_CALLBACK (az_pop_zoom_y_full_action) },
  { "pZoomX", GTK_STOCK_ZOOM_FIT,              /* name, stock id */
   N_("Zoom X..."), NULL,                      /* label, accelerator */     
   N_("Select X range with button 1"),         /* tooltip */
    G_CALLBACK (az_pop_zoom_x_action) },
  { "pZoomY", GTK_STOCK_ZOOM_FIT,              /* name, stock id */
   N_("Zoom Y..."), NULL,                      /* label, accelerator */     
   N_("Select Y range with button 1"),         /* tooltip */
    G_CALLBACK (az_pop_zoom_y_action) },
  { "pZoomXYarea", GTK_STOCK_ZOOM_FIT,         /* name, stock id */
   N_("Zoom X+Y..."), NULL,                    /* label, accelerator */     
   N_("Select X+Y range with button 1"),       /* tooltip */
    G_CALLBACK (az_pop_zoom_xy_area_action) },
  { "pZoomDialog", GTK_STOCK_ZOOM_FIT,         /* name, stock id */
   N_("Zoom Dialog..."), NULL,                 /* label, accelerator */     
   N_("Zoom Dialog menu"),                     /* tooltip */
    G_CALLBACK (az_pop_zoom_dialog_action) },
  { "pZoomIn", GTK_STOCK_ZOOM_IN,              /* name, stock id */
   N_("_Zoom X In"), "Z",                      /* label, accelerator */     
   N_("Zoom X In"),                            /* tooltip */
    G_CALLBACK (az_pop_zoom_in_action) },
  { "pZoomOut", GTK_STOCK_ZOOM_OUT,            /* name, stock id */
   N_("_Zoom X Out"), "<shift>Z",              /* label, accelerator */     
   N_("Zoom X Out"),                           /* tooltip */
    G_CALLBACK (az_pop_zoom_out_action) },
  { "pDeleteWave", GTK_STOCK_DELETE,           /* name, stock id */
   N_("Delete Wave"), NULL,                    /* label, accelerator */     
   N_("Delete all selected Waves"),            /* tooltip */
    G_CALLBACK (aw_pop_delete_all_wave_action) },
  { "pDeleteWaves", GTK_STOCK_DELETE,          /* name, stock id */
   N_("Delete Waves"), NULL,                   /* label, accelerator */     
   N_("Delete all Waves in this panel"),       /* tooltip */
    G_CALLBACK (aw_pop_delete_all_panel_wave_action) },
  { "pReloadAll", GTK_STOCK_FILE,              /* name, stock id */
   N_("Reload All"), NULL,                     /* label, accelerator */     
   N_("Reread all waveform data files"),       /* tooltip */
    G_CALLBACK (aw_pop_reload_all_files_action) },
  { "pGridColor", GTK_STOCK_SELECT_COLOR,      /* name, stock id */
   N_("Grid Change Color..."), NULL,           /* label, accelerator */     
   N_("Change the grid color in this panel"),        /* tooltip */
    G_CALLBACK (ac_pop_color_grid_action) },
  { "pExportPng", GTK_STOCK_CONVERT,          /* name, stock id */
   N_("Export to Image..."), NULL,            /* label, accelerator */     
   N_("Export this wave to Image file"),      /* tooltip */
    G_CALLBACK (im_pop_export_to_img_action) },
  { "pDeleteTexts", GTK_STOCK_DELETE,       /* name, stock id */
   N_("Delete all texts"), NULL,            /* label, accelerator */     
   N_("Delete all texts in this panel"),    /* tooltip */
    G_CALLBACK (gawtext_pop_delete_all_text_action) },
  { "pEditThisText", GTK_STOCK_EDIT,        /* name, stock id */
   N_("Edit this text ..."), NULL,          /* label, accelerator */     
   N_("Edit this text"),                    /* tooltip */
    G_CALLBACK (gawtext_pop_edit_action) },
  { "pDelThisText", GTK_STOCK_DELETE,    /* name, stock id */
   N_("Delete this text"), NULL,            /* label, accelerator */     
   N_("Delete this text"),                  /* tooltip */
    G_CALLBACK (gawtext_pop_delete_action) },
};
static guint n_pop_entries = G_N_ELEMENTS (pop_entries);

static GtkToggleActionEntry pop_toggle_entries[] = {
  { "pPLogY", NULL,                            /* name, stock id */
   N_("log Y scale"), NULL,                    /* label, accelerator */     
   N_("Set Log scale in Y for this panel"),    /* tooltip */
    G_CALLBACK (aw_pop_logy_action), 
    FALSE },                                   /* is_active */
  { "pShowGrid", NULL,                         /* name, stock id */
   N_("Show Grid"), NULL,                      /* label, accelerator */     
   N_("Show Grid in this panel"),              /* tooltip */
    G_CALLBACK (aw_pop_show_grid_action), 
    FALSE },                                   /* is_active */
};

static guint n_pop_toggle_entries = G_N_ELEMENTS (pop_toggle_entries);

static const gchar *pop_ui_info = 
"<ui>"
"  <popup  name='PanelPopup'>"
"    <menuitem action='pZoomCursors'/>"
"    <menuitem action='pZoomX'/>"
"    <menuitem action='pZoomY'/>"
"    <menuitem action='pZoomXFull'/>"
"    <menuitem action='pZoomYFull'/>"
"    <menuitem action='pZoomXYarea'/>"
"    <menuitem action='pZoomDialog'/>"
"    <menuitem action='pZoomIn'/>"
"    <menuitem action='pZoomOut'/>"
"    <separator action='Sep1'/>"
"    <menuitem action='pAddPanelAbove'/>"
"    <menuitem action='pAddPanelBelow'/>"
"    <menuitem action='pDelThisPanel'/>"
"    <menuitem action='pExportPng'/>"
"    <separator action='Sep2'/>"
"    <menuitem action='pReloadAll'/>"
"    <menuitem action='pDeleteWave'/>"
"    <menuitem action='pDeleteWaves'/>"
"    <menuitem action='pPLogY'/>"
"    <menuitem action='pShowGrid'/>"
"    <menuitem action='pGridColor'/>"
"    <separator action='Sep3'/>"
"    <menuitem action='pDeleteTexts'/>"
"  </popup>"
"  <popup  name='PanelTextPopup'>"
"    <menuitem action='pEditThisText'/>"
"    <menuitem action='pDelThisText'/>"
"  </popup>"
"</ui>";


GtkUIManager *
aw_pop_create_ui_manager ( WavePanel *wp )
{
   UserData *ud = wp->ud;
   GtkActionGroup *actions;
   GError *error = NULL;
   GtkUIManager *ui;
   GtkWidget *bar;
   
   actions = gtk_action_group_new ("PopActions");
   gtk_action_group_set_translation_domain (actions, GETTEXT_PACKAGE );
   gtk_action_group_add_actions (actions, pop_entries, n_pop_entries, wp );
   gtk_action_group_add_toggle_actions (actions, 
                                        pop_toggle_entries, n_pop_toggle_entries, 
                                        wp);

   /* ui manager */
   ui = gtk_ui_manager_new ();
   g_signal_connect ( ui, "connect-proxy",
		     G_CALLBACK (aw_connect_proxy), ud->statusbar);
   gtk_ui_manager_insert_action_group (ui, actions, 0);
   gtk_window_add_accel_group (GTK_WINDOW (ud->window), 
                               gtk_ui_manager_get_accel_group (ui));
      
   if ( ! gtk_ui_manager_add_ui_from_string (ui, pop_ui_info, -1, &error)){
      msg_error (_("building menus failed: %s"), error->message);
      g_error_free (error);
   }

   bar = gtk_ui_manager_get_widget (ui, "/PanelPopup");
   gtk_widget_show (bar);

   return ui;
}


/****************************************************************/
/*
 * menu stuff for variable list window
 */

static void
aw_vl_reload_action (GtkAction *action, DataFile *wdata )
{
  ap_reload_wave_file (wdata);
}

static void
aw_vl_unload_action (GtkAction *action, DataFile *wdata )
{
   ap_delete_datafile(wdata);
}

static void
aw_vl_close_action (GtkAction *action, DataFile *wdata )
{
//   msg_dbg("called");
   gtk_widget_destroy (wdata->wlist_win);
}


static GtkActionEntry vlentries[] = {
  { "FileMenu", NULL, N_("_File") },           /* name, stock id, label */
  { "Reload", GTK_STOCK_REFRESH,               /* name, stock id */
   N_("_Reload this file"), NULL,              /* label, accelerator */     
   N_("Reload this file"),                     /* tooltip */
    G_CALLBACK (aw_vl_reload_action) },
  { "Unload", GTK_STOCK_CLEAR,                 /* name, stock id */
   N_("_Unload this file"), NULL,              /* label, accelerator */     
   N_("Unload this file"),                     /* tooltip */
    G_CALLBACK (aw_vl_unload_action) },
  { "Export", GTK_STOCK_CONVERT,               /* name, stock id */
   N_("Export data..."), NULL,                 /* label, accelerator */     
   N_("Export data from this file"),           /* tooltip */
    G_CALLBACK (af_vl_export_data_action) },
  { "Close", GTK_STOCK_QUIT,                   /* name, stock id */
   N_("_Close"), NULL,                         /* label, accelerator */     
   N_("Close this window"),                    /* tooltip */
    G_CALLBACK (aw_vl_close_action) },
};
static guint n_vlentries = G_N_ELEMENTS (vlentries);

static const gchar *vlui_info = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu action='FileMenu'>"
"      <menuitem action='Reload'/>"
"      <menuitem action='Unload'/>"
"      <menuitem action='Export'/>"
"      <separator/>"
"      <menuitem action='Close'/>"
"    </menu>"
"  </menubar>"
"</ui>";

GtkUIManager *
aw_vl_create_ui_manager ( DataFile *wdata )
{
   UserData *ud = (UserData *) wdata->ud;
   GtkActionGroup *actions;
   GError *error = NULL;
   GtkUIManager *ui;
   GtkWidget *bar;
   
   actions = gtk_action_group_new ("Actions");
   gtk_action_group_set_translation_domain (actions, GETTEXT_PACKAGE );
   gtk_action_group_add_actions (actions, vlentries, n_vlentries, wdata );

   /* ui manager */
   ui = gtk_ui_manager_new ();
   wdata->ui = ui;
   g_signal_connect ( ui, "connect-proxy",
		     G_CALLBACK (aw_connect_proxy), ud->statusbar);
   g_object_set_data_full (G_OBJECT (wdata->wlist_win), "ui-manager", ui, 
			      g_object_unref);       
   gtk_ui_manager_insert_action_group (ui, actions, 0);
   gtk_window_add_accel_group (GTK_WINDOW (wdata->wlist_win), 
			       gtk_ui_manager_get_accel_group (ui));
      
   if ( ! gtk_ui_manager_add_ui_from_string (ui, vlui_info, -1, &error)){
      msg_error (_("building menus failed: %s"), error->message);
      g_error_free (error);
   }

   bar = gtk_ui_manager_get_widget (ui, "/MenuBar");
   gtk_widget_show (bar);
   gtk_box_pack_start (GTK_BOX (wdata->wlist_vbox), bar,
		       FALSE, FALSE, 0);

   return ui;
}

/****************************************************************/

/*
 * GtkDialog
 */

int aw_dialog_show ( int type, const char *text)
{
   GtkWidget *dialog;
   UserData *ud = userData;
   char buffer[64] = "";
   gint response;
   GtkWidget *vbox;
   GtkWidget *hbox;
   GtkWidget *image;
   GtkWidget *label;
   char *cancel_button = 0;
   int cancel_resp = 0;
   gchar *stock_id[] = {
      GTK_STOCK_HELP,
	 GTK_STOCK_DIALOG_INFO,
	 GTK_STOCK_DIALOG_WARNING,
	 GTK_STOCK_DIALOG_ERROR,
	 GTK_STOCK_STOP,
   };

   if ( type & AW_MSG_T_SHOW_CANCEL) {
      cancel_button = GTK_STOCK_CANCEL;
      cancel_resp = GTK_RESPONSE_CANCEL;
   }
   type &=  AW_MSG_T_MASK;
   if ( type == MSG_T_FATAL) {
      strcpy( buffer, _("Fatal - "));
   }
   strcat(buffer, _("Gaw Dialog"));
   
   if ( ! gtk_widget_get_realized (GTK_WIDGET(ud->window) ) ) {
      msg_dbg( "%s", text );
      aw_do_exit ( NULL, ud);
   }

   dialog = gtk_dialog_new_with_buttons (buffer,
                                         GTK_WINDOW (ud->window),
                                         GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_OK,
					 cancel_button,
					 cancel_resp,
                                         NULL);

   vbox = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), 
		       hbox, TRUE, TRUE, 0);

   image = gtk_image_new_from_stock  ( stock_id[type],
				      GTK_ICON_SIZE_DIALOG );
   gtk_box_pack_start (GTK_BOX (hbox), 
		       image, TRUE, TRUE, 0);

   label = gtk_label_new ( text );
   gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
   gtk_label_set_max_width_chars (GTK_LABEL (label), 100);
   gtk_misc_set_padding (GTK_MISC (label), 10, 10);
   gtk_box_pack_start (GTK_BOX (hbox), 
		       label, TRUE, TRUE, 0);

   gtk_widget_show_all (vbox);

   response = gtk_dialog_run (GTK_DIALOG (dialog));
   gtk_widget_destroy (dialog);

   if ( type == MSG_T_FATAL && ! prog_debug ){
      gtk_widget_destroy (ud->window);
   }
   return response;
}

/*
 * Main Window and Exit
 */

void
aw_do_exit (GtkWidget *widget,  UserData *ud )
{
   msg_dbg( "called 0x%lx", (unsigned long) ud->window);
   aw_main_window_destroy ( ud ) ;     

   /* if widget is not NULL, this function is called from callback */
   if ( widget == NULL && ud->window ) {
      gtk_widget_destroy (ud->window);
   }

   gtk_main_quit ();
}


/*
 * calculate the size of main window and set size
 */
void aw_window_size(UserData *ud )
{
   int width;
   int height ;
   GtkRequisition req;

   if ( ! ud->panel_scrolled) {
      return ;     /* not yet  ready to go */
   }
   width = 2 * gtk_container_get_border_width( GTK_CONTAINER(ud->globalTable));
   height =  width ;
//   msg_dbg( "w %d, h %d", width, height );

   width += ud->up->lmtableWidth;
//   msg_dbg( "w %d, h %d", width, height );

   /* panelWidth is panel size without scrollbar */  
   width += ud->up->panelWidth;
//   msg_dbg( "w %d, h %d", width, height );

   if ( ! ud->sbSize ) {
     /*  sbSize = w(panel_scrolled) - w(panelTable) */
      gtk_widget_size_request ( GTK_WIDGET(ud->panel_scrolled), &req);
      ud->sbSize = req.width;
      gtk_widget_size_request( GTK_WIDGET(ud->panelTable), &req);
      ud->sbSize -= req.width;
   }
   /* request 1 pixel more */
   width += ud->sbSize ;

   gtk_widget_size_request( GTK_WIDGET(ud->menuBar), &req);
   height += req.height;
//   msg_dbg( "w %d, h %d", width, height );
   
   gtk_widget_size_request( GTK_WIDGET(ud->toolBar), &req);
   height += req.height;
//   msg_dbg( "w %d, h %d", width, height );
   
   if ( ud->meas_hbox_shown ) {
      gtk_widget_size_request( GTK_WIDGET(ud->meas_hbox), &req);
      height += req.height;
      msg_dbg( "w %d, h %d", width, height );
   }
   if ( ud->up->showXLabels ) { 
      gtk_widget_size_request( GTK_WIDGET(ud->xlabel_ev_box), &req);
      height += req.height;
//      msg_dbg( "w %d, h %d", width, height );
   }
   gtk_widget_size_request( GTK_WIDGET(ud->xscrollbar), &req);
   height += req.height;
   msg_dbg( "w %d, h %d", width, height );

   
   /* add the height of scrolled window */
   height += ud->panelScrolledHeight;     /* height = 112 + panelScrolledHeight */
//   msg_dbg( "w %d, h %d", width, height );

   ud->reqWinWidth = width;
   ud->reqWinHeight = height;
   /* when gtk is alive, use the current width */
//   if ( ud->winWidth ) {
//      width = ud->winWidth ;
//   }
   if ( ud->up->allowResize  == 0 || ud->winWidth == 0 ) {
      gtk_window_resize (GTK_WINDOW (ud->window), ud->reqWinWidth, ud->reqWinHeight);
   }
   msg_dbg( "w %d, h %d panelH %d", ud->reqWinWidth, ud->reqWinHeight, ud->panelScrolledHeight );
}

void aw_panel_scrolled_set_size_request( UserData *ud )
{
   int height;
   int np = g_list_length( ud->panelList);
   int h;

   /* 1 + panelHeight + 1 */
   height = np * (ud->up->panelHeight + 2);

   /* if not configured or enough room for panel */
   if ( ud->maxHeight == 0 || height < ud->maxHeight ){
       h = height ;
   } else {
      h = (ud->maxHeight / (ud->up->panelHeight + 2)) *
	 (ud->up->panelHeight + 2);
   }

   h += 4;
   ud->panelScrolledHeight = h ;

//   gtk_widget_set_size_request(GTK_WIDGET(ud->panel_scrolled), -1, h );
   gtk_widget_set_size_request(GTK_WIDGET(ud->panelTable), -1, height );

   msg_dbg( "npanel %d, hscrolled %d htable %d", np, h, height );
   aw_window_size(ud);
}


static gboolean
aw_window_configure_cb( GtkWidget *widget, GdkEventConfigure *event,
		      gpointer data )
{
   UserData *ud = (UserData *) data;
   gint root_x, root_y;
   gint max_y;
   GtkAllocation walloc;

   gtk_widget_get_allocation (widget, &walloc);  
   msg_dbg( "x %d, y %d, w %d, h %d new %d %d 0x%lx",
	    event->x,  event->y,
	    walloc.width,
	    walloc.height,
	    event->width,  event->height,
	    (long unsigned int) widget );

   GdkWindow *window = gtk_widget_get_window (ud->panel_scrolled);
   gdk_window_get_root_coords(window, 0, 0,  &root_x, &root_y);
   GtkAllocation alloc;
   gtk_widget_get_allocation (ud->panel_scrolled, &alloc);

   root_x += alloc.x;
   root_y += alloc.y;
   max_y =  root_y + alloc.height + event->height - walloc.height;
   msg_dbg( "rootx %d, root_y %d, alloc.height %d, max_y %d",
	    root_x, root_y, alloc.height, max_y);

   GdkWindowState wst = gdk_window_get_state (window);
   if ( wst & GDK_WINDOW_STATE_MAXIMIZED) {
      msg_dbg( "maximize max y on screen = %d", max_y );
      if ( ! ud->up->max_ps_y ){
	 gtk_window_unmaximize (GTK_WINDOW(widget));
      }
      ud->up->max_ps_y = max_y;
      return FALSE; /* FALSE to propagate the event further. */
   }
   GtkAdjustment *adj;
   ud->maxHeight = ud->up->max_ps_y - root_y;
   if ( ud->maxHeight <= 0 ){
      ud->maxHeight = 826;
   }
   adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW (ud->panel_scrolled) );
   gtk_adjustment_set_page_size (adj, ud->maxHeight);

   ud->winWidth = walloc.width;
   ud->winHeight = walloc.height;
   
   if ( ud->winWidth != ud->reqWinWidth || ud->winHeight != ud->reqWinHeight ) {
      msg_dbg( "ww %d rww %d wh %d rwh %d", ud->winWidth, ud->reqWinWidth,
             ud->winHeight, ud->reqWinHeight);
      aw_window_size(ud);
   }

   return FALSE; /* FALSE to propagate the event further. */
}


static gboolean
aw_window_expose_cb( GtkWidget *widget, GdkEventExpose *event,
		      gpointer data )
{
   UserData *ud = (UserData *) data;
   GtkAllocation walloc;

   gtk_widget_get_allocation (widget, &walloc);   
#if 0
   msg_dbg( "x %d, y %d, w %d, h %d 0x%lx",
	    event->area.x,  event->area.y,
	    walloc.width,
	    walloc.height,
	    (long unsigned int) widget );
#endif
   
   int w = ud->winWidth;
   ud->winWidth = walloc.width;
   if ( w == 0 ) {
      /* first expose -  compensation - to be explained */ 
      ud->panelScrolledHeight += 2;
      aw_window_size(ud);
   }
   
   return FALSE; /* FALSE to propagate the event further. */
}

gint aw_window_idle( gpointer pdata)
{
   UserData *ud = (UserData *) pdata;
   gtk_window_maximize (GTK_WINDOW(ud->window));
   return FALSE; /* FALSE to remove the function. */
}

void aw_scrollbar_show_cb( GtkWidget *widget, gpointer pdata)
{
   UserData *ud = (UserData *) pdata;
   GawLabels *lbx = ud->xLabels;
   if ( ! lbx ) {
      return;
   }
   lbx->changed &= ~(CV_SBCHANGED | CV_SBSHOW );
   lbx->changed |= CV_SBCHANGED | CV_SBSHOW ;
   msg_dbg("show w %d, h %d, sbstate 0x%x", lbx->w, lbx->h, lbx->changed);
   return;
}
void aw_scrollbar_hide_cb( GtkWidget *widget, gpointer pdata)
{
   UserData *ud = (UserData *) pdata;
   GawLabels *lbx = ud->xLabels;
   if ( ! lbx ) {
      return;
   }
   lbx->changed &= ~(CV_SBCHANGED | CV_SBSHOW );
   lbx->changed |= CV_SBCHANGED ;
   msg_dbg("show w %d, h %d, sbstate 0x%x", lbx->w, lbx->h, lbx->changed);
   return;
}

void aw_create_main_window ( UserData *ud )
{
   GtkWidget *window;
   GtkWidget *globalTable;
   GtkWidget *panel_scrolled;
   GtkAdjustment *adj;
   GdkGeometry geometry;
   int i;

   /* create a global GawLabels object for X labels */
   ud->xLabels = al_label_new(ud->up, ud->up->setLogX, 1);
   /* create a global SelRange object */
   ud->srange = g_new0(SelRange, 1);

   window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   ud->window = window;
   gtk_window_set_title (GTK_WINDOW (window), ud->mainName );
   gtk_widget_set_name (window, ud->mainName );

   g_signal_connect (window, "destroy",
		     G_CALLBACK (aw_do_exit), (gpointer) ud );
   g_signal_connect (window, "delete_event",
		     G_CALLBACK (gtk_false), NULL);
   g_signal_connect (window, "configure-event",
		     G_CALLBACK (aw_window_configure_cb), (gpointer) ud);
   g_signal_connect (window, "expose-event",
		     G_CALLBACK (aw_window_expose_cb), (gpointer) ud);

   /* create global Table */
   globalTable = gtk_table_new(6, 2, FALSE); /* 6 row 2 col */
   ud->globalTable = globalTable;
   gtk_widget_show(globalTable);
   gtk_container_set_border_width (GTK_CONTAINER (globalTable), WIN_BORDER_SIZE);
   gtk_container_add (GTK_CONTAINER (window), globalTable);

   /*  create menu, toolbar, statusBar */
   aw_create_ui_manager (ud);
   gtk_table_attach(GTK_TABLE(globalTable), ud->menuBar, 
		    /* X direction */       /* Y direction */
		    0, 2,                     0, 1, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_table_attach(GTK_TABLE(globalTable), ud->toolBar, 
		    /* X direction */       /* Y direction */
		    0, 2,                     1, 2, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* create the 3 X measure buttons */
   ap_create_xmeasure_unit(ud);
   gtk_table_attach(GTK_TABLE(globalTable),  ud->meas_hbox, 
		    /* X direction */       /* Y direction */
		    0, 2,                     2, 3, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);


   /* horizontal box for X-axis labels */
   ap_xlabel_box_create(ud);
   /* create horizontal scrollbar */
   ap_create_win_bottom(ud);
   
   /* create panel table */
   ud->panelTable = gtk_table_new(1, AW_PANELTABLE_COLS, FALSE); /* 1 row 2 col */
   gtk_widget_show(ud->panelTable);

   /* create a scrolled window to contains panels with vertical scrollbar */
   panel_scrolled = gtk_scrolled_window_new (NULL, NULL);
   ud->panel_scrolled = panel_scrolled ;
   gtk_widget_set_size_request (GTK_WIDGET(panel_scrolled), -1, 30);

   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (panel_scrolled),
                                  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
   GtkWidget *vscrollbar = gtk_scrolled_window_get_vscrollbar(GTK_SCROLLED_WINDOW(panel_scrolled));

   gtk_widget_set_can_focus (vscrollbar, FALSE);
   gtk_widget_show (panel_scrolled);
   gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(panel_scrolled),
                                         ud->panelTable);

   adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW (panel_scrolled) );
   gtk_container_set_focus_vadjustment( GTK_CONTAINER (ud->panelTable), adj );
   gtk_adjustment_set_page_size (adj, gtk_adjustment_get_upper(adj) ); 

   /*  set callback for scroll bar */
   GtkWidget *scr = gtk_scrolled_window_get_vscrollbar(
			    GTK_SCROLLED_WINDOW(ud->panel_scrolled)) ;
   g_signal_connect (scr, "show",
		     G_CALLBACK (aw_scrollbar_show_cb), (gpointer) ud);
   g_signal_connect (scr, "hide",
		     G_CALLBACK (aw_scrollbar_hide_cb), (gpointer) ud);
   
   gtk_table_attach(GTK_TABLE(globalTable), panel_scrolled, 
		    /* X direction */       /* Y direction */
		    0, 2,                     3, 4, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

   
   /* create 2 panels */
   for (i = 0 ; i < ud->reqpanels ; i++) {
      ap_panel_add_line(ud, NULL, 0);
   }

   /*  attach xlabel hbox line to vbox container */
   gtk_table_attach(GTK_TABLE(globalTable), ud->xlabel_ev_box, 
		    /* X direction */       /* Y direction */
		    0, 2,                     4, 5, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* bottom line statusbar, X scrollbar */
   gtk_table_attach(GTK_TABLE(globalTable), ud->statusbar, 
		    /* X direction */       /* Y direction */
		    0, 1,                     5, 6, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_table_attach(GTK_TABLE(globalTable), ud->xscrollbar, 
		    /* X direction */       /* Y direction */
		    1, 2,                     5, 6, 
		    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* status bar test */
   gtk_statusbar_push (GTK_STATUSBAR (ud->statusbar), 0, 
			  _("Welcome to gaw"));

   if ( ! gtk_widget_get_visible (GTK_WIDGET(window)) ) {
      gtk_widget_show (window);
      memset(&geometry, 0, sizeof(geometry));
      geometry.min_width = ud->up->min_win_width;
      geometry.min_height = ud->up->min_win_height;
      gtk_window_set_geometry_hints (GTK_WINDOW(window), NULL,
                                     &geometry,  GDK_HINT_MIN_SIZE );
      gtk_window_move (GTK_WINDOW(window), 160, -1);
   }

   if ( ud->listFiles ){
      af_load_wave_file (ud);
   }
   /* compute the lower point on screen */
   if ( ! ud->up->max_ps_y ) {
      g_idle_add ((GSourceFunc) aw_window_idle, (gpointer) ud );
   }
//   ap_all_redraw(ud);
}
   
void aw_rc_parse_gtkrc(UserData *ud)
{
   gchar *filename;

   gchar *rc = "gaw-gtkrc";
   filename = app_strcatdup ( GAWRCDIR, "/", rc, NULL);
   gtk_rc_parse (filename);
   app_free (filename);
  
   if (ud->up->rcDir) {
      filename = app_strcatdup ( ud->up->rcDir, "/", rc, NULL);
      gtk_rc_parse (filename);
      app_free (filename);
  }
}

void aw_main_window_destroy ( UserData *ud )
{
   int i;

   if ( ! ud->xLabels ){ /* avoid callback */
      return;
   }
   af_unload_all_files(ud);
   
   /* GawLabels object for X labels */
   al_label_destroy(ud->xLabels);
   ud->xLabels = NULL;
   /* global SelRange object */
   g_free(ud->bg_color);
   g_free(ud->pg_color);
   g_free(ud->hl_color);
   g_free(ud->srange->color);
   g_free(ud->srange);
   for ( i = 0 ; i < 2 ; i++) {
      g_free(ud->cursors[i]->color);
   }
   for ( i = 0 ; i < AW_NX_MBTN ; i++) {
      g_free(ud->cursors[i]);
   }
   g_free(ud->cursors);

   /* drag and drop dest data */
   i = g_list_length( ud->destdata_list);
   for (  ; i > 0 ; i--) {
      void *p = g_list_nth_data ( ud->destdata_list, i - 1);
      g_free(p);
   }

   /* destroy panels */
   i = g_list_length( ud->panelList);
   for (  ; i > 0 ; i--) {
      WavePanel *wp = (WavePanel *) g_list_nth_data (ud->panelList, i - 1);
      pa_panel_destroy(wp);
   }
   gtk_widget_destroy(ud->panel_scrolled);
 
   /* global Table */
   gtk_widget_destroy(ud->globalTable);

//   gtk_widget_destroy(ud->window);
}
   

void aw_usage(char *prog)
{
   fprintf(stderr, 
_("Usage: %s [OPTIONS]... [FILE]...\n"
  "\n"
  "                : display analog waveforms from input FILE...\n"
  " [-h]           : print list of options\n"
  " [-d]           : set debug mode\n"
  " [-rf fmt]      : specify spice format for input file\n"
  " [-pf fmt]      : specify a printf format for output file\n"
  " [-p num]       : port number for inside server\n"
  " [-C name]      : configuration dir or file name\n")
       , prog);   
   exit(1);
}

int
main (int argc, char *argv[])
{
   int i;
   int done = 0;
   UserData *ud;
   char *penv;
   char *rcfile = NULL;
   int user_debug = 0;

   tmem_init( 0 );
   srand (time (NULL));
   progname = app_strdup( basename(argv[0]));

//   gtk_set_locale (); /* unusefull, this is set in gtk_init */

   /* init defaults */
   if ( ! userData) {
     userData = g_new0 (UserData, 1);
   }
   ud = userData;
   msg_initlog(progname, MSG_F_NO_DATE, NULL, NULL );
   
   ud->prog = progname ;
   ud->mainName = progname;
   ud->NWColors = 6;

   if ( ( penv = getenv("GAW_HOMEDIR")) ){
      rcfile = app_strdup( penv);
   }

   /* global messages */
  aw_panel_not_selected_msg =
  _("\n\nThis function requires a panel to be selected\n"
   "Select a panel by left clicking in one DrawingArea\n");

   
   for ( i = 1 ; i < argc ; i++ ) {
      if (*argv[i] == '-') {
         if(strcmp(argv[i], "-d") == 0) {
	    user_debug++ ;
#ifndef MSG_DEBUG
	    msg_info(_("Compile with -DMSG_DEBUG, to get debug infos."));
#endif	 
	 } else if (strcmp(argv[i], "-h") == 0) {
	    aw_usage(progname);
	 } else if (strcmp(argv[i], "-rf") == 0) {
	    app_free(ud->format);
	    ud->format = app_strdup( argv[++i] );
	 } else if (strcmp(argv[i], "-pf") == 0) {
	    ud->printFmt = argv[++i];
	 } else if (strcmp(argv[i], "-p") == 0) {
	    ud->listenPort = atoi(argv[++i]);
	 } else if (strcmp(argv[i], "-C") == 0) {
	    app_free(rcfile);
	    rcfile = app_strdup(argv[++i]);
	 }
      } else {
	 if ( ! done ){
	    done++;
	    af_list_files_free(&ud->listFiles);
	 }
         if (file_exists (argv[i])) {
            ud->listFiles = g_slist_append( ud->listFiles, app_strdup( argv[i]) );
         }
      }
   }
   
   UserPrefs *up = up_new( ud->prog, rcfile, NULL);
   if ( user_debug ) {
     prog_debug = user_debug;
   }
   ud->up = up;
   app_free(rcfile);

   ud->reqpanels = up->npanels; 
   ud->listenPort = up->listenPort;
   /* only if data files not given on command line */
   if ( ud->listFiles == NULL && up->lastDataFile && *up->lastDataFile ){
      if (file_exists (up->lastDataFile)) {
         ud->listFiles = g_slist_append( ud->listFiles,
				      app_strdup(up->lastDataFile));
      }
   }
   if ( up->dataFileFormat && *up->dataFileFormat ) {
      ud->format =  app_strdup( up->dataFileFormat );
   }
   ud->bits = 16;    /* for writing wav */
   ud->rate = 48000; /* for writing wav */


   do {
      ud->restart = 0;
      
      /*
       * after gtk_init, gtk_init call setlocale
       * before gtk_init for  g_dgettext !!!
       */

#ifdef ENABLE_NLS
#ifndef PACKAGE_LOCALE_DIR
#define PACKAGE_LOCALE_DIR "share/locale/"
#endif

        void *p = bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
        if (p == NULL) {
	   msg_error("bindtextdomain failed");
	}
        bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
        textdomain (GETTEXT_PACKAGE);
#else
        msg_dbg("NLS disabled.");
#endif

      gtk_init (&argc, &argv);
      /* after gtk_init, gtk_init call setlocale for reading numeric string */
      setlocale(LC_NUMERIC, "C");
      aw_rc_parse_gtkrc(ud);

      aw_create_main_window (ud);
      if ( ud->listenPort ) {
	 aio_create_channel(ud);
      }
      /* just before starting gtk_main !!! */
      msg_set_func( aw_dialog_show ); /* redirect msg to a dialog window */
      gtk_main ();
      if ( userData->gawio ) {
	 aio_destroy_channel(userData);
      }
   } while ( ud->restart );

   af_list_files_free( &ud->listFiles );
 
   as_sound_destroy(ud->sndData);
   
   g_free (ud->prog);
   up_destroy(up);

   g_free (ud);

   tmem_destroy(NULL);
   return 0;
}
