

#include <string.h>
#include "runtime.h"


/*****************************************************************/
/* IEEE-754  Little endian subroutines */

#if defined(FPU_387) || defined(FPU_SSE) || defined(FPU_IA64) || \
    defined(FPU_ALPHA)

/* Kind 4 little endian */

/* The default initializations initialize both real and complex kinds. */

const char prefix(snan_4) [] = { 0x01, 0x00, 0x80, 0x7f,

				 0x01, 0x00, 0x80, 0x7f };

const char prefix(snan_8) [] = { 0x01, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0xf0, 0x7f,

				 0x01, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0xf0, 0x7f };

const char prefix(snan_10)[] = { 0x01, 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0xff, 0x7f, 0x00, 0x00,
#if REAL10_SIZE == 16
				 0x00, 0x00, 0x00, 0x00,
#endif
				 0x01, 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0xff, 0x7f, 0x00, 0x00,
#if REAL10_SIZE == 16
				 0x00, 0x00, 0x00, 0x00,
#endif
};


const char prefix(snan_16)[] = { 0x01, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0xFF, 0x7F,

				 0x01, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0xFF, 0x7F };


const packed16 prefix(huge_v16) = { 0xFFFFFFFFU, 0xFFFFFFFFU,
				    0xFFFFFFFFU, 0x7FFEFFFFU };

const packed16 prefix(hugem_v16) = { 0xFFFFFFFFU, 0xFFFFFFFFU,
				     0xFFFFFFFFU, 0xFFFEFFFFU };



static const float half = 0.5;



static inline int unpack_sign_4(void *v) {
unsigned *p;

    p = v;
    return !!(p[0] & 0x80000000);
}


static inline int unpack_exponent_4(void *v) {
unsigned *p;

    p = v;
    return (p[0] >> 23) & 0xFF;
}


void unpack_real_4(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[0] = p[0] & 0x007FFFFF;

    e = unpack_exponent_4(p);
    if (e != 0 && e != EXP4_NAN)
	mantissa[0] |= MAN4_MSW;

    *exp = e;
    *sign = unpack_sign_4(p);
}


void pack_real_4(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;

    if (mantissa != NULL)
	p[0] = (p[0] & 0xFF800000) | (mantissa[0] & 0x007FFFFF);

    if (exp != NULL)
	p[0] = (p[0] & 0x807FFFFF) | ((*exp & 0x0000FF) << 23);

    if (sign != NULL) {
	if (*sign)
	    p[0] |= 0x80000000;
	else
	    p[0] &= 0x7FFFFFFF;
    }
}


/* Kind 8 little endian */

static inline int unpack_sign_8(void *v) {
unsigned *p;

    p = v;
    return !!(p[1] & 0x80000000);
}



static inline int unpack_exponent_8(void *v) {
unsigned *p;

    p = v;
    return (p[1] >> 20) & 0x7FF;
}


void unpack_real_8(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[1] = p[0];
    mantissa[0] = p[1] & 0x000FFFFF;

    e = unpack_exponent_8(p);
    if (e != 0 && e != EXP8_NAN)
	mantissa[0] |= MAN8_MSW;

    *exp = e;
    *sign = unpack_sign_8(p);
}


void pack_real_8(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL) {
	p[0] = mantissa[1];
	p[1] = (p[1] & 0xFFF00000) | (mantissa[0] & 0x000FFFFF);
    }

    if (exp != NULL)
	p[1] = (p[1] & 0x800FFFFF) | ((*exp & 0x7FF) << 20);

    if (sign != NULL) {
	if (*sign)
	    p[1] |= 0x80000000;
	else
	    p[1] &= 0x7FFFFFFF;
    }
}


/* Kind 10 little endian-- x387 native.  No hidden bit. */

static inline int unpack_sign_10(void *v) {
unsigned *p;

    p = v;
    return !!(((unsigned char *) p)[9] & 0x80);
}


static inline int unpack_exponent_10(void *v) {
unsigned *p;

    p = v;
    return (((short *) p)[4]) & 0x7FFF;
}



void unpack_real_10(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    mantissa[1] = p[0];
    mantissa[0] = p[1];

    *exp = unpack_exponent_10(p);
    *sign = unpack_sign_10(p);
}


void pack_real_10(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL) {
	p[0] = mantissa[1];
	p[1] = mantissa[0];
    }

    if (exp != NULL) {
	((short *) p)[4] &= 0x8000;
	((short *) p)[4] |= (*exp & 0x7FFF);
    }

    if (sign != NULL) {
	if (*sign)
	    ((short *) p)[4] |= 0x8000;
	else
	    ((short *) p)[4] &= 0x7FFF;
    }
}


/* Kind 16 little endian-- */

static inline int unpack_sign_16(void *v) {
unsigned *p;

    p = v;
    return !!(p[3] & 0x80000000);
}


static inline int unpack_exponent_16(void *v) {
unsigned *p;

    p = v;
    return (p[3] >> 16) & 0x7FFF;
}



void unpack_real_16(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[3] = p[0];
    mantissa[2] = p[1];
    mantissa[1] = p[2];
    mantissa[0] = p[3] & 0x0000FFFF;

    e = unpack_exponent_16(p);
    if (e != 0 && e != EXP16_NAN)
	mantissa[0] |= 0x00010000;

    *exp = e;
    *sign = unpack_sign_16(p);
}




void pack_real_16(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;

    if (mantissa != NULL) {
	p[0] = mantissa[3];
	p[1] = mantissa[2];
	p[2] = mantissa[1];
	p[3] = (p[3] & 0xFFFF0000) | (mantissa[0] & 0x0000FFFF);
    }

    if (exp != NULL)
	p[3] = (p[3] & 0x8000FFFF) | ((*exp & 0x7FFF) << 16);

    if (sign != NULL) {
	if (*sign)
	    p[3] |= 0x80000000;

	else
	    p[3] &= 0x7FFFFFFF;
    }

    /* On x86-64, quad values are returned in xmm0.  Set us up for
     * returning that value to our caller's caller.  We can't use the
     * same trick for kind=10 reals because the x87 registers are
     * arranged in a stack as opposed to registers. */

#if HAVE_REAL_10 == 3
    asm("movaps %0, %%xmm0" : : "m" (*p));
#endif

}


/*****************************************************************/
/* IEEE-754  Big endian subroutines */

#elif defined(FPU_PPC1) || defined(FPU_PPC2) || defined(FPU_MIPS) || \
      defined(FPU_HPPA) || defined(FPU_SPARC)


/* Kind 4 big endian, identical to little endian */

const char prefix(snan_4) [] = { 0x7f, 0x80, 0x00, 0x01,
				 0x7f, 0x80, 0x00, 0x01 };

const char prefix(snan_8) [] = { 0x7f, 0xf0, 0x00, 0x00,
                                 0x00, 0x00, 0x00, 0x01,

				 0x7f, 0xf0, 0x00, 0x00,
                                 0x00, 0x00, 0x00, 0x01 };

const char prefix(snan_16)[] = { 0x7F, 0xFF, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,

				 0x7F, 0xFF, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00,
				 0x00, 0x00, 0x00, 0x00 };



const packed16 prefix(huge_v16) = { 0x7FFEFFFFU, 0xFFFFFFFFU,
				    0xFFFFFFFFU, 0xFFFFFFFFU };

const packed16 prefix(hugem_v16) = { 0xFFFEFFFFU, 0xFFFFFFFFU,
				     0xFFFFFFFFU, 0xFFFFFFFFU };




static inline int unpack_sign_4(void *v) {
unsigned *p;

    p = v;
    return !!(p[0] & 0x80000000);
}


static inline int unpack_exponent_4(void *v) {
unsigned *p;

    p = v;
    return (p[0] >> 23) & 0xFF;
}


void unpack_real_4(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[0] = p[0] & 0x007FFFFF;

    e = unpack_exponent_4(p);
    if (e != 0 && e != EXP4_NAN)
	mantissa[0] |= 0x00800000;

    *exp = e;
    *sign = unpack_sign_4(p);
}


void pack_real_4(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL)
	p[0] = (p[0] & 0xFF800000) | (mantissa[0] & 0x007FFFFF);

    if (exp != NULL)
	p[0] = (p[0] & 0x807FFFFF) | ((*exp & 0x0000FF) << 23);

    if (sign != NULL) {
	if (*sign)
	    p[0] |= 0x80000000;
	else
	    p[0] &= 0x7FFFFFFF;
    }
}


/* Kind 8 big endian */

static inline int unpack_sign_8(void *v) {
unsigned *p;

    p = v;
    return !!(p[0] & 0x80000000);
}


static inline int unpack_exponent_8(void *v) {
unsigned *p;

    p = v;
    return (p[0] >> 20) & 0x7FF;
}



void unpack_real_8(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[1] = p[1];
    mantissa[0] = p[0] & 0x000FFFFF;

    e = unpack_exponent_8(p);
    if (e != 0 && e != EXP8_NAN)
	mantissa[0] |= 0x00100000;

    *exp = e;
    *sign = unpack_sign_8(p);
}


void pack_real_8(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL) {
	p[1] = mantissa[1];
	p[0] = (p[0] & 0xFFF00000) | (mantissa[0] & 0x000FFFFF);
    }

    if (exp != NULL)
	p[0] = (p[0] & 0x800FFFFF) | ((*exp & 0x7FF) << 20);

    if (sign != NULL) {
	if (*sign)
	    p[0] |= 0x80000000;
	else
	    p[0] &= 0x7FFFFFFF;
    }
}



/* Kind 10 big endian--  No hidden bit. */

static inline int unpack_sign_10(void *v) {
unsigned *p;

    p = v;
    return !!(((unsigned char *) p)[0] & 0x80);
}


static inline int unpack_exponent_10(void *v) {
unsigned *p;

    return (((short *) p)[0]) & 0x7FFF;
}



void unpack_real_10(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    mantissa[1] = p[1];
    mantissa[0] = p[0];

    *exp = unpack_exponent_8(p);
    *sign = unpack_sign_8(p);
}



void pack_real_10(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL) {
	p[1] = mantissa[1];
	p[0] = mantissa[0];
    }

    if (exp != NULL) {
	((short *) p)[0] &= 0x7FFF;
	((short *) p)[0] |= ((*exp & 0x7FFF) << 20);
    }

    if (sign != NULL) {
	if (*sign)
	    ((short *) p)[0] |= 0x8000;
	else
	    ((short *) p)[0] &= 0x7FFF;
    }
}


/* Kind 16 big endian-- */

static inline int unpack_sign_16(void *v) {
unsigned *p;

    p = v;
    return !!(p[0] & 0x80000000);
}


static inline int unpack_exponent_16(void *v) {
unsigned *p;

    p = v;
    return (p[0] >> 16) & 0x7FFF;
}


void unpack_real_16(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;
int e;

    p = v;
    mantissa[3] = p[3];
    mantissa[2] = p[2];
    mantissa[1] = p[1];
    mantissa[0] = p[0] & 0x0000FFFF;

    e = unpack_exponent_16(p);
    if (e != 0 && e != EXP16_NAN)
	mantissa[0] |= 0x00010000;

    *exp = e;
    *sign = unpack_sign_16(p);
}


void pack_real_16(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned *p;

    p = v;
    if (mantissa != NULL) {
	p[3] = mantissa[3];
	p[2] = mantissa[2];
	p[1] = mantissa[1];
	p[0] = (p[0] & 0xFFFF0000) | (mantissa[0] & 0x0000FFFF);
    }

    if (exp != NULL)
	p[0] = (p[0] & 0x8000FFFF) | ((*exp & 0x7FFF) << 16);

    if (sign != NULL) {
	if (*sign)
	    p[0] |= 0x80000000;
	else
	    p[0] &= 0x7FFFFFFF;
    }
}



#else
#error Unknown floating point unit!
#endif


void build_nan(int sign, int mantissa, void *p, int kind) {
unsigned m[4];
int e;

    switch(kind) {
    case 4:
	m[0] = mantissa;
	e = EXP4_NAN;
	pack_real_4(p, m, &e, &sign);
	break;

    case 8:
	m[0] = m[1] = mantissa;
	e = EXP8_NAN;
	pack_real_8(p, m, &e, &sign);
	break;

    case 10:
	m[0] = m[1] = mantissa;
	e = EXP10_NAN;
	pack_real_10(p, m, &e, &sign);
	break;

    case 16:
	m[0] = m[1] = m[2] = m[3] = mantissa;
	e = EXP16_NAN;
	pack_real_16(p, m, &e, &sign);
	break;

    default:
	internal_error("build_nan(): Bad kind");
	break;
    }
}


void build_infinity(int sign, void *p, int kind) {

    build_nan(sign, 0, p, kind);
}



#define exponent_4 prefix(exponent_4)

G95_DINT exponent_4(unsigned *x) {
unsigned m[1];
int e, i;

    unpack_real_4(x, m, &e, &i);

    if (e != 0)
	return e - EXP4_BIAS + 1;

    /* Denormal, look at the mantissa */

    e = 2-EXP4_BIAS;

    if (m[0] != 0) {
	while((m[0] & MAN4_MSW) == 0) {
	    m[0] <<= 1;
	    e--;
	}

    } else
	e = 0;   /* x = 0 */

    return e;
}



#define exponent_8 prefix(exponent_8)

G95_DINT exponent_8(unsigned *x) {
unsigned m[2];
int e, i;

    unpack_real_8(x, m, &e, &i);

    if (e != 0)
	return e - EXP8_BIAS + 1;

    /* Denormal, look at the mantissa */

    e = 2 - EXP8_BIAS;

    if (m[0] != 0) {
	while((m[0] & MAN8_MSW) == 0) {
	    m[0] <<= 1;
	    e--;
	}

    } else if (m[1] != 0) {
	e -= 21;
	while((m[1] & 0x80000000) == 0) {
	    m[1] <<= 1;
	    e--;
	}

    } else
	e = 0;

    return e;
}



#define exponent_10 prefix(exponent_10)

G95_DINT exponent_10(unsigned *x) {
unsigned m[2];
int e, s;

    unpack_real_10(x, m, &e, &s);

    if (e != 0)
	return e - EXP10_BIAS + 1;

    /* Denormal, look at the mantissa */

    e = 2 - EXP10_BIAS;

    if (m[0] != 0) {
	while((m[0] & 0x80000000) == 0) {
	    m[0] <<= 1;
	    e--;
	}

    } else if (m[1] != 0) {
	e -= 32;

	while((m[1] & 0x80000000) == 0) {
	    m[1] <<= 1; 
	    e--;
	}

    } else
	e = 0;

    return e;
}





#define set_exponent_4 prefix(set_exponent_4)

float set_exponent_4(float *x, G95_DINT *i) {
unsigned m[1];
int e, s;
float r;

    unpack_real_4(x, m, &e, &s);

    if (m[0] == 0)
	return *x;     /* x = 0 */

    e = *i + EXP4_BIAS - 1;

    if (e < -MAN4_LEN) {
	m[0] = 0;
	e = 0;

    } else if (e <= 0) {
	m[0] >>= -e+1;
	e = 0;
    }

    pack_real_4(&r, m, &e, &s);
    return r;
}



#define set_exponent_8 prefix(set_exponent_8)

double set_exponent_8(double *x, G95_DINT *i) {
int e, s, shifts;
unsigned mask, m[2];
double r;

    unpack_real_8(x, m, &e, &s);

    if (m[0] == 0 && m[1] == 0)
	return *x;     /* x = 0 */

    e = *i + EXP8_BIAS - 1;

    if (e < -MAN8_LEN) {
	m[0] = m[1] = 0;
	e = 0;

    } else if (e <= 0) {
	shifts = -e+1;

	while(shifts >= 32) {
	    m[1] = m[0];
	    m[0] = 0;
	    shifts -= 32;
	}

	mask = ~((~0) << shifts);

	m[1] = (m[1] >> shifts) | ((m[0] & mask) << (32-shifts));
	m[0] = m[0] >> shifts;
	e = 0;
    }

    pack_real_8(&r, m, &e, &s);
    return r;
}



#if HAVE_REAL_10
#define set_exponent_10 prefix(set_exponent_10)

void set_exponent_10(unsigned *x, G95_DINT *i) {
int exp;

    exp = *i - 1;

    asm("fild %0\n"        /* i */
	"fldt %1\n"        /* x : i */
	"fxtract\n"        /* sig(x) : exp(x) : i */
	"fstp %%st(1)\n"   /* sig(x) : i */
	"fscale\n"         /* sig(x)*2^i : i */
	"fstp %%st(1)\n"   /* sig(x)*2^i */
	: : "m" (exp), "m" (*x));
}

#endif



#define fraction_4 prefix(fraction_4)

float fraction_4(unsigned *x) {
unsigned m[1];
int e, s;
float r;

    unpack_real_4(x, m, &e, &s);

    if (m[0] != 0) {
	if (e == 0)
	    while((m[0] & MAN4_MSW) == 0)
		m[0] <<= 1;

	e = EXP4_BIAS - 1;
    }

    pack_real_4(&r, m, &e, &s);
    return r;
}



#define fraction_8 prefix(fraction_8)

double fraction_8(unsigned *x) {
unsigned m[2];
int e, s;
double r;

    unpack_real_8(x, m, &e, &s);

    if (m[0] != 0 || m[1] != 0) {
	if (e == 0)
	    while((m[0] & MAN8_MSW) == 0) {
		m[0] = (m[0] << 1) | (m[1] >> 31);
		m[1] = m[1] << 1;
	    }

	e = EXP8_BIAS - 1;
    }

    pack_real_8(&r, m, &e, &s);
    return r;
}



#if HAVE_REAL_10

#define fraction_10 prefix(fraction_10)

void fraction_10(unsigned *x) {

    asm("fldt %0\n"
	"fxtract\n"
	"fstp %%st(1)\n"
	"fld1\n"
	"fchs\n"
	"fxch %%st(1)\n"
	"fscale\n"
	"fstp %%st(1)\n" : : "m" (*x));
}

#endif



#define scale_4 prefix(scale_4)

float scale_4(unsigned *x, G95_DINT *i) {
unsigned m[1];
int e, s, n;
float r;

    unpack_real_4(x, m, &e, &s);

    if (e == 0) {    /* Denormalized */
	n = *i;

	if (n < -MAN4_LEN)
	    m[0] = 0;

	else if (n < 0)
	    m[0] >>= -n;

	else
	    while(n > 0) {
		m[0] = m[0] << 1;
		n--;

		if (m[0] & MAN4_MSW) {
		    e = n + 1;
		    break;
		}
	    }

    } else if (m[0] != 0)
	e += *i;

    pack_real_4(&r, m, &e, &s);
    return r;
}



#define scale_8 prefix(scale_8)

double scale_8(unsigned *x, G95_DINT *i) {
unsigned m[2];
int e, s, n;
double r;

    unpack_real_8(x, m, &e, &s);

    if (e == 0) {   /* Denormalized */
	n = *i;

	if (n < -MAN8_LEN)
	    m[0] = m[1] = 0;

	else if (n < 0) {
	    while(n < 0) {
		m[1] = (m[1] >> 1) | ((m[0] << 31) & (1U << 31));
		m[0] = m[0] >> 1;
		n++;
	    }

	} else
	    while (n > 0) {
		m[0] = (m[0] << 1) | ((m[1] >> 31) & 1);
		m[1] = m[1] << 1;

		n--;

		if (m[0] & MAN8_MSW) {
		    e = n + 1;
		    break;
		}
	    }

    } else if (m[0] != 0 || m[1] != 0)
	e += *i;

    pack_real_8(&r, m, &e, &s);
    return r;
}


#if HAVE_REAL_10

#define scale_10 prefix(scale_10)

void scale_10(unsigned *x, G95_DINT *i) {
int exp;

    exp = *i;

    asm("fild %0\n"
	"fldt %1\n"
	"fscale\n"
	"fstp %%st(1)\n" : : "m" (exp), "m" (*x));
}

#endif


#define rrspacing_4 prefix(rrspacing_4)

float rrspacing_4(unsigned *x) {
unsigned m[1];
int s, e;
float r;

    switch(get_float_flavor(x, 4, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	r = 0.0;
	break;

    case FF_NAN:
	r = *((float *) x);
	break;

    case FF_REGULAR:
	unpack_real_4(x, m, &e, &s);
	s = 0;

	if (m[0] != 0) {
	    e = EXP4_BIAS + MAN4_LEN;

	    /* If denormal, normalize it */

	    while((m[0] & MAN4_MSW) == 0)
		m[0] <<= 1;
	}

	pack_real_4(&r, m, &e, &s);
	break;
    }

    return r;
}



#define rrspacing_8 prefix(rrspacing_8)

double rrspacing_8(unsigned *x) {
unsigned m[2];
int s, e;
double r;

    switch(get_float_flavor(x, 8, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	r = 0.0;
	break;

    case FF_NAN:
	r = *((double *) x);
	break;

    case FF_REGULAR:
	unpack_real_8(x, m, &e, &s);
	s = 0;

	if (m[0] != 0 || m[1] != 0) {
	    e = EXP8_BIAS + MAN8_LEN;

	    /* If denormal, normalize it */

	    while((m[0] & MAN8_MSW) == 0) {
		m[0] = (m[0] << 1) | ((m[1] >> 31) & 1);
		m[1] <<= 1;
	    }
	}

	pack_real_8(&r, m, &e, &s);
	break;
    }

    return r;
}


#if HAVE_REAL_10
#define rrspacing_10 prefix(rrspacing_10)

void rrspacing_10(unsigned *x) {
int exp;

    switch(get_float_flavor(x, 10, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	asm("fldz\n");
	break;

    case FF_NAN:
	asm("mov %0, %" EAX "\n"
	    "fldt (%" EAX ")\n" : : "m" (x));
	break;

    case FF_REGULAR:
	exp = MAN10_LEN - 1;

	asm("fild %0\n"             /* p */
	    "fldt %1\n"             /* x . p */
	    "fxam\n"
	    "fnstsw\n"
	    "and $0x45, %%ah\n"
	    "cmp $0x40, %%ah\n"
	    "je 0f\n"              /* TOS is zero */
	    "fxtract\n"            /* x . e . p */
	    "fstp %%st(1)\n"       /* x . p */
	    "fscale\n"
	    "0: fstp %%st(1)\n"
	    "fabs\n" : : "m" (exp), "m" (*x));
	break;
    }
}


#endif



#define spacing_4 prefix(spacing_4)

float spacing_4(unsigned *x) {
unsigned m[1];
int e, s;
float r;

    unpack_real_4(x, m, &e, &s);

    switch(get_float_flavor(x, 4, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	s = 0;
	break;

    case FF_NAN:
	break;

    case FF_REGULAR:
	s = 0;

	if ((m[0] & MAN4_MSW) == 0)  /* Build TINY(x) */
	    e = 1;

	else {
	    e -= MAN4_LEN;
	    if (e <= 0)
		e = 1;
	}

	s = 0;
	m[0] = MAN4_MSW;
    }

    pack_real_4(&r, m, &e, &s);
    return r;
}


#define spacing_8 prefix(spacing_8)

double spacing_8(unsigned *x) {
unsigned m[2];
double r;
int e, s;

    unpack_real_8(x, m, &e, &s);

    switch(get_float_flavor(x, 8, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	s = 0;
	break;

    case FF_NAN:
	break;

    case FF_REGULAR:
	s = 0;

	if ((m[0] & MAN8_MSW) == 0)  /* Build TINY(x) */
	    e = 1;

	else {
	    e -= MAN8_LEN;
	    if (e <= 0)
		e = 1;
	}

	m[0] = MAN8_MSW;
	m[1] = 0;
    }

    pack_real_8(&r, m, &e, &s);
    return r;
}



#if HAVE_REAL_10

#define spacing_10 prefix(spacing_10)

void spacing_10(unsigned *x) {
unsigned m[2], result[3];
int e, s;

    unpack_real_10(x, m, &e, &s);

    switch(get_float_flavor(x, 10, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
	s = 0;
	break;

    case FF_NAN:
	break;

    case FF_REGULAR:
	s = 0;

	if (m[0] == 0 && m[1] == 0)  /* Build TINY(x) */
	    e = 1;

	else {
	    e -= MAN10_LEN-1;
	    if (e <= 0)
		e = 1;
	}

	m[0] = 0x80000000;
	m[1] = 0;
	break;
    }

    pack_real_10(result, m, &e, &s);

    asm("fldt %0\n" : : "m" (*(&result[0])));
}

#endif



#define sign_r4 prefix(sign_r4)

float sign_r4(float *a, float *b) {
int sign;
float r;

    r = *a;
    sign = unpack_sign_4((unsigned *) b);
    pack_real_4(&r, NULL, NULL, &sign);

    return r;
}



#define sign_r8 prefix(sign_r8)

double sign_r8(double *a, double *b) {
int sign;
double r;

    r = *a;
    sign = unpack_sign_8((unsigned *) b);
    pack_real_8(&r, NULL, NULL, &sign);

    return r;
}



#if HAVE_REAL_10
#define sign_r10 prefix(sign_r10)

void sign_r10(unsigned *a, unsigned *b) {
unsigned result[3];
int sign;

    memcpy(result, a, 3*sizeof(unsigned));
    sign = unpack_sign_10(b);
    pack_real_10(result, NULL, NULL, &sign);

    asm("fldt %0\n" : : "m" (*(&result[0])));
}

#endif


#define huge_4 prefix(huge_4)

float huge_4(void) {
unsigned m[1];
int e, s;
float r;

    m[0] = ~0;
    s = 0;
    e = EXP4_NAN - 1;

    pack_real_4(&r, m, &e, &s);
    return r;
}



#define huge_8 prefix(huge_8)

double huge_8(void) {
unsigned m[2];
int e, s;
double r;

    m[0] = m[1] = ~0;
    s = 0;
    e = EXP8_NAN - 1;

    pack_real_8(&r, m, &e, &s);
    return r;
}


#if HAVE_REAL_10

void huge_10(void) {
unsigned m[2], result[3];
int e, s;

    m[0] = m[1] = ~0;
    s = 0;
    e = EXP10_NAN - 1;

    pack_real_10(result, m, &e, &s);
    asm("fldt %0\n" : : "m" (*(&result[0])));
}

#endif


/* get_sign()-- Return the sign bit of a floating point number. */

int get_sign(void *p, int kind) {

    switch(kind) {
    case  4:  return unpack_sign_4(p);
    case  8:  return unpack_sign_8(p);
    case 10:  return unpack_sign_10(p);
    case 16:  return unpack_sign_16(p);

    default:
	internal_error("get_sign(): Bad kind");
    }

    return 0;
}



/* get_float_flavor()-- Determine what kind of a floating point
 * number we have. */

float_flavor get_float_flavor(void *p, int kind, char *repr) {
unsigned m[4];
int e, s;

    switch(kind) {
    case 4:
	unpack_real_4(p, m, &e, &s);
	if (e != EXP4_NAN)
	    goto regular;
    
	if (m[0] != 0)
	    goto nan;

	break;

    case 8:
	unpack_real_8(p, m, &e, &s);
	if (e != EXP8_NAN)
	    goto regular;

	if (m[0] != 0 || m[1] != 0)
	    goto nan;

	break;

    case 10:
	unpack_real_10(p, m, &e, &s);
	if (e != EXP10_NAN)
	    goto regular;

	if (m[0] != 2147483648U || m[1] != 0)
	    goto nan;

	break;

    case 16:
	unpack_real_16(p, m, &e, &s);
	if (e != EXP16_NAN)
	    goto regular;

	if (m[0] != 0 || m[1] != 0 || m[2] != 0 || m[3] != 0)
	    goto nan;

	break;

    default:
	internal_error("get_float_flavor(): Bad kind");
	break;
    }

    if (s) {
	if (repr != NULL)
	    strcpy(repr, "-Inf");

	return FF_MINUS_INFINITY;
    }

    if (repr != NULL)
	strcpy(repr, "+Inf");

    return FF_PLUS_INFINITY;

regular:
    if (repr != NULL)
	*repr = '\0';

    return FF_REGULAR;

nan:
    if (repr != NULL)
	strcpy(repr, "NaN");

    return FF_NAN;
}



#define isnan_4 prefix(isnan_4)
G95_DINT isnan_4(float *x) {

    return get_float_flavor(x, 4, NULL) == FF_NAN;
}



#define isnan_8 prefix(isnan_8)
G95_DINT isnan_8(double *x) {

    return get_float_flavor(x, 8, NULL) == FF_NAN;
}


#define isnan_10 prefix(isnan_10)
G95_DINT isnan_10(unsigned *x) {

    return get_float_flavor(x, 10, NULL) == FF_NAN;
}



#define is_negative_4 prefix(is_negative_4)
G95_DINT is_negative_4(unsigned *x) {

    return unpack_sign_4(x);
}

#define is_negative_8 prefix(is_negative_8)
G95_DINT is_negative_8(unsigned *x) {

    return unpack_sign_8(x);
}


#if HAVE_REAL_10
#define is_negative_10 prefix(is_negative_10)
G95_DINT is_negative_10(unsigned *x) {

    return unpack_sign_10(x);
}
#endif



#define is_finite_4 prefix(is_finite_4)
G95_DINT is_finite_4(unsigned *x) {

    return (get_float_flavor(x, 4, NULL) == FF_REGULAR);
}


#define is_finite_8 prefix(is_finite_8)
G95_DINT is_finite_8(unsigned *x) {

    return (get_float_flavor(x, 8, NULL) == FF_REGULAR);
}


#if HAVE_REAL_10
#define is_finite_10 prefix(is_finite_10)
G95_DINT is_finite_10(unsigned *x) {

    return (get_float_flavor(x, 10, NULL) == FF_REGULAR);
}
#endif



#define is_normal_4 prefix(is_normal_4)
G95_DINT is_normal_4(unsigned *x) {

    return (get_float_flavor(x, 4, NULL) == FF_REGULAR &&
	    unpack_exponent_4(x) != 0);
}


#define is_normal_8 prefix(is_normal_8)
G95_DINT is_normal_8(unsigned *x) {

    return (get_float_flavor(x, 8, NULL) == FF_REGULAR &&
	    unpack_exponent_8(x) != 0);
}


#if HAVE_REAL_10
#define is_normal_10 prefix(is_normal_10)
G95_DINT is_normal_10(unsigned *x) {

    return (get_float_flavor(x, 10, NULL) == FF_REGULAR &&
	    unpack_exponent_10(x) != 0);
}
#endif





static float next_4(float *x) {
unsigned m[1];
int e, s;
float r;

    switch(get_float_flavor(x, 4, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
    case FF_NAN:
	return *x;

    case FF_REGULAR:
	break;
    }

    unpack_real_4(x, m, &e, &s);

    if (e == 0 && m[0] == MAN4_MSW-1) {
	e++;
	m[0]++;

    } else if (++m[0] >= 2*MAN4_MSW && ++e == EXP4_NAN)
	m[0] = 0;

    pack_real_4(&r, m, &e, &s);
    return r;
}



static float prev_4(float *x) {
unsigned m[1];
int e, s;
float r;

    switch(get_float_flavor(x, 4, NULL)) {
    case FF_NAN:
	return *x;

    case FF_PLUS_INFINITY:
	return huge_4();

    case FF_MINUS_INFINITY:
	return -huge_4();

    case FF_REGULAR:
	break;
    }

    unpack_real_4(x, m, &e, &s);

    if (m[0] == 0)
	return -next_4(x);

    if (m[0]-- == MAN4_MSW)
	e--;

    pack_real_4(&r, m, &e, &s);
    return r;
}



#define nearest_4_4 prefix(nearest_4_4)

float nearest_4_4(float *x, float *s) {

    return (unpack_sign_4((unsigned *) x) ^ unpack_sign_4((unsigned *) s))
	? prev_4(x)
	: next_4(x);
}



#define nearest_4_8 prefix(nearest_4_8)

float nearest_4_8(float *x, double *s) {

    return (unpack_sign_4((unsigned *) x) ^ unpack_sign_8((unsigned *) s))
	? prev_4(x)
	: next_4(x);
}



static double next_8(double *x) {
unsigned m[2];
int e, s;
double r;

    switch(get_float_flavor(x, 8, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
    case FF_NAN:
	return *x;

    case FF_REGULAR:
	break;
    }

    unpack_real_8(x, m, &e, &s);

    if (e == 0 && m[0] == MAN8_MSW-1 && m[1] == ~0) {
	e++;
	m[0]++;
	m[1]++;

    } else if (++m[1] == 0 && ++m[0] >= 2*MAN8_MSW && ++e == EXP4_NAN)
	m[0] = m[1] = 0;

    pack_real_8(&r, m, &e, &s);
    return r;
}



static double prev_8(double *x) {
unsigned m[2];
int e, s;
double r;

    switch(get_float_flavor(x, 8, NULL)) {
    case FF_NAN:
	return *x;

    case FF_PLUS_INFINITY:
	return huge_8();

    case FF_MINUS_INFINITY:
	return -huge_8();

    case FF_REGULAR:
	break;
    }

    unpack_real_8(x, m, &e, &s);

    if (m[0] == 0 && m[1] == 0)
	return -next_8(x);

    if (m[1]-- == 0 && m[0]-- == MAN8_MSW && --e < 0) {
	e = 0;
	m[0] = m[1] = 0;
    }

    pack_real_8(&r, m, &e, &s);
    return r;
}



#define nearest_8_4 prefix(nearest_8_4)

double nearest_8_4(double *x, float *s) {

    return (unpack_sign_8(x) ^ unpack_sign_4(s))
	? prev_8(x)
	: next_8(x);
}



#define nearest_8_8 prefix(nearest_8_8)

double nearest_8_8(double *x, double *s) {

    return (unpack_sign_8(x) ^ unpack_sign_8(s))
	? prev_8(x)
	: next_8(x);
}



#if HAVE_REAL_10

static void next_10(unsigned *x) {
unsigned m[2], result[3];
int e, s;

    switch(get_float_flavor(x, 10, NULL)) {
    case FF_PLUS_INFINITY:
    case FF_MINUS_INFINITY:
    case FF_NAN:
	memcpy(result, x, 3*sizeof(unsigned));
	goto done;

    case FF_REGULAR:
	break;
    }

    unpack_real_10(x, m, &e, &s);

    if (e == 0 && m[0] == ~0 && m[1] == ~0) {
	e++;
	m[0] = 0x80000000;
	m[1] = 0;

    } else if (++m[1] == 0 && ++m[0] == 0) {
	e++;
	m[0] = 0x80000000;
    }

    pack_real_10(result, m, &e, &s);

done:
    asm("fldt %0\n" : : "m" (*(&result[0])));
}



static void prev_10(unsigned *x) {
unsigned m[2], result[3];
int e, s;

    switch(get_float_flavor(x, 10, NULL)) {
    case FF_NAN:
	memcpy(result, x, 3*sizeof(unsigned));
	goto done;

    case FF_PLUS_INFINITY:
	huge_10();
	return;

    case FF_MINUS_INFINITY:
	huge_10();
	asm("fchs\n");
	return;

    case FF_REGULAR:
	break;
    }

    unpack_real_10(x, m, &e, &s);

    if (m[0] == 0 && m[1] == 0) {
	next_10(x);
	asm("fchs\n");
	return;
    }

    if (m[1]-- == 0 && m[0]-- == 0x80000000) {
	if (--e < 0) {
	    e = 0;
	    m[0] = m[1] = 0;

	} else if (e > 0)
	    m[0] = 0xFFFFFFFF;
    }

    pack_real_10(result, m, &e, &s);

done:
    asm("fldt %0\n" : : "m" (*(&result[0])));
}



#define nearest_4_10 prefix(nearest_4_10)

float nearest_4_10(float *x, unsigned *s) {

    return (unpack_sign_4((unsigned *) x) ^ unpack_sign_10((unsigned *) s))
	? prev_4(x)
	: next_4(x);
}


#define nearest_8_10 prefix(nearest_8_10)

double nearest_8_10(double *x, unsigned *s) {

    return (unpack_sign_8((unsigned *) x) ^ unpack_sign_10((unsigned *) s))
	? prev_8(x)
	: next_8(x);
}


#define nearest_10_4 prefix(nearest_10_4)

void nearest_10_4(unsigned *x, float *s) {

    (unpack_sign_10((unsigned *) x) ^ unpack_sign_4((unsigned *) s))
	? prev_10(x)
	: next_10(x);
}


#define nearest_10_8 prefix(nearest_10_8)

void nearest_10_8(unsigned *x, double *s) {

    (unpack_sign_10((unsigned *) x) ^ unpack_sign_8((unsigned *) s))
	? prev_10(x)
	: next_10(x);
}


#define nearest_10_10 prefix(nearest_10_10)

void nearest_10_10(unsigned *x, double *s) {

    (unpack_sign_10((unsigned *) x) ^ unpack_sign_10((unsigned *) s))
	? prev_10(x)
	: next_10(x);
}

#endif



#if HAVE_REAL_10

void trunc_4(float *x) {
short cw, newcw;

    newcw = 0x0f7f;
    asm("fnstcw %0\n"
	"fldcw %1\n"
	"flds %2\n"
	"frndint\n"
	"fstps %2\n"
	"fldcw %0\n" : : "m" (cw), "m" (newcw), "m" (*x));
}

void trunc_8(double *x) {
short cw, newcw;

    newcw = 0x0f7f;
    asm("fnstcw %0\n"
	"fldcw %1\n"
	"fldl %2\n"
	"frndint\n"
	"fstpl %2\n"
	"fldcw %0\n" : : "m" (cw), "m" (newcw), "m" (*x));
}

#else

void trunc_4(float *x) {
unsigned m[1];
int exp, e, s;

    unpack_real_4(x, m, &e, &s);

    if (e == EXP4_NAN)
	return;

    exp = e - EXP4_BIAS + 1;

    if (exp < 1) {
	e = 0;
	m[0] = 0;

    } else if (exp < MAN4_LEN+1)
	m[0] &= (~0) << (MAN4_LEN - exp + 1);

    else
	return;

    pack_real_4(x, m, &e, &s);
}



void trunc_8(double *x) {
int n, exp, e, s;
unsigned m[2];

    unpack_real_8(x, m, &e, &s);

    if (e == EXP8_NAN)
	return;

    exp = e - EXP8_BIAS + 1;

    if (exp < 1) {
	e = 0;
	m[0] = m[1] = 0;

    } else if (exp < MAN8_LEN+1) {
	n = (MAN8_LEN - exp + 1);

	if (n <= 32)
	    m[1] &= (~0) << n;

	else {
	    m[1] = 0;
	    m[0] &= (~0) << (n-32);
	}

    } else
	return;

    pack_real_8(x, m, &e, &s);
}

#endif


#if HAVE_REAL_10

void round_4(float *x) {
short cw, newcw;

    newcw = 0x0f7f;
    asm("fnstcw %0\n"
	"fldcw %1\n"
	"flds %2\n"
	"flds %3\n"
	"fxch %%st(1)\n"
	"fxam\n"
	"fnstsw %%ax\n"
	"test $0x200, %%eax\n"
	"je 0f\n"
	"fxch %%st(1)\n"
	"fchs\n"
	"0: faddp\n"
	"frndint\n"
	"fstps %2\n"
	"fldcw %0\n" : : "m" (cw), "m" (newcw), "m" (*x), "m" (half) : "%eax");
}

void round_8(double *x) {
short cw, newcw;

    newcw = 0x0f7f;
    asm("fnstcw %0\n"
	"fldcw %1\n"
	"fldl %2\n"
	"flds %3\n"
	"fxch %%st(1)\n"
	"fxam\n"
	"fnstsw %%ax\n"
	"test $0x200, %%eax\n"
	"je 0f\n"
	"fxch %%st(1)\n"
	"fchs\n"
	"0: faddp\n"
	"frndint\n"
	"fstpl %2\n"
	"fldcw %0\n" : : "m" (cw), "m" (newcw), "m" (*x), "m" (half) : "%eax");
}


#else


void round_4(float *x) {
int b, n, exp, e, s;
unsigned m[1];

    unpack_real_4(x, m, &e, &s);

    if (e == EXP4_NAN)
	return;

    exp = e - EXP4_BIAS + 1;

    if (exp < 0) {
	e = 0;
	m[0] = 0;

    } else if (exp == 0) {  /* Special case, 0.5 <= |x| < 1.0 */
	m[0] = MAN4_MSW;
	e++;

    } else if (exp < MAN4_LEN+1) {
	n = MAN4_LEN - exp + 1;     /* Bits to get rid of */
	b = m[0] & (~((~0) << n));

	if (b >= (1 << (n-1)))
	    b = 1 << n;

	else
	    b = 0;

	m[0] = (m[0] & ((~0) << n)) + b;

	if (m[0] >= 2*MAN4_MSW)
	    e++;   /* Can't be infinity */

    } else
	return;

    pack_real_4(x, m, &e, &s);
}



void round_8(double *x) {
int b, n, exp, e, s;
unsigned m[2];

    unpack_real_8(x, m, &e, &s);

    if (e == EXP8_NAN)
	return;

    exp = e - EXP8_BIAS + 1;

    if (exp < 0) {
	e = 0;
	m[0] = m[1] = 0;

    } else if (exp == 0) {  /* Special case, 0.5 <= |x| < 1.0 */
	e++;
	m[0] = MAN8_MSW;
	m[1] = 0;

    } else if (exp < MAN8_LEN+1) {
	n = MAN8_LEN - exp + 1;     /* Bits to get rid of */

	if (n < 32) {
	    b = m[1] & (~((~0) << n));

	    if (b >= (1 << (n-1)))
		b = 1 << n;

	    else
		b = 0;

	    m[1] = (m[1] & ((~0) << n)) + b;
	    if (m[1] < b && ++m[0] >= 2*MAN8_MSW)
		e++;   /* Can't be infinity */

	} else if (n > 32) {
	    n = n - 32;
 
	    b = m[0] & (~((~0) << n));

	    if (b >= (1 << (n-1)))
		b = 1 << n;

	    else
		b = 0;

	    m[0] = (m[0] & ((~0) << n)) + b;
	    m[1] = 0;

	    if (m[0] >= 2*MAN8_MSW)
		e++;   /* Can't be infinity */

	} else {    /* Bizzaro case, n == 32 */
	    if (m[1] == 0x8000000)
		b = m[0] & 1;

	    else
		b = (m[1] > 0x80000000);

	    m[1] = 0;
	    m[0] += b;

	    if (m[0] >= 2*MAN8_MSW)
		e++;
	}

    } else
	return;

    pack_real_8(x, m, &e, &s);
}

#endif


/* onebits()-- Returns the number of significant bits in the mantissa.
 * The specal case of zero is not important. */

static int onebits(unsigned m) {
int n;

    n = 0;
    while(m != 0) {
	n++;
	m = m >> 1;
    }

    return n;
}



/* unpack_real()-- Common code for unpacking a real number prior to
 * converting it to decimal.  Returns the number of words in the
 * mantissa array that are active and sets a lot of flags that give
 * information about the number. */

int unpack_real(void *p, int kind, unsigned *mantissa, int *exp,
		int *evenp, int *zerop, int *denormp, int *b1p, int *kp) {
int bits, e, sign, active;

    switch(kind) {
    case 4:
	unpack_real_4(p, mantissa, &e, &sign);

	*evenp = (mantissa[0] & 1) ^ 1;
	*zerop = (mantissa[0] == 0);
	*b1p = (mantissa[0] == MAN4_MSW);

	if (e == 0) {
	    *exp = 1 - EXP4_BIAS - MAN4_LEN;
	    *denormp = 1;
	    bits = onebits(mantissa[0]) - EXP4_BIAS - MAN4_LEN;

	} else {
	    *exp = e - EXP4_BIAS - MAN4_LEN;
	    *denormp = 0;
	    bits = e - EXP4_BIAS;
	}

	active = 1;
	break;

    case 8:
	unpack_real_8(p, mantissa, &e, &sign);

	*evenp = (mantissa[1] & 1) ^ 1;
	*zerop = (mantissa[0] == 0 && mantissa[1] == 0);
	*b1p = (mantissa[0] == MAN8_MSW && mantissa[1] == 0);

	if (e == 0) {
	    *exp = 1 - EXP8_BIAS - MAN8_LEN;
	    *denormp = 1;

	    if (mantissa[0] != 0)
		bits = onebits(mantissa[0]) + 32 - EXP8_BIAS - MAN8_LEN;
	    else
		bits = onebits(mantissa[1]) - EXP8_BIAS - MAN8_LEN;

	} else {
	    *exp = e - EXP8_BIAS - MAN8_LEN;
	    *denormp = 0;
	    bits = e - EXP8_BIAS;
	}

	active = 2;
	break;

    case 10:    
	unpack_real_10(p, mantissa, &e, &sign);

	*evenp = (mantissa[1] & 1) ^ 1;
	*zerop = (mantissa[0] == 0 && mantissa[1] == 0);
	*b1p = (mantissa[0] == 0x80000000 && mantissa[1] == 0);

	if (e == 0) {
	    *exp = 2 - EXP10_BIAS - MAN10_LEN;
	    *denormp = 1;

	    if (mantissa[0] != 0)
		bits = onebits(mantissa[0]) + 32 - EXP10_BIAS - MAN10_LEN;
	    else
		bits = onebits(mantissa[1]) - EXP10_BIAS - MAN10_LEN;

	} else {
	    *exp = e - EXP10_BIAS - MAN10_LEN + 1;
	    *denormp = 0;
	    bits = e - EXP10_BIAS;
	}

	active = 2;
	break;

    case 16:
#if HAVE_REAL_10 == 3
	asm("movaps %0, %%xmm0" : : "m" (*((char *) p)));
#endif
	unpack_real_16(p, mantissa, &e, &sign);

	*evenp = (mantissa[3] & 1) ^ 1;
	*zerop = (mantissa[0] == 0 && mantissa[1] == 0 &&
		  mantissa[2] == 0 && mantissa[3] == 0);
	*b1p = (mantissa[0] == MAN16_MSW && mantissa[1] == 0 &&
		mantissa[2] == 0         && mantissa[3] == 0);

	if (e == 0) {
	    *exp = 1 - EXP16_BIAS - MAN16_LEN;
	    *denormp = 1;

	    if (mantissa[0] != 0)
		bits = onebits(mantissa[0]) + 96 - EXP16_BIAS - MAN16_LEN;
	    else if (mantissa[1] != 0)
		bits = onebits(mantissa[1]) + 64 - EXP16_BIAS - MAN16_LEN;
	    else if (mantissa[2] != 0)
		bits = onebits(mantissa[2]) + 32 - EXP16_BIAS - MAN16_LEN;
	    else
		bits = onebits(mantissa[3] - EXP16_BIAS - MAN16_LEN);

	} else {
	    *exp = e - EXP16_BIAS - MAN16_LEN;
	    *denormp = 0;
	    bits = e - EXP16_BIAS;
	}

	active = 4;
	break;

    default:
	active = 0;
	bits = 0;
	internal_error("unpack_real(): Bad kind");
    }

    /* Estimate the scale factor.  Bob's original code was:
     *
     *   *kp = (bits < 0)
     *       ? (int)(bits*0.3010299956639812)
     *       : 1+(int)(bits*0.3010299956639811);
     *
     * To avoid fp instructions, we use integer scaling.  After a
     * search, it turns out that 8651/28738 is 0.3010299951284014197,
     * compared with the above value of log10(2.0), and gives the
     * correct result without overflow for abs(bits) < 70777. */

    *kp = (bits < 0)
	? (8651 * bits) / 28738
	: (8651 * bits) / 28738 + 1;

    return active;
}



#define class_4 prefix(class_4)

G95_DINT class_4(float *x) {
unsigned mantissa[1];
int t, sign, exp;

    unpack_real_4(x, mantissa, &exp, &sign);

    switch(get_float_flavor(x, 4, NULL)) {
    case FF_PLUS_INFINITY:
	t = CLASS_POSITIVE_INF;
	break;

    case FF_MINUS_INFINITY:
	t = CLASS_NEGATIVE_INF;
	break;

    case FF_NAN:
	t = (mantissa[0] & MAN4_MSW) ? CLASS_QUIET_NAN : CLASS_SIGNALING_NAN;
	break;

    case FF_REGULAR:
	if (exp != 0)
	    t = sign ? CLASS_NEGATIVE_NORMAL : CLASS_POSITIVE_NORMAL;

	else if (mantissa[0] != 0)
	    t = sign ? CLASS_NEGATIVE_DENORMAL : CLASS_POSITIVE_DENORMAL;

	else
	    t = sign ? CLASS_NEGATIVE_ZERO : CLASS_POSITIVE_ZERO;

	break;

    default:
	t = CLASS_OTHER_VALUE;
	break;
    }

    return t;
}



#define class_8 prefix(class_8)

G95_DINT class_8(double *x) {
unsigned mantissa[2];
int t, sign, exp;

    unpack_real_8(x, mantissa, &exp, &sign);

    switch(get_float_flavor(x, 8, NULL)) {
    case FF_PLUS_INFINITY:
	t = CLASS_POSITIVE_INF;
	break;

    case FF_MINUS_INFINITY:
	t = CLASS_NEGATIVE_INF;
	break;

    case FF_NAN:
	t = (mantissa[0] & MAN8_MSW) ? CLASS_QUIET_NAN : CLASS_SIGNALING_NAN;
	break;

    case FF_REGULAR:
	if (exp != 0)
	    t = sign ? CLASS_NEGATIVE_NORMAL : CLASS_POSITIVE_NORMAL;

	else if (mantissa[0] != 0 || mantissa[0] != 0)
	    t = sign ? CLASS_NEGATIVE_DENORMAL : CLASS_POSITIVE_DENORMAL;

	else
	    t = sign ? CLASS_NEGATIVE_ZERO : CLASS_POSITIVE_ZERO;

	break;

    default:
	t = CLASS_OTHER_VALUE;
	break;
    }

    return t;
}


#if HAVE_REAL_10
#define class_10 prefix(class_10)

G95_DINT class_10(void *x) {
unsigned mantissa[2];
int t, sign, exp;

    unpack_real_10(x, mantissa, &exp, &sign);

    switch(get_float_flavor(x, 10, NULL)) {
    case FF_PLUS_INFINITY:
	t = CLASS_POSITIVE_INF;
	break;

    case FF_MINUS_INFINITY:
	t = CLASS_NEGATIVE_INF;
	break;

    case FF_NAN:
	t = (mantissa[0] & 0x40000000) ? CLASS_QUIET_NAN : CLASS_SIGNALING_NAN;
	break;

    case FF_REGULAR:
	if (exp != 0)
	    t = sign ? CLASS_NEGATIVE_NORMAL : CLASS_POSITIVE_NORMAL;

	else if (mantissa[0] != 0 || mantissa[1] != 0)
	    t = sign ? CLASS_NEGATIVE_DENORMAL : CLASS_POSITIVE_DENORMAL;

	else
	    t = sign ? CLASS_NEGATIVE_ZERO : CLASS_POSITIVE_ZERO;

	break;

    default:
	t = CLASS_OTHER_VALUE;
	break;
    }

    return t;
}
#endif



#define value_4 prefix(value_4)

float value_4(void *x, G95_INT4 *type) {
unsigned m[1];
int e, s;
float r;

    switch(*type) {
    case CLASS_SIGNALING_NAN:
	m[0] = 1;
	e = EXP4_NAN;
	s = 0;
	break;

    case CLASS_QUIET_NAN:
	m[0] = ~0;
	e = EXP4_NAN;
	s = 0;
	break;

    case CLASS_NEGATIVE_INF:
	m[0] = 0;
	e = EXP4_NAN;
	s = 1;
	break;

    case CLASS_NEGATIVE_DENORMAL:
	m[0] = 1;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_ZERO:
	m[0] = 0;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_NORMAL:
	return -1.0;

    case CLASS_POSITIVE_INF:
	m[0] = 0;
	e = EXP4_NAN;
	s = 0;
	break;

    case CLASS_POSITIVE_DENORMAL:
	m[0] = 1;
	e = 0;
	s = 0;
	break;

    case CLASS_POSITIVE_ZERO:
	return 0.0;

    case CLASS_POSITIVE_NORMAL:
	return 1.0;

    default:
	runtime_error("Bad class value passed to IEEE_VALUE()");
    }

    pack_real_4(&r, m, &e, &s);
    return r;
}


#define value_8 prefix(value_8)

double value_8(void *x, G95_INT4 *type) {
unsigned m[2];
int e, s;
double r;

    switch(*type) {
    case CLASS_SIGNALING_NAN:
	m[0] = 0;
	m[1] = 1;
	e = EXP8_NAN;
	s = 0;
	break;

    case CLASS_QUIET_NAN:
	m[0] = m[1] = ~0;
	e = EXP8_NAN;
	s = 0;
	break;

    case CLASS_NEGATIVE_INF:
	m[0] = m[1] = 0;
	e = EXP8_NAN;
	s = 1;
	break;

    case CLASS_NEGATIVE_DENORMAL:
	m[0] = 0;
	m[1] = 1;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_ZERO:
	m[0] = m[1] = 0;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_NORMAL:
	return -1.0;

    case CLASS_POSITIVE_INF:
	m[0] = m[1] = 0;
	e = EXP8_NAN;
	s = 0;
	break;

    case CLASS_POSITIVE_DENORMAL:
	m[0] = 0;
	m[1] = 1;
	e = 0;
	s = 0;
	break;

    case CLASS_POSITIVE_ZERO:
	return 0.0;

    case CLASS_POSITIVE_NORMAL:
	return 1.0;

    default:
	runtime_error("Bad class value passed to IEEE_VALUE()");
    }

    pack_real_8(&r, m, &e, &s);
    return r;
}



#if HAVE_REAL_10
#define value_10 prefix(value_10)

void value_10(void *x, G95_INT4 *type) {
unsigned *p, value[4], m[2];
int e, s;

    switch(*type) {
    case CLASS_SIGNALING_NAN:
	m[0] = 0;
	m[1] = 1;
	e = EXP10_NAN;
	s = 0;
	break;

    case CLASS_QUIET_NAN:
	m[0] = m[1] = ~0;
	e = EXP10_NAN;
	s = 0;
	break;

    case CLASS_NEGATIVE_INF:
	m[0] = 0x80000000;
	m[1] = 0;
	e = EXP10_NAN;
	s = 1;
	break;

    case CLASS_NEGATIVE_DENORMAL:
	m[0] = 0;
	m[1] = 1;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_ZERO:
	m[0] = m[1] = 0;
	e = 0;
	s = 1;
	break;

    case CLASS_NEGATIVE_NORMAL:
	asm("fld1\n"
	    "fchs\n");
	return;

    case CLASS_POSITIVE_INF:
	m[0] = 0x80000000;
	m[1] = 0;
	e = EXP10_NAN;
	s = 0;
	break;

    case CLASS_POSITIVE_DENORMAL:
	m[0] = 0;
	m[1] = 1;
	e = 0;
	s = 0;
	break;

    case CLASS_POSITIVE_ZERO:
	asm("fldz\n");
	return;

    case CLASS_POSITIVE_NORMAL:
	asm("fld1\n");
	return;

    default:
	runtime_error("Bad class value passed to IEEE_VALUE()");
    }

    pack_real_10(value, m, &e, &s);

    p = &value[0];
    asm("mov %0, %" EAX "\n"
	"fldt (%" EAX ")\n" : : "m" (p));
}

#endif

