
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

/* Read integers */

#ifndef FIRST
#define FIRST 1

#include "runtime.h"

/* next_char()-- Read the next character in a decimal number. */

static char next_char(char **p, int *w) {
char c, *q;

    for(;;) {
	if (*w == 0)
	    return '\0';

	q = *p;
	c = *q++;
	*p = q;

	(*w)--;

	if (c != ' ')
	    break;

	if (ioparm->blank_status == BLANK_ZERO)
	    return '0';
    }

    return c;
}

#endif


#ifdef SUBROUTINE_RD

static void SUBROUTINE_RD(char *p, int w, TYPE *dest) {
unsigned TYPE value, maxv_10;
int negative;
char c;
TYPE v;

    maxv_10 = IMAX / 10;

    negative = 0;
    value = 0;

    switch(*p) {
    case '-':
	negative = 1;
	/* Fall through */

    case '+':
	p++;
	if (--w == 0)
	    goto bad;
	/* Fall through */

    default:
	break;
    }

    /* At this point we have a digit-string */
    value = 0;

    for(;;) {
	c = next_char(&p, &w);
	if (c == '\0')
	    break;

	if (c < '0' || c > '9')
	    goto bad;

	if (value > maxv_10)
	    goto overflow;

	c -= '0';
	value = 10*value;

	if (value > IMAX - c)
	    goto overflow;

	value += c;
    }

    v = (TYPE) value;
    if (negative)
	v = -v;

    *dest = v;
    return;

bad:
    generate_error(ERROR_READ_VALUE, "Bad value during integer read");
    return;

overflow:
    generate_error(ERROR_READ_OVERFLOW,
		   "Value overflowed during integer read");
    return;
}


/* read_radix_int()-- Read a nondecimal radix into an integer */

static void SUBROUTINE_RR(int radix, char *p, int w, TYPE *dest) {
unsigned TYPE value, maxv_r;
int negative;
TYPE v;
char c;

    maxv_r = UMAX / radix;

    negative = 0;
    value = 0;

    switch(*p) {
    case '-':
	negative = 1;
	/* Fall through */

    case '+':
	p++;
	if (--w == 0)
	    goto bad;
	/* Fall through */

    default:
	break;
    }

    /* At this point we have a digit-string */
    value = 0;

    for(;;) {
	c = next_char(&p, &w);
	if (c == '\0')
	    break;

	switch(radix) {
	case 2:
	    if (c < '0' || c > '1')
		goto bad;
	    break;

	case 8:
	    if (c < '0' || c > '7')
		goto bad;
	    break;

	case 16:
	    switch(c) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		break;

	    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
		c = c - 'a' + '9' + 1;
		break;

	    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
		c = c - 'A' + '9' + 1;
		break;

	    default:
		goto bad;
	    }

	    break;
	}

	if (value > maxv_r)
	    goto overflow;

	c -= '0';
	value = radix*value;

	if (UMAX - ((unsigned) c) < value)
	    goto overflow;

	value += c;
    }

    v = (TYPE) value;
    if (negative)
	v = -v;

    *dest = v;
    return;

bad:
    generate_error(ERROR_READ_VALUE, "Bad value during integer read");
    return;

overflow:
    generate_error(ERROR_READ_OVERFLOW,
		   "Value overflowed during integer read");
    return;
}


static int SUBROUTINE_RS(char *buffer, int negative, TYPE *dest) {
char c, message[100];
TYPE v;

    v = 0;

    for(;;) {
	c = *buffer++;
	if (c == '\0')
	    break;

	c -= '0';

	if (v > IMAX / 10)
	    goto overflow;

	v = 10*v;

	if (v > IMAX-c)
	    goto overflow;

	v += c;
    }

    if (negative)
	v = -v;

    *dest = v;
    return 0;

overflow:
    st_sprintf(message, "Integer overflow while reading item %d",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return 1;
}

#undef TYPE
#undef IMAX
#undef UMAX
#undef SUBROUTINE_RD
#undef SUBROUTINE_RR
#undef SUBROUTINE_RS

#else

#define TYPE G95_INT1
#define IMAX G95_INT1_MAX
#define UMAX G95_UINT1_MAX
#define SUBROUTINE_RD read_decimal_1
#define SUBROUTINE_RR read_radix_1
#define SUBROUTINE_RS read_list_1
#include "iread.c"

#define TYPE G95_INT2
#define IMAX G95_INT2_MAX
#define UMAX G95_UINT2_MAX
#define SUBROUTINE_RD read_decimal_2
#define SUBROUTINE_RR read_radix_2
#define SUBROUTINE_RS read_list_2
#include "iread.c"

#define TYPE G95_INT4
#define IMAX G95_INT4_MAX
#define UMAX G95_UINT4_MAX
#define SUBROUTINE_RD read_decimal_4
#define SUBROUTINE_RR read_radix_4
#define SUBROUTINE_RS read_list_4
#include "iread.c"

#ifdef G95_INT8
#define TYPE G95_INT8
#define IMAX G95_INT8_MAX
#define UMAX G95_UINT8_MAX
#define SUBROUTINE_RD read_decimal_8
#define SUBROUTINE_RR read_radix_8
#define SUBROUTINE_RS read_list_8
#include "iread.c"
#endif

#ifdef G95_INT16
#define TYPE G95_INT16
#define IMAX G95_INT16_MAX
#define UMAX G95_UINT16_MAX
#define SUBROUTINE_RD read_decimal_16
#define SUBROUTINE_RR read_radix_16
#define SUBROUTINE_RS read_list_16
#include "iread.c"
#endif

/* read_decimal()-- Read a decimal integer value.  The values here are
 * signed values. */

void read_decimal(fnode *f, void *dest, int length) {
unsigned w;
char *p;

    w = f->u.w;
    p = read_block(&w);
    if (p == NULL)
	return;

    p = eat_leading_spaces(&w, p);
    if (w == 0) {
	set_integer(0, dest, length);
	return;
    }

    switch(length) {
    case 1:   read_decimal_1(p, w, dest); break;
    case 2:   read_decimal_2(p, w, dest); break;
    case 4:   read_decimal_4(p, w, dest); break;

#ifdef G95_INT8
    case 8:   read_decimal_8(p, w, dest); break;
#endif

#ifdef G95_INT16
    case 16:  read_decimal_16(p, w, dest); break;
#endif

    default:
	internal_error("read_decimal(): Bad kind");
    }
}


/* read_radix()-- This function reads values for non-decimal radixes.
 * The difference here is that we treat the values here as unsigned
 * values for the purposes of overflow.  If minus sign is present and
 * the top bit is set, the value will be incorrect. */

void read_radix(fnode *f, void *dest, int length, int radix) {
unsigned w;
char *p; 

    w = f->u.w;
    p = read_block(&w);
    if (p == NULL)
	return;

    p = eat_leading_spaces(&w, p);
    if (w == 0) {
	set_integer(0, dest, length);
	return;
    }

    switch(length) {
    case 1:   read_radix_1(radix, p, w, dest);  break;
    case 2:   read_radix_2(radix, p, w, dest);  break;
    case 4:   read_radix_4(radix, p, w, dest);  break;

#ifdef G95_INT8
    case 8:   read_radix_8(radix, p, w, dest);  break;
#endif

#ifdef G95_INT16
    case 16:  read_radix_16(radix, p, w, dest); break;
#endif

    default:
	internal_error("read_decimal(): Bad kind");
    }
}


int read_list_integer(char *buffer, int negative, int len, void *dest) {
int m;

    switch(len) {
    case 1:  m = read_list_1(buffer, negative, dest);   break;
    case 2:  m = read_list_2(buffer, negative, dest);   break;
    case 4:  m = read_list_4(buffer, negative, dest);   break;

#ifdef G95_INT8
    case 8:  m = read_list_8(buffer, negative, dest);   break;
#endif

#ifdef G95_INT16
    case 16: m = read_list_16(buffer, negative, dest);  break;
#endif

    default:
	internal_error("read_list_integer(): Bad kind");
	m = 0;
    }

    return m;
}

#endif
