
/* All software  1996 Robert G. Burger.  Permission is hereby granted,
 * free of charge, to any person obtaining a copy of this software, to
 * deal in the software without restriction, including without
 * limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the software.
 *
 * The software is provided ``as is''; without warranty of any kind,
 * express or implied, including but not limited to the warranties of
 * merchantability, fitness for a particular purpose and
 * noninfringement.  In no event shall the author be liable for any
 * claim, damages or other liability, whether in an action of
 * contract, tort or otherwise, arising from, out of or in connection
 * with the software or the use or other dealings in the software.
 *
 * This code is described in an article by Robert G. Burger and
 * R. Kent Dybvig in "Proceedings of the ACM SIGPLAN '96 Conference on
 * Programming Language Design and Implementation", pages 108-116, May
 * 1996.
 *
 * Modified for use with g95 by Andy Vaught.  Thanks Bob!
 */


#include <string.h>
#include "runtime.h"


/* Floating point printer */

typedef unsigned long long Bigit;

#define BIGSIZE 190


typedef struct {
    int l;
    Bigit d[BIGSIZE];
} Bignum;



typedef struct {
    int exp;
    unsigned m[4];
} bigfloat;



#include "power5.h"
#include "cutoff.h"


#define OUTDIG(d) do { *buf++ = (d) + '0'; *buf = 0; return k; } while(0)

static Bignum R, S, MP, MM;
static Bignum S2, S3, S4, S5, S6, S7, S8, S9;
static int s_n, use_mp, qr_shift, sl, slr;

#define ADD(x, y, z, k) {  \
    Bigit x_add, z_add;    \
    x_add = (x);           \
    if ((k))               \
      z_add = x_add + (y) + 1, (k) = (z_add <= x_add);  \
    else  \
      z_add = x_add + (y), (k) = (z_add < x_add);  \
    (z) = z_add;   \
}

#define SUB(x, y, z, b) {      \
    Bigit x_sub, y_sub;        \
    x_sub = (x); y_sub = (y);  \
    if ((b))                   \
      (z) = x_sub - y_sub - 1, b = (y_sub >= x_sub); \
    else  \
      (z) = x_sub - y_sub, b = (y_sub > x_sub);  \
}

#define MUL(x, y, z, k) {    \
    Bigit x_mul, low, high;  \
    x_mul = (x);             \
    low = (x_mul & 0xffffffff) * (y) + (k);    \
    high = (x_mul >> 32) * (y) + (low >> 32);  \
    (k) = high >> 32;                          \
    (z) = (low & 0xffffffff) | (high << 32);   \
}

#define SLL(x, y, z, k) {         \
    Bigit x_sll = (x);            \
    (z) = (x_sll << (y)) | (k);   \
    (k) = x_sll >> (64 - (y));    \
}



/* mul10()-- Multiply a bignum by 10 */

#if 0 && HAVE_REAL_10 == 1

static inline void mul10(Bignum *x) {

/*  edi     points to x->l
 *  esi     points to word we are working on 
 *  cx      word counter
 *  bl      carry from last partial product
 *  bh:eax  accumulator */

    asm("push %%ebx          ;"
	"mov %0, %%edi       ;"
	"mov %1, %%esi       ;"
	"mov (%%edi), %%ecx  ;"
	"inc %%ecx           ;"
	"mov $0, %%ebx       ;"
	"add %%ecx, %%ecx    ;"

	"0: "
	"mov $10, %%edx      ;"
	"mov (%%esi), %%eax  ;"
	"mul %%edx           ;"

	"add %%ebx, %%eax    ;"
	"adc $0, %%edx       ;"
	"mov %%eax, (%%esi)  ;"

	"add $4, %%esi       ;"
	"dec %%ecx           ;"
	"mov %%edx, %%ebx    ;"
	"jne 0b              ;"

	"cmp $0, %%ebx       ;"
	"je 1f               ;"
	"mov %%ebx, (%%esi)  ;"
	"movl $0, 4(%%esi)   ;"
	"incl (%%edi)        ;"
	"1: pop %%ebx        ;" : : "g" (&x->l), "g" (&x->d[0])
	   : "%esi", "%edi", "%ecx", "%eax", "memory");

}

#else


static void mul10(Bignum *x) {
Bigit *p, k;
int i, l;

    l = x->l;
    for(i=l, p=&x->d[0], k=0; i>=0; i--)
	MUL(*p, 10, *p++, k);

    if (k != 0) {
	*p = k;
	x->l = l+1;
    }
}

#endif

static void big_short_mul(Bignum *x, Bigit y, Bignum *z) {
Bigit *xp, *zp, k;
int i, xl, zl;
unsigned high, low;

    xl = x->l;
    xp = &x->d[0];
    zl = xl;
    zp = &z->d[0];
    high = y >> 32;
    low = y & 0xffffffff;

    for(i=xl, k=0; i >= 0; i--, xp++, zp++) {
	Bigit xlow, xhigh, z0, t, c, z1;

	xlow = *xp & 0xffffffff;
	xhigh = *xp >> 32;
	z0 = (xlow * low) + k; /* Cout is (z0 < k) */
	t = xhigh * low;
	z1 = (xlow * high) + t;
	c = (z1 < t);
	t = z0 >> 32;
	z1 += t;
	c += (z1 < t);
	*zp = (z1 << 32) | (z0 & 0xffffffff);
	k = (xhigh * high) + (c << 32) + (z1 >> 32) + (z0 < k);
    }

    if (k != 0)
	*zp = k, zl++;

    z->l = zl;
}


#if 0
static print_big(Bignum *x) {
    int i;
    Bigit *p;

    printf("#x");
    i = x->l;
    p = &x->d[i];
    for (p = &x->d[i]; i >= 0; i--) {
	Bigit b = *p--;
	printf("%08x%08x", (int)(b >> 32), (int)(b & 0xffffffff));
    }
}
#endif



static void one_shift_left(int y, Bignum *z) {
int n, m, i;
Bigit *zp;

    n = y / 64;
    m = y % 64;
    zp = &z->d[0];

    for(i=n; i>0; i--)
	*zp++ = 0;

    *zp = (Bigit) 1 << m;
    z->l = n;
}


static void short_shift_left(Bigit x, int y, Bignum *z) {
int n, m, i, zl;
Bigit *zp;
   
    n = y / 64;
    m = y % 64;
    zl = n;
    zp = &(z->d[0]);

    for(i=n; i>0; i--)
	*zp++ = 0;

    if (m == 0)
	*zp = x;

    else {
	Bigit high = x >> (64 - m);
	*zp = x << m;

	if (high != 0)
	    *++zp = high, zl++;
    }

    z->l = zl;
}


static void big_shift_left(Bignum *x, int y, Bignum *z) {
int n, m, i, xl, zl;
Bigit *xp, *zp, k;
   
    n = y / 64;
    m = y % 64;
    xl = x->l;
    xp = &(x->d[0]);
    zl = xl + n;
    zp = &(z->d[0]);

    for (i = n; i > 0; i--)
	*zp++ = 0;

    if (m == 0)
	for(i=xl; i>=0; i--)
	    *zp++ = *xp++;
    else {
	for(i=xl, k=0; i>=0; i--)
	    SLL(*xp++, m, *zp++, k);

	if (k != 0)
	    *zp = k, zl++;
    }

    z->l = zl;
}


static int big_comp(Bignum *x, Bignum *y) {
int i, xl, yl;
Bigit *xp, *yp;

    xl = x->l;
    yl = y->l;
    if (xl > yl) return 1;
    if (xl < yl) return -1;

    xp = &x->d[xl];
    yp = &y->d[xl];

    for(i=xl; i>=0; i--, xp--, yp--) {
	Bigit a = *xp;
	Bigit b = *yp;

	if (a > b) return 1;
	if (a < b) return -1;
    }

    return 0;
}


static int sub_big(Bignum *x, Bignum *y, Bignum *z) {
int xl, yl, zl, b, i;
Bigit *xp, *yp, *zp;

    xl = x->l;
    yl = y->l;
    if (yl > xl)
	return 1;

    xp = &x->d[0];
    yp = &y->d[0];
    zp = &z->d[0];
  
    for(i=yl, b=0; i>=0; i--)
	SUB(*xp++, *yp++, *zp++, b);

    for(i=xl-yl; b && i>0; i--) {
	Bigit x_sub;
	x_sub = *xp++;
	*zp++ = x_sub - 1;
	b = (x_sub == 0);
    }

    for(; i>0; i--)
	*zp++ = *xp++;

    if (b)
	return 1;

    zl = xl;
    while(zl > 0 && *--zp == 0)
	zl--;

    z->l = zl;
    return 0;
}


static void add_big(Bignum *x, Bignum *y, Bignum *z) {
Bigit *xp, *yp, *zp;
int xl, yl, k, i;

    xl = x->l;
    yl = y->l;

    if (yl > xl) {
	int tl;
	Bignum *tn;

	tl = xl; xl = yl; yl = tl;
	tn = x; x = y; y = tn;
    }

    xp = &x->d[0];
    yp = &y->d[0];
    zp = &z->d[0];
  
    for(i=yl, k=0; i>=0; i--)
	ADD(*xp++, *yp++, *zp++, k);

    for(i=xl-yl; k && i>0; i--) {
	Bigit z_add;

	z_add = *xp++ + 1;
	k = (z_add == 0);
	*zp++ = z_add;
    }

    for(; i>0; i--)
	*zp++ = *xp++;

    if (k) {
	*zp = 1;
	z->l = xl+1;

    } else
	z->l = xl;
}



static int add_cmp(void) {
int rl, ml, sl, suml;
static Bignum sum;

    rl = R.l;
    ml = (use_mp ? MP.l : MM.l);
    sl = S.l;
   
    suml = rl >= ml ? rl : ml;
    if ((sl > suml+1) || ((sl == suml+1) && (S.d[sl] > 1)))
	return -1;

    if (sl < suml)
	return 1;

    add_big(&R, (use_mp ? &MP : &MM), &sum);
    return big_comp(&sum, &S);
}



static int qr(void) {
int q;

    if (R.l == 0 && S.l == 0) {
	q = R.d[0] / S.d[0];

	R.d[0] -= q*S.d[0];
	return q;
    }

    if (big_comp(&R, &S5) < 0)
	if (big_comp(&R, &S2) < 0) {
	    if (big_comp(&R, &S) < 0)
		return 0;

	    sub_big(&R, &S, &R);
	    return 1;

	} else if (big_comp(&R, &S3) < 0) {
	    sub_big(&R, &S2, &R);
	    return 2;

	} else if (big_comp(&R, &S4) < 0) {
	    sub_big(&R, &S3, &R);
	    return 3;

	} else {
	    sub_big(&R, &S4, &R);
	    return 4;
	}

    else if (big_comp(&R, &S7) < 0)
	if (big_comp(&R, &S6) < 0) {
	    sub_big(&R, &S5, &R);
	    return 5;

	} else {
	    sub_big(&R, &S6, &R);
	    return 6;

	} else if (big_comp(&R, &S9) < 0)
	    if (big_comp(&R, &S8) < 0) {
		sub_big(&R, &S7, &R);
		return 7;

	    } else {
		sub_big(&R, &S8, &R);
		return 8;

	    } else {
		sub_big(&R, &S9, &R);
		return 9;
	    }
}



/* get_pow5()-- Get a power of five.  For small powers, this is a
 * table lookup.  For powers larger than 325, we do a pair of lookups
 * and a multiplication involving one word and multiple words. */

static Bignum *get_pow5(int p) {
static Bignum pow;
int p1, p2, m;

    if (p < MAX_FIVE) {
	m = small_p5[p].len;
	pow.l = m-1;
	memcpy(&pow.d[0], &power5[small_p5[p].offset], m*sizeof(Bigit));
	return &pow;
    }

    p -= MAX_FIVE;

    p1 = p / STEP_FIVE;
    p2 = p % STEP_FIVE;

    m = big_p5[p1].len;
    pow.l = m - 1;
    memcpy(&pow.d[0], &power5[big_p5[p1].offset], m*sizeof(Bigit));

    if (p2 == 0)
	return &pow;

    big_short_mul(&pow, power5[small_p5[p2-1].offset], &pow);

    return &pow;
}



/* set16_R()-- Set R from a quad precision mantissa. */

static void set16_R(int f_n, unsigned *m) {
Bignum temp;
Bigit f;

    f = m[0];
    f = (f << 32) | m[1];

    short_shift_left(f, 64+f_n, &temp);

    f = m[2];
    f = (f << 32) | m[3];

    short_shift_left(f, f_n, &R);

    add_big(&temp, &R, &R);
}



/* set15_S()-- Set S from a quad precision mantissa. */

static void set16_S(Bignum *power, unsigned *m) {
Bignum temp;
Bigit f;

    f = m[0];
    f = (f << 32) | m[1];
    big_short_mul(power, f, &temp);

    big_shift_left(&temp, 64, &S);

    f = m[2];
    f = (f << 32) | m[3];
    big_short_mul(power, f, &temp);

    add_big(&temp, &S, &S);
}



/* format_free()-- Print a floating point number, using as many digits
 * as necessary to distinguish it from other numbers. */

int format_free(char *buf, void *v, int kind) {
int n, e, k, f_n, m_n, i, d, ruf, tc1, tc2, zero, denorm, b1;
unsigned mantissa[4];
Bigit f;

    n = unpack_real(v, kind, mantissa, &e, &ruf, &zero, &denorm, &b1, &k);

    switch(n) {
    case 1:  f = mantissa[0];  break;
    case 2:  f = mantissa[0]; f = (f << 32) | mantissa[1]; break;

    default: f = 0; break;
    }

    if (zero) {
	*buf++ = '0';
	*buf = 0;
	return 0;
    }

    if (e >= 0) {
	if (!b1)
	    use_mp = 0, f_n = e+1, s_n = 1, m_n = e;
	else
	    use_mp = 1, f_n = e+2, s_n = 2, m_n = e;

    } else if (denorm || !b1)
	use_mp = 0, f_n = 1, s_n = 1-e, m_n = 0;

    else
	use_mp = 1, f_n = 2, s_n = 2-e, m_n = 0;

    /* Scale it! */
    if (k == 0) {
	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	one_shift_left(s_n, &S);
	one_shift_left(m_n, &MM);

	if (use_mp)
	    one_shift_left(m_n+1, &MP);

	qr_shift = 1;

    } else if (k > 0) {
	s_n += k;

	if (m_n >= s_n)
	    f_n -= s_n, m_n -= s_n, s_n = 0;
	else
	    f_n -= m_n, s_n -= m_n, m_n = 0;

	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	big_shift_left(get_pow5(k-1), s_n, &S);
	one_shift_left(m_n, &MM);

	if (use_mp)
	    one_shift_left(m_n+1, &MP);

	qr_shift = 0;

    } else {
	Bignum *power = get_pow5(-k-1);

	s_n += k;

	if (n == 4)
	    set16_S(power, mantissa);
	else
	    big_short_mul(power, f, &S);

	big_shift_left(&S, f_n, &R);
	one_shift_left(s_n, &S);
	big_shift_left(power, m_n, &MM);

	if (use_mp)
	    big_shift_left(power, m_n+1, &MP);

	qr_shift = 1;
    }

    /* fixup */
    if (add_cmp() <= -ruf) {
	k--;
	mul10(&R);
	mul10(&MM);
	if (use_mp)
	    mul10(&MP);
    }

   /*
   printf("\nk = %d\n", k);
   printf("R = "); print_big(&R);
   printf("\nS = "); print_big(&S);
   printf("\nM- = "); print_big(&MM);
   if (use_mp) printf("\nM+ = "), print_big(&MP);
   putchar('\n');
   fflush(0);
   */
   
    if (qr_shift) {
	sl = s_n / 64;
	slr = s_n % 64;

    } else {
	big_shift_left(&S, 1, &S2);
	add_big(&S2, &S, &S3);
	big_shift_left(&S2, 1, &S4);
	add_big(&S4, &S, &S5);
	add_big(&S4, &S2, &S6);
	add_big(&S4, &S3, &S7);
	big_shift_left(&S4, 1, &S8);
	add_big(&S8, &S, &S9);
    }

again:
    if (qr_shift) { /* Take advantage of the fact that S = (ash 1 s_n) */
	if (R.l < sl)
	    d = 0;

	else if (R.l == sl) {
	    Bigit *p;

	    p = &R.d[sl];
	    d = *p >> slr;
	    *p &= ((Bigit)1 << slr) - 1;

	    for(i=sl; (i>0) && (*p==0); i--)
		p--;

	    R.l = i;

	} else {
	    Bigit *p;
	 
	    p = &R.d[sl+1];
	    d = *p << (64 - slr) | *(p-1) >> slr;
	    p--;
	    *p &= ((Bigit)1 << slr) - 1;

	    for (i = sl; (i > 0) && (*p == 0); i--)
		p--;
	    R.l = i;
	}

    } else /* We need to do quotient-remainder */
	d = qr();

    tc1 = big_comp(&R, &MM) < ruf;
    tc2 = add_cmp() > -ruf;

    if (!tc1) {
	if (!tc2) {
	    mul10(&R);
	    mul10(&MM);
	    if (use_mp)
		mul10(&MP);

	    *buf++ = d + '0';
	    goto again;
	}

	OUTDIG(d+1);

    } else if (!tc2)
	OUTDIG(d);

    else {
	big_shift_left(&R, 1, &MM);
	if (big_comp(&MM, &S) == -1)
	    OUTDIG(d);
	else
	    OUTDIG(d+1);
    }
}



/* format_fixed()-- Print a fixed number of digits.  Returns nonzero
 * if the star-format is indicated, which can only happen in D/E mode
 * and the scale factor is too small.  The 'type' is 'S' for ES
 * notation and 'E' for D/E formatted. */

int format_fixed(char type, char *buf, void *p, int kind, int prec, int *exp) {
int n, k, e, f_n, i, d, digits, ruf, zero, denorm, b1;
unsigned mantissa[4];
Bigit f;
char c;

    n = unpack_real(p, kind, mantissa, &e, &ruf, &zero, &denorm, &b1, &k);

    switch(n) {
    case 1:  f = mantissa[0];  break;
    case 2:  f = mantissa[0]; f = (f << 32) | mantissa[1]; break;
    default: f = 0; break;
    }

    if (zero) {
	i = prec;
	if (i == 0)
	    i = 1;

	while(i-- > 0)
	    *buf++ = '0';

	*buf = '\0';

	*exp = (type == 'S')
	    ? 0
	    : ioparm->scale_factor - 1;

	return 0;
    }

    if (e >= 0)
	f_n = e, s_n = 0;
    else
	f_n = 0, s_n = -e;

    /* Scale it! */
    if (k == 0) {
	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	one_shift_left(s_n, &S);
	qr_shift = 1;

    } else if (k > 0) {
	s_n += k;

	if (f_n >= s_n) {
	    f_n -= s_n;
	    s_n = 0;
	} else {
	    s_n -= f_n;
	    f_n = 0;
	}

	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	big_shift_left(get_pow5(k-1), s_n, &S);
	qr_shift = 0;

    } else {
	s_n += k;

	if (n == 4)
	    set16_S(get_pow5(-k-1), mantissa);
	else
	    big_short_mul(get_pow5(-k-1), f, &S);

	big_shift_left(&S, f_n, &R);
	one_shift_left(s_n, &S);
	qr_shift = 1;
    }

    /* fixup */
    if (big_comp(&R, &S) < 0) {
	k--;
	mul10(&R);
    }

    if (qr_shift) {
	sl  = s_n / 64;
	slr = s_n % 64;

    } else {
	big_shift_left(&S, 1, &S2);
	add_big(&S2, &S, &S3);
	big_shift_left(&S2, 1, &S4);
	add_big(&S4, &S, &S5);
	add_big(&S4, &S2, &S6);
	add_big(&S4, &S3, &S7);
	big_shift_left(&S4, 1, &S8);
	add_big(&S8, &S, &S9);
    }

    switch(type) {
    case 'S':
	digits = prec;
	break;

    case 'E':
	if (ioparm->scale_factor > 0) {
	    digits = prec + 1;     /* Vanilla exponential formatting */
	    break;
	}

	if (ioparm->scale_factor <= -prec)
	    return 1;

	n = -ioparm->scale_factor;
	for(i=0; i<n; i++)
	    *buf++ = '0';

	digits = prec + ioparm->scale_factor;
	break;
    }

    *exp = k;
    n = digits;

    for(;;) {
	if (qr_shift) { /* Take advantage of the fact that S = (ash 1 s_n) */
	    if (R.l < sl)
		d = 0;

	    else if (R.l == sl) {
		Bigit *p;
	    
		p = &R.d[sl];
		d = *p >> slr;
		*p &= ((Bigit)1 << slr) - 1;

		for(i=sl; (i>0) && (*p==0); i--)
		    p--;

		R.l = i;

	    } else {
		Bigit *p;
	    
		p = &R.d[sl+1];
		d = *p << (64 - slr) | *(p-1) >> slr;
		p--;
		*p &= ((Bigit)1 << slr) - 1;
		for (i = sl; (i > 0) && (*p == 0); i--)
		    p--;

		R.l = i;
	    }

	} else /* We need to do quotient-remainder */
	    d = qr();
      
	*buf++ = d + '0';
	if (--n == 0)
	    break;

	mul10(&R);
    }

    *buf = '\0';

    big_shift_left(&R, 1, &MM);
    switch(big_comp(&MM, &S)) {
    case -1: /* No rounding needed */
	return 0;

    case 0: /* Exactly in the middle, round to even. */
	if (buf[-1] & 1)
	    break;

	return 0;

    default:  /* Round up */
	break;
    }

    buf--;

    for(n=digits; n>0; n--) {
	c = *buf;
	if (c != '9') {
	    *buf = c+1;
	    return 0;
	}

	*buf-- = '0';
    }

    *++buf = '1';

    (*exp)++;
    return 0;
}



/* format_f()-- Format suitable for F printing.  We always generate at
 * least prec+1 digits.  We return nonzero if the number exceeds the
 * maximum width. */

int format_f(char *buf, void *p, int kind, int prec, int width) {
int m, n, k, e, f_n, i, d, ruf, zero, denorm, b1, digits;
unsigned mantissa[4];
char c, *q;
Bigit f;

    n = unpack_real(p, kind, mantissa, &e, &ruf, &zero, &denorm, &b1, &k);
    q = buf;

    switch(n) {
    case 1:  f = mantissa[0];  break;
    case 2:  f = mantissa[0]; f = (f << 32) | mantissa[1]; break;
    default: f = 0; break;
    }

    if (zero) {
	i = prec+1;
	while(i-- > 0)
	    *buf++ = '0';

	*buf = '\0';
	return 0;
    }

    if (e >= 0)
	f_n = e, s_n = 0;
    else
	f_n = 0, s_n = -e;

    /* Scale it! */
    if (k == 0) {
	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	one_shift_left(s_n, &S);
	qr_shift = 1;

    } else if (k > 0) {
	s_n += k;

	if (f_n >= s_n) {
	    f_n -= s_n;
	    s_n = 0;
	} else {
	    s_n -= f_n;
	    f_n = 0;
	}

	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	big_shift_left(get_pow5(k-1), s_n, &S);
	qr_shift = 0;

    } else {
	s_n += k;

	if (n == 4)
	    set16_S(get_pow5(-k-1), mantissa);
	else
	    big_short_mul(get_pow5(-k-1), f, &S);

	big_shift_left(&S, f_n, &R);
	one_shift_left(s_n, &S);
	qr_shift = 1;
    }

    /* fixup */
    if (big_comp(&R, &S) < 0) {
	k--;
	mul10(&R);
    }

    if (qr_shift) {
	sl  = s_n / 64;
	slr = s_n % 64;

    } else {
	big_shift_left(&S, 1, &S2);
	add_big(&S2, &S, &S3);
	big_shift_left(&S2, 1, &S4);
	add_big(&S4, &S, &S5);
	add_big(&S4, &S2, &S6);
	add_big(&S4, &S3, &S7);
	big_shift_left(&S4, 1, &S8);
	add_big(&S8, &S, &S9);
    }

    m = k + ioparm->scale_factor;

    if (m+prec > width)
	return 1;

    if (m >= 0)  /* Digits to the left of the decimal point. */
	digits = prec + m + 1;

    else if (m < -prec) {  /* No significant digits to print. */
	for(i=0; i<prec; i++)
	    *buf++ = '0';

	if (k < -(prec+1)) {
	    *buf++ = '0';
	    *buf = '\0';
	    return 0;
	}

	if (qr_shift) {
	    big_shift_left(&S, 2, &S4);
	    add_big(&S4, &S, &S5);
	}

	*buf++ = '0' + (big_comp(&S5, &R) <= 0);  /* Round */
	*buf = '\0';
	return 0;

    } else {   /* Partially filled field to the right of the dp. */
	i = -m;
	digits = prec - i + 1;

	while(i-- > 0)
	    *buf++ = '0';
    }

    n = digits;

    for(;;) {
	if (qr_shift) { /* Take advantage of the fact that S = (ash 1 s_n) */
	    if (R.l < sl)
		d = 0;

	    else if (R.l == sl) {
		Bigit *p;

		p = &R.d[sl];
		d = *p >> slr;
		*p &= ((Bigit)1 << slr) - 1;

		for(i=sl; (i>0) && (*p==0); i--)
		    p--;

		R.l = i;

	    } else {
		Bigit *p;
	    
		p = &R.d[sl+1];
		d = *p << (64 - slr) | *(p-1) >> slr;
		p--;
		*p &= ((Bigit)1 << slr) - 1;

		for (i = sl; (i > 0) && (*p == 0); i--)
		    p--;

		R.l = i;
	    }

	} else /* We need to do quotient-remainder */
	    d = qr();
      
	*buf++ = d + '0';
	if (--n == 0)
	    break;

	mul10(&R);
    }

    *buf = '\0';

    big_shift_left(&R, 1, &MM);
    switch(big_comp(&MM, &S)) {
    case -1: /* No rounding needed */
	return 0;

    case 0: /* Exactly in the middle, round to even. */
	if (buf[-1] & 1)
	    break;

	return 0;

    default:  /* Round up */
	break;
    }

    buf--;
    while(buf >= q) {
	c = *buf;
	if (c != '9') {
	    *buf = c+1;
	    return 0;
	}

	*buf-- = '0';
    }

    /* Since we're not returning a power of ten, we actually have to
     * tack on an extra zero at this point. */

    *++buf = '1';

    while(*buf != '\0')
	buf++;

    *buf++ = '0';
    *buf = '\0';

    return 0;
}



/* format_en()-- Print a set of digits suitable for EN output.  The
 * invariants in the output are 1 <= p <= 3 and k-(p-1) mod 3 = 0
 * where k is the returned power of ten, and p is the number of
 * non-prec digits. */

int format_en(char *buf, void *p, int kind, int prec) {
int n, k, e, f_n, i, d, ruf, zero, denorm, b1, digits;
unsigned mantissa[4];
Bigit f;
char c;

    n = unpack_real(p, kind, mantissa, &e, &ruf, &zero, &denorm, &b1, &k);

    switch(n) {
    case 1:  f = mantissa[0]; break;
    case 2:  f = mantissa[0]; f = (f << 32) | mantissa[1]; break;
    default: f = 0; break;
    }

    if (zero) {
	i = prec+1;
	while(i-- > 0)
	    *buf++ = '0';

	*buf = '\0';
	return 0;
    }

    if (e >= 0)
	f_n = e, s_n = 0;
    else
	f_n = 0, s_n = -e;

    /* Scale it! */
    if (k == 0) {
	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	one_shift_left(s_n, &S);
	qr_shift = 1;

    } else if (k > 0) {
	s_n += k;

	if (f_n >= s_n) {
	    f_n -= s_n;
	    s_n = 0;
	} else {
	    s_n -= f_n;
	    f_n = 0;
	}

	if (n == 4)
	    set16_R(f_n, mantissa);
	else
	    short_shift_left(f, f_n, &R);

	big_shift_left(get_pow5(k-1), s_n, &S);
	qr_shift = 0;

    } else {
	s_n += k;

	if (n == 4)
	    set16_S(get_pow5(-k-1), mantissa);
	else
	    big_short_mul(get_pow5(-k-1), f, &S);

	big_shift_left(&S, f_n, &R);
	one_shift_left(s_n, &S);
	qr_shift = 1;
    }

    /* fixup */
    if (big_comp(&R, &S) < 0) {
	k--;
	mul10(&R);
    }

    if (qr_shift) {
	sl = s_n / 64;
	slr = s_n % 64;
	
    } else {
	big_shift_left(&S, 1, &S2);
	add_big(&S2, &S, &S3);
	big_shift_left(&S2, 1, &S4);
	add_big(&S4, &S, &S5);
	add_big(&S4, &S2, &S6);
	add_big(&S4, &S3, &S7);
	big_shift_left(&S4, 1, &S8);
	add_big(&S8, &S, &S9);
    }

    i = k % 3;
    if (i < 0)
	i += 3;

    digits = prec + i + 1;
    n = digits;

    for(;;) {
	if (qr_shift) { /* Take advantage of the fact that S = (ash 1 s_n) */
	    if (R.l < sl)
		d = 0;

	    else if (R.l == sl) {
		Bigit *p;
	    
		p = &R.d[sl];
		d = *p >> slr;
		*p &= ((Bigit)1 << slr) - 1;

		for(i=sl; (i>0) && (*p==0); i--)
		    p--;

		R.l = i;

	    } else {
		Bigit *p;
	    
		p = &R.d[sl+1];
		d = *p << (64 - slr) | *(p-1) >> slr;
		p--;
		*p &= ((Bigit)1 << slr) - 1;
		for (i = sl; (i > 0) && (*p == 0); i--)
		    p--;
	
		R.l = i;
	    }

	} else /* We need to do quotient-remainder */
	    d = qr();

	*buf++ = d + '0';
	if (--n == 0)
	    break;

	mul10(&R);
    }

    *buf = '\0';

    big_shift_left(&R, 1, &MM);
    switch(big_comp(&MM, &S)) {
    case -1: /* No rounding needed */
	return k;

    case 0: /* Exactly in the middle, round to even. */
	if (buf[-1] & 1)
	    break;

	return k;

    default:  /* Round up */
	break;
    }

    buf--;

    for(n=digits; n>0; n--) {
	c = *buf;
	if (c != '9') {
	    *buf = c+1;
	    return k;
	}

	*buf-- = '0';
    }

    *++buf = '1';

    /* At the point, the original estimate of k was one lower than it
     * needed to be.  We have to preserve the invariant by messing with
     * the number of digits we're returning.  If k goes up by one, p
     * also has to go up by one or down by two. */

    while(*buf != '\0')
	buf++;

    if (digits - prec == 3)
	buf[-2] = '\0';

    else {
	*buf++ = '0';
	*buf = '\0';
    }

    return k+1;
}



/* big_compare()-- Compare a floating point number to a bigfloat.  Returns
 * -1 if the fp number is less than the bigfloat, zero if equal, and
 * +1 if the fp number is greater than the bigfloat.  The fp number is
 * guaranteed to be a regular number. */

static int big_compare(int exp, unsigned m[], const bigfloat *b) {
unsigned m0, m1, m2, m3;

    if (exp < b->exp)
	return -1;

    if (exp > b->exp)
	return 1;

    if (m[0] < b->m[0])
	return -1;

    if (m[0] > b->m[0])
	return 1;

    if (m[1] < b->m[1])
	return -1;

    if (m[1] > b->m[1])
	return 1;

    if (m[2] < b->m[2])
	return -1;

    if (m[2] > b->m[2])
	return 1;

    if (m[3] < b->m[3])
	return -1;

    if (m[3] > b->m[3])
	return 1;

    return 0;
}



/* power_method()-- This subroutine is plan b when figuring out G
 * formatting.  In this case, d > 35, which is larger than the table,
 * but at that size, the subtrahend is insignificant with respect to
 * the subtractor, so we're figuring out what power of ten we have.
 * We accomplish this by using the power of five table.  On a binary
 * computer, powers of ten are powers of five with some shifts. */

int power_method(int d, int kind, int exp, unsigned m[]) {

    return 1;

}



/* get_f_fmt()-- Compute the F format to be used during G formatting.
 * Returns the F format to use, or -1 for out of range. */

int get_f_fmt(int d, void *source, int kind) {
int i, exp, zero, denorm, dummy;
const bigfloat *low, *high, *b;
unsigned n[4], m[4];

    unpack_real(source, kind, m, &exp, &dummy, &zero, &denorm, &dummy, &dummy);

    if (zero)
	return (d == 0) ? -1 : d - 1;

    if (denorm)
	return -1;   /* Not zero, and too small for F formatting */

    if (d > 35)
	return power_method(d, kind, exp, m);

    low  = cutoffs + d*(d+3)/2;   /* Triangular array access */
    high = low + d + 1;

    switch(kind) {
    case 4:
	exp += 23;
	n[0] = m[0] << 8;
	n[1] = n[2] = n[3] = 0;
	break;

    case 8:
	exp += 52;
	n[0] = (m[0] << 11) | (m[1] >> 21);
	n[1] = m[1] << 11;
	n[2] = n[3] = 0;
	break;

    case 10:
	exp += 63;
	n[0] = m[0];
	n[1] = m[1];
	n[2] = n[3] = 0;
	break;

    case 16:
	exp += 116;
	n[0] = (m[0] << 15) | (m[1] >> 17);
	n[1] = (m[1] << 15) | (m[2] >> 17);
	n[2] = (m[2] << 15) | (m[3] >> 17);
	n[3] = (m[3] << 15);
	break;
    }

    if (big_compare(exp, n, low) < 0 ||
	big_compare(exp, n, high) >= 0)
	return -1;

    /* Instead of a binary search, which would take O(log(n)) work, we
     * use the fact that the exponents of the table values are ordered
     * linearly, because the values themselves are exponentially
     * ordered, this ends up being O(1) work.  The minimum value of
     * the exponent is always -4, so the minimum i is zero. */

    i = ((d + 1)*(exp + 4)) / (high->exp + 4);

    /* Search forward. Usually one step here, sometimes none. */

    while(big_compare(exp, n, low+i) >= 0)
	i++;

    return d + 1 - i;
}

